#!/bin/bash
# Author songrq@raisecom.com
#
# Parameters:
# $1 host: IP address of any cluster node.
# $2 source: cluster/netconf/inventory. Data source.
#
# Description:
# Search all online devices from netconf/cluster/inventory and write them into ./logs/online_dev_$source.log
#
# Versions:
# v1.4.16 2020-09-30 create
# v1.4.17 2020-10-09 show real device count for every page
# v1.4.17 2020-10-10 add choice to query from all netconf nodes
# v1.4.17 2020-10-30 show connection status of netconf
# v1.4.19 2020-12-03 remove page information while query devices from inventory & query all devices not only online devices

ipv4_expr='^(([0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])$'
port_expr='^([1-9][0-9]{0,3}|[1-5][0-9]{4}|6[0-4][0-9]{3}|65[0-4][0-9]{2}|655[0-2][0-9]|6553[0-5])$'

host=$1
checkIp=`echo $host | egrep $ipv4_expr`
while [ -z "$checkIp" -a "$host" != "localhost" ]
do
    read -p "Input host of master(IPv4): " host
    checkIp=`echo $host | egrep $ipv4_expr`
done

source=$2
while [ "$source" != "cluster" -a "$source" != "netconf" -a "$source" != "inventory" ]
do
    read -p "Input data source(netconf/cluster/inventory): " source
done
echo "source=$source"

dir=$(cd `dirname $0`; pwd)
mkdir -p $dir/logs
filename=$dir/logs/online_dev_${source}.log
rm -f $filename

modules=`curl -s http://$host:60216/cluster/allModule | grep -Po '(?<=ip":")[^"]*'`
echo -e "modules:\n$modules\n"

limit=100
if [ "$source" == "inventory" ]; then
    total=`curl -s http://$host:60175/inventory/deviceinfo?limit=1 | grep -Po '(?<=count":)[0-9]*'`
    echo "Total $total devices on inventory."

    if [ $total -eq 0 ]; then
        touch $filename
        exit 0
    fi
    count=0
    page=0
    while true
    do
        real_count=`curl -s http://$host:60175/inventory/deviceinfo?page=$page\&limit=$limit | sed 's/{"pppoe/\n{"pppoe/g' | tee -a $filename | grep -v '{"data' | wc -l`
        count=`expr $count + $real_count`
        page=`expr $page + 1`
        if [ $real_count -lt $limit -o $count -ge $total ]; then
            break
        fi
    done
    sed -i '/^{"data":\[$/d' $filename
    echo "$count devices on inventory."
    exit 0
fi

total=0
for module in $modules
do
    if [ "$source" == "cluster" ]; then
        total_module=`curl -s http://$host:60216/cluster/route/statistics?nodeIp=$module | grep -Po '(?<=count":)[0-9]*'`
        echo "$total_module devices on node $module"
        count=0
        page=1
        while true
        do
            curl -s http://$host:60216/cluster/route/statistics?nodeIp=$module\&limit=$limit\&page=$page\&detail=true | sed 's/{"deviceType/\n{"deviceType/g' >> $filename 
            echo -e "\n" >> $filename
            realCount=`egrep -o '(?<=currentCount":)[0-9]*' $filename | tail -1`
            count=`expr $count + $realCount`
            page=`expr $page + 1`
            total=`expr $total + $realCount`
            if [ $realCount -lt $limit -o $count -ge $total_module ]; then
                break
            fi
        done
    else
        total_last=$total
        curl -s http://admin:admin@$module:8181/restconf/operational/network-topology:network-topology/topology/topology-netconf?depth=3 | sed "s#{\"node-id#\n${module} {\"node-id#g" | tee -a $filename | grep -Po '(?<=status":")[^"]*' | sort | uniq -c
        echo -e "\n" >> $filename
        total=`grep -o 'node-id' $filename | wc -l`
        total_module=`expr $total - $total_last`
        echo "$total_module devices on node $module"
    fi
done

[ "$source" == "cluster" ] && sed -i '/^.*result.*$/d' $filename
[ "$source" == "netconf" ] && sed -i '/^.*topology-id.*$/d' $filename
sed -i '/^$/d' $filename
rm -f temp
echo -e "\nTotal $total online devices on $source."
echo -e "Please check $filename.\n"

exit 0
