/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.jsonrpc.impl;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import org.opendaylight.jsonrpc.bus.messagelib.ResponderSession;
import org.opendaylight.jsonrpc.bus.messagelib.TransportFactory;
import org.opendaylight.jsonrpc.impl.MappedPeerContext;
import org.opendaylight.jsonrpc.impl.RemoteControl;
import org.opendaylight.jsonrpc.impl.Util;
import org.opendaylight.jsonrpc.model.RemoteGovernance;
import org.opendaylight.mdsal.binding.api.ClusteredDataTreeChangeListener;
import org.opendaylight.mdsal.binding.api.DataBroker;
import org.opendaylight.mdsal.binding.api.DataTreeChangeListener;
import org.opendaylight.mdsal.binding.api.DataTreeIdentifier;
import org.opendaylight.mdsal.binding.api.ReadTransaction;
import org.opendaylight.mdsal.common.api.LogicalDatastoreType;
import org.opendaylight.mdsal.dom.api.DOMDataBroker;
import org.opendaylight.mdsal.dom.api.DOMMountPointService;
import org.opendaylight.mdsal.dom.api.DOMSchemaService;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.Uri;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.Config;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.ConfigBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.Entity;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.ForceRefreshInput;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.ForceRefreshOutput;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.ForceRefreshOutputBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.ForceReloadInput;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.ForceReloadOutput;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.ForceReloadOutputBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.JsonrpcService;
import org.opendaylight.yang.gen.v1.urn.opendaylight.jsonrpc.rev161201.Peer;
import org.opendaylight.yangtools.yang.binding.InstanceIdentifier;
import org.opendaylight.yangtools.yang.common.RpcResult;
import org.opendaylight.yangtools.yang.common.RpcResultBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JsonRPCProvider
implements JsonrpcService,
AutoCloseable {
    private static final String ME = "JSON RPC Provider";
    private static final Logger LOG = LoggerFactory.getLogger(JsonRPCProvider.class);
    private static final InstanceIdentifier<Config> GLOBAL_CFG_II = InstanceIdentifier.create(Config.class);
    private static final DataTreeIdentifier<Config> OPER_DTI = DataTreeIdentifier.create((LogicalDatastoreType)LogicalDatastoreType.OPERATIONAL, GLOBAL_CFG_II);
    private static final DataTreeIdentifier<Config> CFG_DTI = DataTreeIdentifier.create((LogicalDatastoreType)LogicalDatastoreType.CONFIGURATION, GLOBAL_CFG_II);
    private TransportFactory transportFactory;
    private DataBroker dataBroker;
    private DOMDataBroker domDataBroker;
    private DOMSchemaService schemaService;
    private volatile RemoteGovernance governance;
    private volatile ResponderSession remoteControl;
    private final Map<String, MappedPeerContext> peerState = Maps.newConcurrentMap();
    private final List<AutoCloseable> toClose = new LinkedList<AutoCloseable>();
    private final ReentrantReadWriteLock changeLock = new ReentrantReadWriteLock();
    private volatile boolean sessionInitialized = false;
    private volatile boolean providerClosed = false;
    private DOMMountPointService domMountPointService;
    private String lastGovernanceUri;
    private String lastWhoAmIUri;

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    private Config getConfig() {
        try (ReadTransaction roTrx = this.dataBroker.newReadOnlyTransaction();){
            Config config = ((Optional)roTrx.read(LogicalDatastoreType.CONFIGURATION, GLOBAL_CFG_II).get()).orElse(null);
            return config;
        }
        catch (InterruptedException | ExecutionException e) {
            LOG.error("Failed to read configuration", (Throwable)e);
            return null;
        }
    }

    private Map<String, Peer> generateCache(List<? extends Peer> arg) {
        return Optional.ofNullable(arg).orElse(Collections.emptyList()).stream().collect(Collectors.toMap(Entity::getName, Function.identity()));
    }

    private boolean processNotificationInternal() {
        LOG.debug("Processing notification");
        if (!this.sessionInitialized) {
            LOG.debug("Can't process configuration change at this time, need provider session first");
            return false;
        }
        if (this.providerClosed) {
            LOG.debug("{} was closed already, ignoring configuration change", (Object)ME);
            return false;
        }
        Config peersConfState = this.getConfig();
        if (peersConfState == null) {
            LOG.info("{} configuration absent", (Object)ME);
            this.unmountPeers(new ConfigBuilder().setConfiguredEndpoints(Collections.emptyList()).build());
            this.stopGovernance();
            this.stopRemoteControl();
            this.lastGovernanceUri = null;
            this.lastWhoAmIUri = null;
            return false;
        }
        if (!this.resetGovernance(peersConfState.getGovernanceRoot())) {
            return false;
        }
        if (!this.initRemoteControl(peersConfState.getWhoAmI())) {
            return false;
        }
        boolean result = this.mountPeers(peersConfState);
        return result &= this.unmountPeers(peersConfState);
    }

    private boolean unmountPeers(Config peersConfState) {
        boolean result = true;
        Map<String, Peer> cache = this.generateCache(peersConfState.getConfiguredEndpoints());
        List toUnmountList = this.peerState.entrySet().stream().filter(e -> !cache.containsKey(e.getKey())).map(Map.Entry::getKey).collect(Collectors.toList());
        for (String toUnmount : toUnmountList) {
            result &= this.doUnmount(toUnmount);
        }
        return result;
    }

    private boolean mountPeers(Config peersConfState) {
        boolean result = true;
        if (peersConfState.getConfiguredEndpoints() != null) {
            for (Peer confPeer : peersConfState.getConfiguredEndpoints()) {
                LOG.debug("Processing peer from conf {}", (Object)confPeer.getName());
                if (this.peerState.containsKey(confPeer.getName())) continue;
                result &= this.doMountDevice(confPeer);
            }
        } else {
            LOG.debug("No configured endpoints");
        }
        return result;
    }

    private void stopRemoteControl() {
        if (this.remoteControl != null) {
            Util.closeNullableWithExceptionCallback((AutoCloseable)this.remoteControl, t -> LOG.warn("Failed to close RemoteControl", (Throwable)t));
            this.remoteControl = null;
        }
    }

    private void stopGovernance() {
        if (this.governance != null) {
            Util.closeNullableWithExceptionCallback(this.governance, t -> LOG.warn("Failed to close RemoteGovernance", (Throwable)t));
            this.governance = null;
        }
    }

    private boolean initRemoteControl(Uri whoAmI) {
        try {
            if (whoAmI != null) {
                if (!whoAmI.getValue().equals(this.lastWhoAmIUri)) {
                    this.lastWhoAmIUri = whoAmI.getValue();
                    this.stopRemoteControl();
                    LOG.debug("Exposing remote control at {}", (Object)whoAmI);
                    this.remoteControl = this.transportFactory.endpointBuilder().responder().create(whoAmI.getValue(), (AutoCloseable)new RemoteControl(this.domDataBroker, this.schemaService.getGlobalContext(), this.transportFactory));
                }
            } else {
                this.remoteControl = null;
                LOG.debug("Remote control not configured");
            }
        }
        catch (Exception e) {
            LOG.error("Unable to initialize remote control, can't continue", (Throwable)e);
            return false;
        }
        return true;
    }

    private boolean resetGovernance(Uri rootOm) {
        try {
            if (rootOm != null) {
                if (!rootOm.getValue().equals(this.lastGovernanceUri)) {
                    LOG.debug("(Re)setting governance root for JSON RPC to {}", (Object)rootOm);
                    this.lastGovernanceUri = rootOm.getValue();
                    this.stopGovernance();
                    this.governance = (RemoteGovernance)this.transportFactory.endpointBuilder().requester().createProxy(RemoteGovernance.class, rootOm.getValue());
                }
            } else {
                this.governance = null;
            }
            return true;
        }
        catch (Exception e) {
            LOG.error("Unable to initialize governance, can't continue", (Throwable)e);
            return false;
        }
    }

    @GuardedBy(value="changeLock")
    private boolean processNotification() {
        ReentrantReadWriteLock.WriteLock wLock = this.changeLock.writeLock();
        try {
            wLock.lock();
            boolean bl = this.processNotificationInternal();
            return bl;
        }
        finally {
            wLock.unlock();
        }
    }

    public void init() {
        LOG.debug("JSON RPC Provider init");
        Objects.requireNonNull(this.transportFactory, "TransportFactory was not set");
        Objects.requireNonNull(this.dataBroker, "DataBroker was not set");
        Objects.requireNonNull(this.domDataBroker, "DOMDataBroker was not set");
        Objects.requireNonNull(this.domMountPointService, "DOMMountPointService was not set");
        this.toClose.add((AutoCloseable)this.dataBroker.registerDataTreeChangeListener(OPER_DTI, (DataTreeChangeListener)((ClusteredDataTreeChangeListener)changes -> this.processNotification())));
        this.toClose.add((AutoCloseable)this.dataBroker.registerDataTreeChangeListener(CFG_DTI, (DataTreeChangeListener)((ClusteredDataTreeChangeListener)changes -> this.processNotification())));
        this.sessionInitialized = true;
        this.processNotification();
    }

    @Override
    public void close() {
        this.providerClosed = true;
        this.peerState.values().forEach(p -> this.doUnmount(p.getName()));
        this.peerState.clear();
        this.stopRemoteControl();
        this.stopGovernance();
        this.toClose.forEach(c -> Util.closeNullableWithExceptionCallback(c, e -> LOG.warn("Failed to close object {}", c, e)));
        LOG.debug("JsonRPCProvider Closed");
    }

    private boolean doMountDevice(Peer peer) {
        if (!this.sessionInitialized) {
            return false;
        }
        try {
            LOG.debug("Creating mapping context for peer {}", (Object)peer.getName());
            MappedPeerContext ctx = new MappedPeerContext(peer, this.transportFactory, this.schemaService, this.dataBroker, this.domMountPointService, this.governance);
            this.peerState.put(peer.getName(), ctx);
            LOG.info("Peer mounted : {}", (Object)ctx);
            return true;
        }
        catch (Exception e) {
            LOG.error("Device '{}' can't be mounted", (Throwable)e);
            return false;
        }
    }

    public boolean doUnmount(String deviceName) {
        if (Strings.isNullOrEmpty((String)deviceName)) {
            return false;
        }
        MappedPeerContext toDelete = this.peerState.remove(deviceName);
        if (toDelete == null) {
            LOG.error("Device '{}' did not complete mount, cannot remove", (Object)deviceName);
            return false;
        }
        try {
            LOG.debug("Destroying mapping context of peer '{}'", (Object)toDelete.getName());
            toDelete.close();
            LOG.debug("Device '{}' unmounted successfully", (Object)deviceName);
            return true;
        }
        catch (Exception e) {
            LOG.error("Device '{}' unmount, failed", (Object)deviceName, (Object)e);
            return false;
        }
    }

    public ListenableFuture<RpcResult<ForceRefreshOutput>> forceRefresh(ForceRefreshInput input) {
        LOG.debug("Refreshing json rpc state");
        return Futures.immediateFuture((Object)RpcResultBuilder.success((Object)new ForceRefreshOutputBuilder().setResult(Boolean.valueOf(this.processNotification())).build()).build());
    }

    public ListenableFuture<RpcResult<ForceReloadOutput>> forceReload(ForceReloadInput input) {
        ForceReloadOutputBuilder outputBuilder = new ForceReloadOutputBuilder();
        LOG.debug("Remounting all json rpc peers");
        boolean result = true;
        ArrayList<String> toUnmountList = new ArrayList<String>();
        toUnmountList.addAll(this.peerState.keySet());
        for (String toUnmount : toUnmountList) {
            result &= this.doUnmount(toUnmount);
        }
        outputBuilder.setResult(Boolean.valueOf(result &= this.processNotification()));
        return Futures.immediateFuture((Object)RpcResultBuilder.success((Object)outputBuilder.build()).build());
    }

    public void setTransportFactory(TransportFactory transportFactory) {
        this.transportFactory = transportFactory;
    }

    public void setDataBroker(DataBroker dataBroker) {
        this.dataBroker = dataBroker;
    }

    public void setDomMountPointService(DOMMountPointService domMountPointService) {
        this.domMountPointService = domMountPointService;
    }

    public void setDomDataBroker(DOMDataBroker domDataBroker) {
        this.domDataBroker = domDataBroker;
    }

    public void setSchemaService(DOMSchemaService schemaService) {
        this.schemaService = schemaService;
    }
}

