#!/bin/bash

###############################################################################
# Main script that monitors the necessary services and interfaces.
###############################################################################

declare TIMEOUT=${1:-3}
declare DEPLOY=${2:-no-name-set-for-deploy}

declare packets_gtpu=""
declare packets_gtpbr=""
declare FAIL_GTPU_OUT=0
declare FAIL_GTPBR_OUT=0
declare IN_FAILURE_STATE=0

declare FAIL_TYPE_USER_PLANE=0
declare FAIL_TYPE_S1=0

declare -r LOOP_DELAY=10

declare -r REMADDR="/proc/net/sctp/remaddr"

source utils.sh
source check-params.sh
source dump-config.sh
source iptables-config.sh
source magma-status.sh
source monitor-sctp.sh
source monitor-coredumps.sh
source monitor-memory.sh

# check if gtpu got any packets
# Params:
#    NONE
# Returns:
#    NONE
function __check_gtpu() {
    FAIL_GTPU_OUT=0

    if [[ $packets_gtpu == "0 packets captured" ]]; then
        FAIL_GTPU_OUT=1

        log_message "[$DEPLOY][$(date)] : Info! 0 packets captured on GTPU" "${FUNCNAME[0]}" "$LINENO"
    fi
}

# check if gtpbr got any packets
# Params:
#    NONE
# Returns:
#    NONE
function __check_gtpbr() {
    FAIL_GTPBR_OUT=0

    if [[ $packets_gtpbr == "0 packets captured" ]]; then
        FAIL_GTPBR_OUT=1

        log_message "[$DEPLOY][$(date)] : Info! 0 packets captured on GTPBR" "${FUNCNAME[0]}" "$LINENO"
    fi
}

# Capture packets on the required interfaces
# Params:
#    NONE
# Returns:
#    NONE
function traffic_outgoing() {
    gtpu="gtpu_sys_2152"
    gtpbr="gtp_br0"

    check_interface $gtpu
    check_interface $gtpbr

    packets_gtpu=$(timeout $TIMEOUT tcpdump -i $gtpu not host 10.1.0.1 -Q out 2>&1)
    if [[ $(grep -iE "You don't have permission" <<<$packets_gtpu) ]]; then
        send_notification "[$DEPLOY][$(date)] : Error! You don't have permission to capture on $gtpu"
        log_message "[$DEPLOY][$(date)] : Error! You don't have permission to capture on $gtpu" "${FUNCNAME[0]}" "$LINENO"
        exit 3
    fi
    packets_gtpu=$(grep -iE "^0 packets captured" <<<$packets_gtpu)
    __check_gtpu

    packets_gtpbr=$(timeout $TIMEOUT tcpdump -i $gtpbr not host 10.1.0.1 -Q out 2>&1)
    if [[ $(grep -iE "You don't have permission" <<<$packets_gtpbr) ]]; then
        send_notification "[$DEPLOY][$(date)] : Error! You don't have permission to capture on $gtpbr"
        log_message "[$DEPLOY][$(date)] : Error! You don't have permission to capture on $gtpbr" "${FUNCNAME[0]}" "$LINENO"
        exit 4
    fi
    packets_gtpbr=$(grep -iE "^0 packets captured" <<<$packets_gtpbr)
    __check_gtpbr
}

# Do a ping test with the eNB to check if they are communicating
# Params:
#    NONE
# Returns:
#    0: on sucess
#    1: on failure
function check_enb() {
    ip_enb=$(cat $REMADDR | awk 'FNR==2{print $1}')

    ping -i 3 -c 3 -W 3 -w 10 $ip_enb
    if [[ $? -eq 0 ]]; then
        return 0
    fi

    log_message "[$DEPLOY][$(date)] : Warn! Failed to ping $ip_enb" "${FUNCNAME[0]}" "$LINENO"

    return 1
}

# Check if there are subscribers authenticated
# Params:
#    NONE
# Returns:
#    0: on sucess
#    1: on failure
function check_subscribers() {
    count=$(mobility_cli.py get_subscriber_table 2>/dev/null | grep ^IMSI | wc -l)
    log_message "[$DEPLOY][$(date)] : Info! Current subscribers: $count" "${FUNCNAME[0]}" "$LINENO"
    if [[ "$count" -gt 0 ]]; then
        return 0
    else
        return 1
    fi
}

# Check all traffic related tests
# Params:
#    NONE
# Returns:
#    0: on sucess
#    1: on failure related to GTPU and GTPBR interfaces
#    2: on failure related to eNB tests
function check_traffic() {
    local err_enb=0
    local err_subs=0

    check_enb
    if [[ $? -eq 1 ]]; then err_enb=1; fi

    check_subscribers
    if [[ $? -eq 1 ]]; then err_subs=1; fi

    traffic_outgoing    # sets FAIL_GPTU and/or FAIL_GPTBR if error

    if [[ $err_enb -eq 0 && $err_subs -eq 0 ]]; then
        if [[ $FAIL_GTPU_OUT -eq 1 || $FAIL_GTPBR_OUT -eq 1 ]]; then
            return 1
        fi
    elif [[ $err_enb -eq 1 ]]; then
        return 2
    fi

    return 0
}

# disable unattended upates
# Params:
#    NONE
# Returns:
#    1: error trying to disable unattended updates
#    0: success
function disable_unattended_updates() {
    local apt_auto_file="/etc/apt/apt.conf.d/20auto-upgrades"
    if [[ ! -f $apt_auto_file ]]; then
        touch $apt_auto_file || {
            send_notification "[$DEPLOY][$(date)] : Error! Could not create file for unattended updates"
            log_message "[$DEPLOY][$(date)] : Error! Could not create file for unattended updates" "${FUNCNAME[0]}" "$LINENO"
            return 1
        }
    fi

    cat <<EOF > $apt_auto_file
APT::Periodic::Update-Package-Lists "0";
APT::Periodic::Download-Upgradeable-Packages "0";
APT::Periodic::AutocleanInterval "0";
APT::Periodic::Unattended-Upgrade "0";
EOF

    systemctl disable unattended-upgrades || {
        send_notification "[$DEPLOY][$(date)] : Error! Could not disable unattended updates"
        log_message "[$DEPLOY][$(date)] : Error! Could not disable unattended updates" "${FUNCNAME[0]}" "$LINENO"
        return 1
    }
    systemctl stop unattended-upgrades || {
        send_notification "[$DEPLOY][$(date)] : Error! Could not stop unattended updates"
        log_message "[$DEPLOY][$(date)] : Error! Could not stop unattended updates" "${FUNCNAME[0]}" "$LINENO"
        return 1
    }

    send_notification "[$DEPLOY][$(date)] : Info! Disabled unattended updates"
    log_message "[$DEPLOY][$(date)] : Info! Disabled unattended updates" "${FUNCNAME[0]}" "$LINENO"

    return 0
}

# Send error message user plane and restart sctpd
# Params:
#    NONE
# Returns:
#    NONE
function __send_error_restart_sctpd() {
    systemctl restart sctpd
    send_notification "[$DEPLOY][$(date)] : Error! No packets on User Plane, restarting sctpd"
    log_message "[$DEPLOY][$(date)] : Error! No packets on User Plane, restarting sctpd" "${FUNCNAME[0]}" "$LINENO"
}

# Send error message for enb check
# Params:
#    NONE
# Returns:
#    NONE
function __send_error_check_enb() {
    send_notification "[$DEPLOY][$(date)] : Error! Check eNB ping failed: IP $(cat $REMADDR | awk 'FNR==2{print $1}')"
    log_message "[$DEPLOY][$(date)] : Error! Check eNB ping failed" "${FUNCNAME[0]}" "$LINENO"
}

# Throws error via notification and log message
# Params:
#    $1: number to identify the return code passed from the previous 'check_traffic' call
# Returns:
#    NONE
function throw_error() {
    if [[ $1 -eq 0 ]]; then
        return
    fi

    if [[ $IN_FAILURE_STATE -eq 0 ]]; then
        IN_FAILURE_STATE=1
        if [[ $1 -eq 1 ]]; then
            FAIL_TYPE_USER_PLANE=1
            __send_error_restart_sctpd
        fi
        if [[ $1 -eq 2 ]]; then
            FAIL_TYPE_S1=1
            __send_error_check_enb
        fi
        dump_config "$DEPLOY"
    else
        # The idea here is that, if we are in fail state, we entered in fail state
        # due to one of the possible case: FAIL_TYPE_USER_PLANE or FAIL_TYPE_S1.
        # So, the idea is that, even if we are in fail state, if the other error
        # happens, we also want to know that, and be able to get the notification
        # and the corresponding logs.
        if [[ $FAIL_TYPE_USER_PLANE -eq 1 && $FAIL_TYPE_S1 -eq 0 ]]; then
            FAIL_TYPE_S1=1
            __send_error_check_enb
            dump_config "$DEPLOY"
        elif [[ $FAIL_TYPE_USER_PLANE -eq 0 && $FAIL_TYPE_S1 -eq 1 ]]; then
            FAIL_TYPE_USER_PLANE=1
            __send_error_restart_sctpd
            dump_config "$DEPLOY"
        fi
    fi
}

# Run the probe in a loop
# Params:
#     $1: timeout for the tcpdump probes
#     $2: deploy name to be used when sending notifications/logging
# Returns:
#     NONE
function probe_run() {
    check_sudo
    check_params "$@"

    monitor_sctp_init "$DEPLOY"

    send_notification "[$DEPLOY][$(date)] : Info init! Starting probe script. Timeout='$TIMEOUT' Deploy='$DEPLOY'..."
    log_message "[$DEPLOY][$(date)] : Info init! Starting probe script. Timeout='$TIMEOUT' Deploy='$DEPLOY'..." "${FUNCNAME[0]}" "$LINENO"

    check_netstat
    check_tcpdump

    disable_unattended_updates

    monitor_coredumps_init "$DEPLOY"
    init_magma_status "$DEPLOY"
    monitor_memory_init

    # This function MUST run in background because it will do
    # a more real time monitoring on the sctp counters.
    # Thus, the 'monitor_sctp_run' loop will execute every 5
    # seconds independently of the configured timeout used by
    # the tcpdump capture.
    monitor_sctp_run "$DEPLOY" &

    while true; do

        itables_run "$DEPLOY"
        magma_status_run "$DEPLOY"
        monitor_memory_run "$DEPLOY"
        monitor_coredumps_run "$DEPLOY"

        check_traffic
        ret=$?
        if [[ $ret -ne 0 ]]; then
            throw_error $ret
        else
            if [[ $IN_FAILURE_STATE -eq 1 ]]; then
                send_notification "[$DEPLOY][$(date)] : Info! recovered from failure state"
                log_message "[$DEPLOY][$(date)] : Info! Info! recovered from failure state" "${FUNCNAME[0]}" "$LINENO"
            fi
            FAIL_TYPE_USER_PLANE=0
            FAIL_TYPE_S1=0
            IN_FAILURE_STATE=0
        fi

        echo "Sleeping 10 seconds"
        sleep $LOOP_DELAY
    done
}

probe_run "$@"
