#!/bin/bash

###############################################################################
# This script saves temporarily the iptables of the host, then compare it
# with the "iptables.base" which is our baseline one.
# If they differ, the script applies the "iptables.base" on the host.
###############################################################################
source utils.sh

IPTABLES_SAVE_FILE=""

declare -r IPTABLES_BASE_PATH="/usr/bin/"
declare -r IPTABLES_BASE_FILE="${IPTABLES_BASE_PATH}iptables.base"
declare -r IPTABLES_LOG_ERROR_FOLDER="/var/log/venko/"
declare diff=0

function __get_iptables() {
    if [[ ! -f "${IPTABLES_SAVE_FILE}" ]]; then
        touch "${IPTABLES_SAVE_FILE}"
    fi
    iptables-save > "${IPTABLES_SAVE_FILE}"
}

function __compare_iptables() {
    diff=0

    #-A PREROUTING -d 192.88.99.142/32 -p tcp -m tcp --dport 48080 -j DNAT --to-destination 10.0.2.1
    #-A FORWARD -i gtp_br0 -p tcp -m tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss 1400
    #-A FORWARD -o gtp_br0 -p tcp -m tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss 1400
    #-A POSTROUTING -o <interface> -j MASQUERADE

    first="\-A PREROUTING -d 192.88.99.142/32 -p tcp -m tcp --dport 48080 -j DNAT --to-destination 10.0.2.1"
    second="\-A FORWARD -i gtp_br0 -p tcp -m tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss 1400"
    third="\-A FORWARD -o gtp_br0 -p tcp -m tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss 1400"

    interface=$(ip r s default | awk '{print $5}')
    check_interface "${interface}"
    fourth="\-A POSTROUTING -o $interface -j MASQUERADE"

    result=$(cat "$IPTABLES_SAVE_FILE" | grep "$first")
    if [[ -z "${result}" ]]; then
        diff=1;
        send_notification "[$DEPLOY][$(date)] : Error! not found PREROUTING rule"
        log_message "[$DEPLOY][$(date)] : Error! not found PREROUTING rule" "${FUNCNAME[0]}" "$LINENO"
    fi

    result=$(cat "$IPTABLES_SAVE_FILE" | grep "$second")
    if [[ -z "${result}" ]]; then
        diff=1;
        send_notification "[$DEPLOY][$(date)] : Error! not found FORWARD i rule"
        log_message "[$DEPLOY][$(date)] : Error! not found FORWARD i rule" "${FUNCNAME[0]}" "$LINENO"
    fi

    result=$(cat "$IPTABLES_SAVE_FILE" | grep "$third")
    if [[ -z "${result}" ]]; then
        diff=1;
        send_notification "[$DEPLOY][$(date)] : Error! not found FORWARD o rule"
        log_message "[$DEPLOY][$(date)] : Error! not found FORWARD o rule" "${FUNCNAME[0]}" "$LINENO"
    fi

    result=$(cat "$IPTABLES_SAVE_FILE" | grep "$fourth")
    if [[ -z "${result}" ]]; then
        diff=1;
        send_notification "[$DEPLOY][$(date)] : Error! not found POSTROUTING o eth0 rule"
        log_message "[$DEPLOY][$(date)] : Error! not found POSTROUTING o eth0 rule" "${FUNCNAME[0]}" "$LINENO"
    fi

    if [[ $diff -ne 0 ]]; then
        mkdir -p $IPTABLES_LOG_ERROR_FOLDER || { echo "Error: could not create folder $IPTABLES_LOG_ERROR_FOLDER"; }
        cp $IPTABLES_SAVE_FILE $IPTABLES_LOG_ERROR_FOLDER || { echo "Error: could not copy file $IPTABLES_SAVE_FILE"; }
    fi

    rm $IPTABLES_SAVE_FILE
}

function __apply_iptables() {
    local DEPLOY=${1:-no-client-set}

    if [[ ! -f "${IPTABLES_BASE_FILE}" ]]; then
        echo "Error: File $IPTABLES_BASE_FILE does not exist"
        return 1
    fi

    interface=$(ip r s default | awk '{print $5}')
    check_interface "${interface}"
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    sed -i "s/<interface>/$interface/g" "${IPTABLES_BASE_FILE}"

    iptables-restore < "${IPTABLES_BASE_FILE}"
    if [[ $? -ne 0 ]]; then
        echo "Error: iptables-restore failed"
        return 1
    fi
}

function itables_run() {
    local DEPLOY=${1:-no-client-set}

    IPTABLES_SAVE_FILE="iptables-$(date +%Y-%m-%d-%H%M%S).save"

    __get_iptables
    __compare_iptables

    if [[ $diff -ne 0 ]]; then
        __apply_iptables "$DEPLOY"
        if [[ $? -ne 0 ]]; then
            send_notification "[$DEPLOY][$(date)] : Failed to apply iptables"
            log_message "[$DEPLOY][$(date)] : Failed to apply iptables" "${FUNCNAME[0]}" "$LINENO"
        fi
    fi
}

#DEPLOY=${1:-no-client-set}
#itables_run "$DEPLOY"
