#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# --------------------------------------------------------------------------#
import json
# Usage: python3 generate_keyring_key.py -g

import sys
import os
import shutil
import stat
import argparse
import getpass
from subprocess import STDOUT, check_output
import keyring
from keyrings.cryptfile.cryptfile import CryptFileKeyring

FILE_FLAGS = os.O_WRONLY | os.O_CREAT | os.O_EXCL  # Refer to "man 2 open".
FILE_MODE = stat.S_IRUSR | stat.S_IWUSR  # This is 0o600.
FILE_ENCODING = 'utf-8'
FILE_UMASK = 0o777 ^ FILE_MODE  # Prevents always downgrading umask to 0.

def generate_keyring_key(keyring_key_path: str) -> str:
    """
    Generates keyring key and writes to a specified file path

    Args:
        keyring_key_path (str) : Path to keyring key file location on filesystem

    Returns:
        (str) Keyring key
    """
    # Generate key
    new_key = check_output(['openssl', 'rand', '-base64', '2048'], stderr=STDOUT, universal_newlines=True)

    # Delete existing key
    if os.path.isfile(keyring_key_path):
        os.remove(keyring_key_path)

    # Open file with 600 permissions and write key
    umask_original = os.umask(FILE_UMASK)
    try:
        fdesc = os.open(keyring_key_path, FILE_FLAGS, FILE_MODE)
    finally:
        os.umask(umask_original)
    with os.fdopen(fdesc, 'w', encoding=FILE_ENCODING) as fout:
        fout.write(new_key)
    return new_key

def set_keyring(keyring_cryptfile_path: str, keyring_key_path: str, password: str, generate: bool, usernames_to_add = None):
    """
    Sets provided MongoDB password into Python keyring

    Args:
        keyring_cryptfile_path (str) : Path to keyring data file location on filesystem
        keyring_key_path (str) : Path to keyring key file location on filesystem
        password (str) : MongoDB password to be set in keyring
        generate (bool) : Flag indicating if keyring key shall be generated
        **usernames_to_add (list) : List of usernames to set with the given password
    Returns:
        None
    """
    # Get new MongoDB password as stdin
    if password is None:
        mongodb_credential = getpass.getpass("Please enter password: ")
        mongodb_credential_verify = getpass.getpass("Verify password: ")

        if mongodb_credential != mongodb_credential_verify:
            raise ValueError("Entered passwords do not match")
    else:
        mongodb_credential = password

    # If valid input, delete existing keyring and write new value using new key
    if mongodb_credential:

        if generate:
            # Generate key
            keyring_key = generate_keyring_key(keyring_key_path)
        else:
            # Using existing key
            if os.path.isfile(keyring_key_path):
                key_file = open(keyring_key_path, "r", encoding=FILE_ENCODING)
                keyring_key = key_file.read()
            else:
                raise ValueError("Keyring Key File Not Found: {}".format(keyring_key_path))

        #  Set env var for keyring path
        os.environ["KEYRING_CRYPTFILE_PATH"] = keyring_cryptfile_path

        # Delete existing keyring
        if generate and os.path.isfile(keyring_cryptfile_path):
            os.remove(keyring_cryptfile_path)

        # Instantiating keyring object
        kr = CryptFileKeyring()
        kr.keyring_key = keyring_key
        keyring.set_keyring(kr)

        # Write all username and passwords to the keyring data file
        for username in usernames_to_add:
            keyring.set_password("ponmgr", username, mongodb_credential)

        # Changing ownership of key and data files
        if generate:
            shutil.chown(keyring_cryptfile_path, 'www-data', 'www-data')
            shutil.chown(keyring_key_path, 'www-data', 'www-data')

    else:
        raise ValueError("Entered password not valid")


if __name__ == "__main__":
    if os.geteuid() != 0:
        raise Exception("Must use root permissions. (Sudo)")
    try:
        parser = argparse.ArgumentParser(add_help=True, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
        parser.add_argument("-d", "--keyring", action="store", dest="keyring", default="/etc/tibit/ponmgr/keyring.data", required=False, help="Path for keyring data cryptfile")
        parser.add_argument("-k", "--key", action="store", dest="key", default="/etc/tibit/ponmgr/keyring.key",required=False, help="Path for keyring key")
        parser.add_argument("-p", "--password", action="store", dest="password", required=False, help="Password for MongoDB authentication to be set in keyring")
        parser.add_argument("-g", "--generate", action="store_true", dest="generate", default=False, required=False, help="Generate keyring key")
        parser.add_argument("-t", "--target", action="store", dest="target", default="mongodb", required=False, help="Specify service to create encrypted password", choices=['mongodb', 'recovery_email'])
        parser.parse_args()
        args = parser.parse_args()

        fields_to_generate = []
        if args.target == "mongodb":
            fields_to_generate = ['user_database.password', 'databases.Default.password']
        elif args.target == "recovery_email":
            fields_to_generate = ['recovery_email.password']
        else:
            print("Invalid Target given. Use 'mongodb' or 'recovery_email'")
            sys.exit(2)

        set_keyring(args.keyring, args.key, args.password, args.generate, usernames_to_add=fields_to_generate)
        print("Password(s) successfully set")
    except Exception as err:
        print("ERROR: Unable to set password in keyring - {}".format(err))
        sys.exit(1)

