"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""
import json
import os
import pprint
import time
import datetime
import hashlib
import traceback

from django.conf import settings
from django.http import HttpResponse, HttpResponseServerError, JsonResponse, HttpResponseForbidden
from django.core.handlers.exception import response_for_exception
from django.contrib import messages
from log.PonManagerLogger import pon_manager_logger
from log.TracebackLogger import traceback_logger
from database_manager import database_manager
from urllib.parse import unquote
from api.settings import IN_PRODUCTION
import re


def update_session(get_response):
    def middleware(request):
        """ Updates the request.session database field for every request if it is not the correct selection """
        _url = str(request.path)
        set_session_data = True

        # All request except for users/authenticate/ and user/exist/
        if "user/logout/" in _url or "users/logout/" in _url:
            try:
                request.session.update({"database": ""})
            except KeyError:
                pass  # Do not care if the session key was not found if only trying to remove it
        elif request.user.is_authenticated and "users/authenticate/" not in _url and "user/exist/" not in _url:
            # Get session db from django sessions, if the "database" field doesn't exist it creates one and sets "" as its value.
            database_id = request.session.get("database", "")
            if database_id == "":
                try:
                    database_id = database_manager.get_users_selected_database(request.user.email)
                    request.session.update({"database": database_id})
                except KeyError:
                    pon_manager_logger.warning("User's selected database was not found. Switching user to 'Default'")
                    database_id = "Default"
                    database_manager.set_users_selected_database(request.user.email, database_id)
                    request.session.update({"database": database_id})
        else:
            set_session_data = False

        if set_session_data:
            # Get client's IP address
            forwarded = request.META.get('HTTP_X_FORWARDED_FOR')
            if forwarded:
                request.session['remote address'] = forwarded.split(',')[0]
            else:
                request.session['remote address'] = request.META.get('REMOTE_ADDR')

            if request.environ['HTTP_USER_AGENT']:
                request.session['user agent'] = request.environ['HTTP_USER_AGENT']

        # Execute view for request
        response = get_response(request)

        # Logging in request
        # Must be done after the view or else request.user will be Anonymous and request.session.session_key will be None
        if "users/authenticate/" in _url and response.status_code == 200:
            database_id = database_manager.get_users_selected_database(request.user.email)
            request.session.update({"database": database_id})

        return response

    return middleware


def set_headers(get_response):
    def middleware(request):
        """ Sets the response headers for every response before sending it to the client """
        _url = str(request.path)
        cookie_id = "__Host-sessionexpire="
        secure_flag = "Secure"

        if request.session.session_key:
            expiry_date = database_manager.get_session_key_expiry_date(request.session.session_key)
        else:
            expiry_date = request.session.get_expiry_date()

        if expiry_date:
            headers = {
                'set-cookie': cookie_id + expiry_date.strftime('%Y-%m-%dT%H:%M:%SZ') +
                              "; expires=" + expiry_date.strftime('%a, %d %b %Y %H:%M:%S GMT') +
                              "; Max-Age=31449600; Path=/; SameSite=Strict; " + secure_flag,
            }
        else:
            headers = {
                'set-cookie': cookie_id + '0' +
                              "; expires=" + '0' +
                              "; Max-Age=31449600; Path=/; SameSite=Strict; " + secure_flag,
            }

        try:
            referer = request.META.get('HTTP_REFERER', '')
        except Exception as e:
            referer = ''

        # applying swagger changes for development
        if not IN_PRODUCTION:
            # excluding swagger to set content type
            if 'docs/swagger' not in _url and 'docs/schema' not in _url and 'docs/swagger' not in referer:
                headers['Content-Type'] = 'utf-8'
            else:
                # setting the content type for swagger page rendering
                if 'docs/swagger' in _url:
                    headers["Content-Type"] = "text/html"
                # setting the content type of the open api document
                elif 'docs/schema' in _url:
                    headers["Content-Type"] = "application/vnd.oai.openapi"
                    headers['Content-Disposition'] = 'attachment; filename=\"PON Manager REST API.yml\"'
                # setting the content type for the response body to format properly
                else:
                    headers["Content-Type"] = "application/json"
        # applying swagger changes for production
        else:
            # excluding swagger so apache config will handle content type
            if 'docs/swagger' not in _url and 'docs/schema' not in _url and 'docs/swagger' not in referer:
                headers['Content-Type'] = 'utf-8'

        if expiry_date == 0:
            # Sends 503 with custom message. Prevents other pieces of code failing and taking time to return
            return HttpResponse("user database is unavailable", status=503)
        else:
            response = get_response(request)
            response.headers = headers

        return response

    return middleware


def handle_server_errors(get_response):
    def middleware(request):
        """ Overrides the response body in the case of certain errors """
        response = get_response(request)

        if response.status_code >= 500:
            # Only overwrite default message. Allows for custom 500 catching and message if desired
            if "<h1>" in str(response.content):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": "Internal server error. See server logs for details."
                    }
                })
            # Catch APIException for database being offline/empty
            elif "detail" in json.loads(response.content):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": json.loads(response.content)["detail"]
                    }
                })
        # Alert client if they made a request to a URL without a trailing slash
        elif response.status_code == 404:
            # Only overwrite default message
            if "<h1>" in str(response.content) and not str(request.path).endswith("/"):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": "URL not found. All URLs must end with a slash ('/')"
                    }
                })
        elif response.status_code == 403:
            # Only overwrite default message
            if "<h1>" in str(response.content):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": "You are not authorized to access this resource"
                    }
                })

        return response

    return middleware


def _get_nested_value(dictionary, key_list):
    for key in key_list[:-1]:
        dictionary = dictionary[key]
    return dictionary[key_list[-1]]


def _set_nested_value(dictionary, key_list, value):
    for key in key_list[:-1]:
        if type(key) is str:
            dictionary = dictionary.setdefault(key, {})
        else:
            dictionary = dictionary[key]
    dictionary[key_list[-1]] = value


def _hash(value: str):
    h = hashlib.shake_128(value.encode('utf-8'))
    # return sha128 hash value of length 20
    return f'sha128({h.hexdigest(20)})'


class log_request:
    def __init__(self, get_response):
        self.get_response = get_response
        self.stack_trace = None
        self.last_settings_update = None
        self.pon_mgr_settings = None

    def _parse_request_info(self, request):
        full_uri = request.build_absolute_uri()
        # Splitting full_uri on a path with spaces causes error
        formatted_path = request.path.replace(" ", "%20")
        params = full_uri.split(formatted_path)[1]
        url = request.path + params
        method = request.method
        email = "unauthenticated"

        # Get client's IP address
        forwarded = request.META.get('HTTP_X_FORWARDED_FOR')
        if forwarded:
            client_addr = forwarded.split(',')[0]
        else:
            client_addr = request.META.get('REMOTE_ADDR')

        if request.user.is_authenticated:
            email = request.user.email.lower()

        return {'email': email, 'client_addr': client_addr, 'url': url, 'method': method}

    def _parse_response_info(self, request, response, email='unauthenticated'):
        data_change = ""
        status = response.status_code
        if hasattr(response, "data_change"):
            data_change = response.data_change
        if hasattr(response, "data"):
            if isinstance(response.data, dict) and "details" in response.data:
                data_change = str(response.data['details'])
                if isinstance(response.data['details'], dict) and "message" in response.data['details']:
                    data_change = str(response.data['details']['message'])

        # Check for email again (won't have email previously if request is to login)
        if request.user.is_authenticated:
            email = request.user.email.lower()

        return {'data_change': data_change, 'email': email or None, 'status': status}

    def __call__(self, request):
        response = None

        try:
            request_info = self._parse_request_info(request)
            body = request.body

            start_process_time = time.process_time()
            start_duration = datetime.datetime.now()

            # Complete request
            response = self.get_response(request)

            process_time = round((time.process_time() - start_process_time) * 100, 2)
            duration = round((datetime.datetime.now() - start_duration).total_seconds() * 1000, 2)

            response_info = self._parse_response_info(request, response, email=request_info['email'])

            try:
                # All per request logging is handled here
                # If the request is a PUT/POST/PATCH then the old and new data must be passed to the PonManagerApiResponse object
                # Any exception that is thrown will log via the process_exception hook. Some 500s may not throw exception, allow 500s if they have certain values in the response

                exception_msg = ''
                stack_trace = None
                storage = messages.get_messages(request)
                for message in storage:  # If there is a message then the exception was caught within the view and process_exception won't be hit
                    if message.level == 40:  # Error level for the exception type
                        if message.extra_tags == 'stacktrace':  # Extra tag for the stacktrace
                            stack_trace = message.message
                        else:
                            exception_msg = message.message

                if int(response_info['status']) < 500 or exception_msg is not None:
                    log_message = "{} {} {} {} status: {} {}ms {} {}".format(request_info['client_addr'],
                                                                             response_info['email'],
                                                                             request_info['method'].upper(),
                                                                             request_info['url'],
                                                                             response_info['status'], duration,
                                                                             response_info['data_change'],
                                                                             exception_msg)
                    if int(response_info['status']) < 400 or int(response_info['status']) == 404:
                        pon_manager_logger.info(log_message)
                    elif int(response_info['status']) == 400:
                        pon_manager_logger.warning(log_message)
                    elif int(response_info['status']) > 400 and int(response_info['status']) != 404:
                        pon_manager_logger.error(log_message)
                    self.debug_log(request, response=response, stack_trace=stack_trace, elapsed=duration,
                                   process_time=process_time, body=body, parsed_request=request_info,
                                   status=response_info['status'])
            except Exception as e:
                try:
                    pon_manager_logger.error(f"Failed to log request: {e}")
                except Exception:
                    print(f"Failed to log request: {e}")
        except Exception as e:
            if response is None:
                response = self.get_response(request)
            pon_manager_logger.error(f"Failed to log request for: {request.build_absolute_uri()} - {e}")

        return response

    def process_exception(self, request, exception):
        # django method to convert known 4xx exceptions into 4xx responses
        response = response_for_exception(request, exception)
        if hasattr(response, 'status_code'):
            if response.status_code < 500:
                return
            status_code = response.status_code
        else:
            status_code = 500

        stack_trace = traceback.format_exc()
        request_info = self._parse_request_info(request)

        # normal ponMgr.log
        try:
            built_in_exception = False
            if hasattr(exception, '__class__') and hasattr(exception.__class__,
                                                           '__module__') and exception.__class__.__module__ == 'builtins':
                built_in_exception = True

            if built_in_exception:
                log_message = f"{request_info['client_addr']} {request_info['email']} {request_info['method'].upper()} {request_info['url']} status: {status_code} {type(exception)} {exception}"
            else:
                log_message = f"{request_info['client_addr']} {request_info['email']} {request_info['method'].upper()} {request_info['url']} status: {status_code} {type(exception)}"

            pon_manager_logger.error(log_message)
        except Exception as err:
            pon_manager_logger.error(f"Failed to log 500 request for: {request.build_absolute_uri()} - {err}")

        # ponMgrTrace.log
        try:
            self.debug_log(request, stack_trace=stack_trace, status=status_code)
        except Exception as err:
            traceback_logger.error(f"Failed to log request for: {request.build_absolute_uri()} - {err}")

    def _get_keys(self, dictionary):
        for key, value in dictionary.items():
            if type(value) is dict:
                for nested_key in self._get_keys(value):
                    yield nested_key + [key]
            elif type(value) is list:
                for index, list_value in enumerate(value):
                    if type(list_value) is dict:
                        for nested_key in self._get_keys(list_value):
                            yield nested_key + [index, key]
                    else:
                        yield [key]
            else:
                yield [key]

    def _hash_sensitive_values(self, document):
        keys_to_hash = ('password', 'sessionid', 'current_password', 'newPassword')
        for nested_key_list in self._get_keys(document):
            if nested_key_list[0] in keys_to_hash:
                key_list_in_order = nested_key_list[::-1]
                unhashed_value = _get_nested_value(document, key_list_in_order)
                hashed_value = _hash(unhashed_value)
                _set_nested_value(document, key_list_in_order, hashed_value)
        return document

    def debug_log(self, request, stack_trace=None, response=None, elapsed='-', process_time='-', body=None,
                  parsed_request=None, status=None):
        current_time = datetime.datetime.now()

        # Retrieve settings every minute
        if self.last_settings_update is None or current_time > self.last_settings_update + datetime.timedelta(
                seconds=60):
            pon_mgr_settings = database_manager.get_ponmgr_cfg(projection={'Trace Logging': 1})
            self.pon_mgr_settings = pon_mgr_settings
            self.last_settings_update = current_time
        else:
            pon_mgr_settings = self.pon_mgr_settings

        try:
            trace_enable = pon_mgr_settings['Trace Logging']['Enable']
            http_method_list = pon_mgr_settings['Trace Logging']['HTTP Methods']
            url_filter_list = pon_mgr_settings['Trace Logging']['URL Filter']
            status_code_list = pon_mgr_settings['Trace Logging']['Status Codes']
            email_filter_list = pon_mgr_settings['Trace Logging']['Users']
            client_filter_list = pon_mgr_settings['Trace Logging']['Clients']
        except Exception as e:
            traceback_logger.error(e)
            return

        if trace_enable:
            if not parsed_request:
                parsed_request = self._parse_request_info(request)

            url = parsed_request['url']
            client_addr = parsed_request['client_addr']
            email = parsed_request['email']
            method = parsed_request['method']

            # All trace log filters
            if (url_filter_list == [] or any(url_filter in url for url_filter in url_filter_list)) \
                    and (http_method_list == [] or method.upper() in http_method_list) \
                    and (status_code_list == [] or status in status_code_list) \
                    and (email_filter_list == [] or email in email_filter_list) \
                    and (client_filter_list == [] or client_addr in client_filter_list):

                headers = request.headers.__dict__
                session_mongo_id = getattr(request.session, '_mongo_id', '-')
                cookie = list(headers['_store']['cookie'])
                x_csrf_token = None
                if 'x-csrftoken' in headers['_store']:
                    x_csrf_token = list(headers['_store']['x-csrftoken'])

                try:
                    referrer = headers['_store']['referer']
                except Exception:
                    referrer = '-'
                try:
                    user_agent = headers['_store']['user-agent']
                except Exception:
                    user_agent = '-'

                try:
                    # find session id
                    regex_search = re.search('sessionid=([^;]*);', cookie[1])
                    if regex_search is None:
                        regex_search = re.search("sessionid=([^;]*)'", repr(cookie[1]))
                    if regex_search is None:
                        regex_search = re.search('sessionid=([^;]*)"', repr(cookie[1]))
                    session_id = regex_search.group(1)

                    # find csrf token
                    regex_search = re.search('csrftoken=([^;]*);', cookie[1])
                    if regex_search is None:
                        regex_search = re.search("csrftoken=([^;]*)'", repr(cookie[1]))
                    if regex_search is None:
                        regex_search = re.search('csrftoken=([^;]*)"', repr(cookie[1]))
                    csrf_token = regex_search.group(1)
                except Exception:
                    session_id = '-'
                    csrf_token = '-'

                if session_id and session_id != '-':
                    cookie[1] = cookie[1].replace(session_id, _hash(session_id))
                if csrf_token and csrf_token != '-':
                    cookie[1] = cookie[1].replace(csrf_token, _hash(csrf_token))
                if x_csrf_token and x_csrf_token[1]:
                    x_csrf_token[1] = _hash(x_csrf_token[1])

                try:
                    response_data = response.data
                except Exception:
                    response_data = None

                message = f'{client_addr} {email} {method.upper()} {url} status: {status} pid: {os.getpid()} duration: {elapsed}ms process time: {process_time}ms \n' \
                          f'referer: {referrer} user agent: {user_agent}\n' \
                          f'cookie: {cookie}\n' \
                          f'x-csrftoken: {x_csrf_token} session id mongodb: {session_mongo_id}\n'
                if response and body and (method.upper() == 'PUT' or method.upper() == 'POST'):
                    message += f'request data: {pprint.pformat(self._hash_sensitive_values(json.loads(body)))}\n'
                if response:
                    message += f'response data: {pprint.pformat(response_data)}\n'
                if stack_trace:
                    message += f'{stack_trace}\n'

                if int(status) < 400 or int(status) == 404:
                    traceback_logger.info(message)
                elif int(status) == 400:
                    traceback_logger.warning(message)
                elif int(status) > 400 and int(status) != 404:
                    traceback_logger.error(message)


def get_user_db_from_cookie(get_response):
    # Gets the user selected database from the cookie passed from the client.
    # If no cookie exists, it queries the user_db for the active database.
    # This works for clients like the PM but not for sessions.

    def middleware(request):
        # Code to be executed for each request before
        _url = str(request.path)

        if ("users/authenticate/" not in _url and
                "user/recover/" not in _url and
                "user/password/requirements/unauthenticated/" not in _url and
                "user/recover/" not in _url and
                "user/exist/" not in _url and
                request.user.is_authenticated):
            try:
                formatted_db = unquote(request.COOKIES['__Host-user_db'], encoding='utf-8', errors='replace')
                request.user.user_db = formatted_db
                request.session.update({"database": formatted_db})
            except Exception as e:
                try:
                    user = request.user.email
                    db_name = database_manager.get_users_selected_database(user)
                    request.COOKIES['__Host-user_db'] = db_name
                    formatted_db = unquote(request.COOKIES['__Host-user_db'], encoding='utf-8', errors='replace')
                    request.user.user_db = formatted_db
                    request.session.update({"database": formatted_db})
                except Exception as e:
                    pon_manager_logger.error(f"Failed to get cookie in middleware - {e}")
        else:
            db_name = "Default"
            request.COOKIES['__Host-user_db'] = db_name
            request.user.user_db = request.COOKIES['__Host-user_db']

        response = get_response(request)

        # Code to be executed for each request/response after the view is called.

        return response

    return middleware
