#!/usr/bin/env python3
# --------------------------------------------------------------------------#
#  Copyright (c) 2022-2025, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
# --------------------------------------------------------------------------#
from copy import deepcopy
import json

class DatabaseInit():
    """ Represents configuration from NetconfInit.json  """

    # Default configuration settings
    default_database_info = {
        "Default": {
            "auth_db": "tibit_users",
            "auth_enable": False,
            "ca_cert_path": "",
            "db_uri": "",
            "dns_srv": False,
            "host": "10.1.20.102",
            "name": "tibit_pon_controller",
            "password": "pdmPass",
            "port": "27017",
            "replica_set_enable": False,
            "replica_set_hosts": [
                "localhost",
                "127.0.0.1"
            ],
            "replica_set_name": "rs0",
            "tls_enable": False,
            "username": "pdmPonManager"
        }
    }

    default_user_database_info = {
        "host": "10.1.20.102",
        "name": "tibit_users",
        "port": "27017",
        "auth_enable": False,
        "auth_db": "tibit_users",
        "username": "pdmUserAdmin",
        "password": "pdmPass",
        "password_opts": {"type": "password", "keyring_path": "/etc/tibit/ponmgr/keyring.data", "keyring_key_path": "/etc/tibit/ponmgr/keyring.key"},
        "tls_enable": False,
        "ca_cert_path": "",
        "compression": False,
        "dns_srv": False,
        "db_uri": "",
        "replica_set_enable": False,
        "replica_set_name": "",
        "replica_set_hosts": [],
        "django_key": ""
    }

    def __init__(self, databases="/api/databases.json", user_database="api/user_database.json"):
        self.databases_filename = databases
        self.user_database_filename = user_database
        self.databases = {}
        self.user_database_fields = {}
        self.reload()

    def _load_config_json(self, filename, config):
        """ Load the configuration from NetconfInit.json (json format) """
        try:
            with open(filename, 'r', encoding="utf-8") as in_file:
                json_data = json.load(in_file)
                if json_data:
                    config.update(json_data)
        except (FileNotFoundError, IOError):
            print(f"Configuration file {filename} not found, using defaults.")
        except Exception as err:
            print(f"Failed to load configuration from file {filename}, error: {err}")

    def reload(self, databases_json_content=None):
        """ (Re)Load contents of the databases.json and user_database.json files

        This operation starts with default configuration values defined by in this
        class and merges the contents of databases.json on top of the defaults.
        """
        database_init_json = {}
        user_database_init_json = {}
        if databases_json_content:
            database_init_json = databases_json_content
        else:
            self._load_config_json(self.databases_filename, database_init_json)
        db_defaults = deepcopy(self.default_database_info)
        self.databases = self.merge_dict(db_defaults, database_init_json)

        # user database information will only be loaded once
        if not self.user_database_fields:
            self._load_config_json(self.user_database_filename, user_database_init_json)
            user_db_defaults = deepcopy(self.default_user_database_info)
            self.user_database_fields = self.merge_dict(user_db_defaults, user_database_init_json)


    def merge_dict(self, a, b):
        """ Merges two dictionaries together, outputs result """
        if isinstance(a, dict):
            # dicts must be merged
            if isinstance(b, dict):
                for key in b:
                    if key in a:
                        a[key] = self.merge_dict(a[key], b[key])
                    else:
                        a[key] = b[key]
            else:
                print(f"Cannot merge non-dict '{b}' into dict '{a}'")
        else:
            a = b

        return a

    def as_dict(self, obj):
        """ Return the configuration as a dict """
        return deepcopy(obj)

    @property
    def databases_auth_db(self):
        """ databases.auth_db """
        return self.databases['auth_db']

    @property
    def user_database_auth_db(self):
        """ user_database.auth_db """
        return self.user_database_fields['auth_db']

    @property
    def databases_auth_enable(self):
        """ databases.auth_enable """
        return self.databases['auth_enable']

    @property
    def user_database_auth_enable(self):
        """ user_database.auth_enable """
        return self.user_database_fields['auth_enable']
    @property
    def databases_ca_cert_path(self):
        """ databases.ca_cert_path """
        return self.databases['ca_cert_path']

    @property
    def user_database_ca_cert_path(self):
        """ user_database.ca_cert_path """
        return self.user_database_fields['ca_cert_path']
    @property
    def databases_compression(self):
        """ databases.compression """
        return self.databases['compression']

    @property
    def user_database_compression(self):
        """ user_database.compression """
        return self.user_database_fields['compression']
    @property
    def databases_host(self):
        """ databases.host """
        return self.databases['host']

    @property
    def user_database_host(self):
        """ user_database.host """
        return self.user_database_fields['host']
    @property
    def databases_name(self):
        """ databases.name """
        return self.databases['name']

    @property
    def user_database_name(self):
        """ user_database.name """
        return self.user_database_fields['name']
    @property
    def databases_password(self):
        """ databases.password """
        return self.databases['password']

    @property
    def user_database_password(self):
        """ user_database.password """
        return self.user_database_fields['password']
    @property
    def databases_port(self):
        """ databases.port """
        return self.databases['port']

    @property
    def user_database_port(self):
        """ user_database.port """
        return self.user_database_fields['port']
    @property
    def databases_tls_enable(self):
        """ databases.tls_enable """
        return self.databases['tls_enable']

    @property
    def user_database_tls_enable(self):
        """ user_database.tls_enable """
        return self.user_database_fields['tls_enable']
    @property
    def databases_username(self):
        """ databases.username """
        return self.databases['username']

    @property
    def user_database_username(self):
        """ user_database.username """
        return self.user_database_fields['username']
    @property
    def databases_dns_srv(self):
        """ databases.dns_srv """
        return self.databases['dns_srv']

    @property
    def user_database_dns_srv(self):
        """ user_database.dns_srv """
        return self.user_database_fields['dns_srv']
    @property
    def databases_db_uri(self):
        """ databases.db_uri """
        return self.databases['db_uri']

    @property
    def user_database_db_uri(self):
        """ user_database.db_uri """
        return self.user_database_fields['db_uri']
    @property
    def databases_replica_set_enable(self):
        """ databases.replica_set_enable """
        return self.databases['replica_set_enable']

    @property
    def user_database_replica_set_enable(self):
        """ user_database.replica_set_enable """
        return self.user_database_fields['replica_set_enable']
    @property
    def databases_replica_set_name(self):
        """ databases.replica_set_name """
        return self.databases['replica_set_name']

    @property
    def user_database_replica_set_name(self):
        """ user_database.replica_set_name """
        return self.user_database_fields['replica_set_name']
    @property
    def databases_replica_set_hosts(self):
        """ databases.replica_set_hosts """
        return self.databases['replica_set_hosts']

    @property
    def get_databases_info(self):
        """ databases.json """
        return self.databases

    @property
    def get_user_database_info(self):
        """ user_database.json """
        return self.user_database_fields