"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import datetime
import hashlib
import json

from django.contrib.auth import authenticate, login, logout, _get_backends
from django.conf import settings
from django.contrib.auth.models import User
from django.core import serializers
from django.http import JsonResponse
from django.utils.decorators import method_decorator
from drf_spectacular.types import OpenApiTypes
from rest_framework.fields import EmailField, CharField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiRequest, OpenApiResponse
from rest_framework import status

import DBdjango
from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import RequestSerializer
from utils.tools import PonManagerApiResponse, validate_data, permission_required
from utils.serializers import schema, get_schema


use_mongo_engine = True
if 'django.contrib.auth.backends.ModelBackend' in settings.AUTHENTICATION_BACKENDS:
    use_mongo_engine = False


# ==================================================
# ============ Authenticate User View ==============
# ==================================================
class Authenticate(GenericAPIView):
    queryset = ''

    swaggerSchema = get_schema('auth-user')

    @extend_schema(
        operation_id="authenticate_user",
        request=RequestSerializer(name="AuthenticateUser", data_fields={
            "email": EmailField(),
            "password": CharField()
        }),
        responses=None,
        # request={
        #     "application/json": schema(swaggerSchema),
        # },
        # responses={
        #     201: OpenApiResponse(response=schema(swaggerSchema),
        #                          description='Created'),
        # },
        # request= OpenApiRequest(request= None),
        tags=['users'],
        summary="Authenticate and login the user",
        description=" "
    )
    @validate_data(collection="auth-user", resource_id_param=None, validate_required=False)
    def post(self, request, data, version):
        """Authenticate and login the user"""
        DBdjango.views.purge_expired_sessions(request)
        DBdjango.views.purge_inactive_radius_users(request)
        DBdjango.views.purge_inactive_recovery_code()

        email = data['email'].lower()
        password = hashlib.md5(data['password'].encode('utf-8')).hexdigest()

        pon_mgr_cfg = database_manager.get_ponmgr_cfg()
        user_info = database_manager.get_users_database_type(email)
        is_radius_user = False

        user = None

        # Local user exists
        if user_info and 'email' in user_info and ('database_type' not in user_info or (
                'database_type' in user_info and user_info['database_type'] != 'radius')):
            # responsible for actual authentication. If it succeeds, it returns an object of user data. If it fails, it returns nothing
            if use_mongo_engine:
                user = authenticate(username=email, password=password, backend='backends.mongoengine.MongoBackend')
            else:
                user = authenticate(username=email, password=password,
                                    backend='django.contrib.auth.backends.ModelBackend')
        elif 'Radius' in pon_mgr_cfg and 'Enable' in pon_mgr_cfg['Radius'] and pon_mgr_cfg['Radius'][
            'Enable']:  #No local user
            is_radius_user = True
            user = authenticate(username=email, password=data['password'], backend='backends.radius.RADIUSBackend')

        # If user is not null, that means it returned user data, which means user authentication was successful
        if user is not None:
            if not use_mongo_engine:
                user = User.objects.get(
                    username=email)  # Getting user object based on primary key Username( which is email)

            user.last_login = datetime.datetime.now(datetime.timezone.utc)

            if is_radius_user:
                user.password = ''
                if use_mongo_engine:
                    user.is_active = True

            user.save()

            # Set user session timeout
            # Database value is in minutes. However, the set_expiry method takes seconds.
            timeout_sec = database_manager.get_user_session_expiry(email)
            timeout_min = timeout_sec * 60
            request.session.set_expiry(timeout_min)
            request.session.__setitem__('creation_date', user.last_login.strftime('%Y-%m-%d %T'))

            # Get and save the user selected database in the users session
            try:
                active_database = database_manager.get_users_selected_database(user_email=email)
            except KeyError:
                print("Users selected database was not found. Using 'Default'")
                active_database = "Default"
                database_manager.set_users_selected_database(user_email=email, database_id=active_database)

            # Set the user connection type to radius if a non pon manager db user
            if is_radius_user:
                database_manager.set_users_database_type(email, 'radius')
                request.session.__setitem__("_auth_user_backend", 'backends.radius.RADIUSBackend')

            # Update private endpoints database reference
            database_manager.set_users_selected_database(email, active_database)
            request.session.update({"database": active_database})

            # Update database reference for the local user object
            if use_mongo_engine:
                user.active_database = active_database

            # Log in the user
            if use_mongo_engine:
                for backend, backend_path in _get_backends(return_tuples=True):
                    if backend_path == 'backends.mongoengine.MongoBackend':
                        session_backend_path = backend_path
                        if is_radius_user:
                            session_backend_path = 'backends.radius.RADIUSBackend'
                        backend.login(request, user, backend=session_backend_path)
            else:
                if is_radius_user:
                    login(request, user, backend='backends.radius.RADIUSBackend')
                else:
                    login(request, user, backend='django.contrib.auth.backends.ModelBackend')

            # User data to be used in view context
            msg = {
                'auth': 'True',
                'fName': user.first_name,
                'lName': user.last_name,
                'email': user.email,
                'lastLogin': user.last_login,
                'dateJoined': user.date_joined,
                'roles': database_manager.get_user_groups(user.email),
                'permissions': database_manager.get_user_permissions(user.email),
                'active_database': active_database
            }
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=msg)
        else:
            response = PonManagerApiResponse(status=status.HTTP_401_UNAUTHORIZED, data={'auth': 'False'})

        return response


# ==================================================
# =============== Logout User View =================
# ==================================================
class Logout(GenericAPIView):
    queryset = ''

    @extend_schema(
        operation_id="logout_user",
        responses=None,
        tags=['users'],
        summary="Logout the user",
        description=" "
    )
    def get(self, request, version):
        """ Logout the user """
        # Logout always return successfully regardless if user is logged in/exists
        logout(request)
        response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
        response.delete_cookie('__Host-user_db', samesite='strict')
        response.delete_cookie('__Host-sessionexpire', samesite='strict')
        response.delete_cookie('__Host-csrftoken', samesite='strict')
        response.delete_cookie('__Host-sessionid', samesite='strict')

        return response


# ==================================================
# ================ User Logs View ==================
# ==================================================
class Logs(GenericAPIView):
    queryset = ''

    @extend_schema(
        operation_id="get_user_logs",
        parameters=[
            OpenApiParameter(name="start-time", description="UTC timestamp to begin getting logs at",
                             type=OpenApiTypes.DATETIME, required=True)
        ],
        responses=None,
        tags=['users'],
        summary="Get all logs",
        description=" "
    )
    @method_decorator(permission_required('can_read_accounts_admin', raise_exception=True))
    def get(self, request, version):
        """ Get all logs """
        start_time = request.GET.get('start-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             data="Parameter 'start-time' is required")
        else:
            collection = database_manager.user_database.get_collection("SYSLOG-ACTIONS")
            logs = collection.find({"Time": {"$gt": start_time}}, no_cursor_timeout=True)
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=list(logs))

        return response
