"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from drf_spectacular.types import OpenApiTypes
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import get_nested_value, PonManagerApiResponse, validate_data, validate_query_params, permission_required_any_of, validate_device_id, permission_required
from utils.serializers import schema, get_schema




# ==================================================
# ============= One Switch State View ==============
# ==================================================
class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_one_switch_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['switch state'],
        summary="Get the state for the specified Switch",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the state for the specified Switch"""
        # Validate Switch ID
        if not validate_device_id(swi_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Invalid Switch ID format")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("OLT-STATE")
            cursor = collection.find({"Switch.Chassis ID": swi_id})
            res_data = None
            if cursor.collection.count_documents({}) > 0:
                res_data = cursor.sort("Time", pymongo.DESCENDING)[0]
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"Switch ID "+str(swi_id)+" has no state document"})

        return response


# ==================================================
# ============== Switch States View ================
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(

        operation_id="get_switch_states",
        parameters=[
            OpenApiParameter(name="scale_db", description="true/false sort by Time",
                             type=OpenApiTypes.BOOL, required=False),
        ],
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch state'],
        summary="Get the states for all Switches",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    def get(self, request, version):
        scale_db = request.GET.get('scale_db', None)
        """Get the states for all Switches"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("OLT-STATE")
        cursor = collection.find({}, {"_id": 0, "Switch": 1, "Time": 1})
        if cursor.collection.count_documents({}) > 0 and not scale_db:
            cursor = cursor.sort("Time", pymongo.DESCENDING)

        # Duplicate switch states get overwritten by most recent entry
        all_switch_states = {}
        for switch_state in cursor:
            if switch_state["Switch"]:
                switch_id = switch_state["Switch"]["Chassis ID"]
                all_switch_states[switch_id] = switch_state["Switch"]
                all_switch_states[switch_id]["Time"] = switch_state["Time"]
        res_data = all_switch_states.values()

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========= One Switch Configuration View ==========
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-CFG')

    @extend_schema(

        operation_id="get_one_switch_config",
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch config'],
        summary="Get the config for the specified Switch",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the config for the specified Switch"""
        # Validate Switch ID
        if not validate_device_id(swi_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid Switch ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-CFG", query={"_id": swi_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"Switch ID "+str(swi_id)+" has no config document"})

        return response

    swaggerSchema = get_schema('SWI-CFG')

    @extend_schema(

        operation_id="put_one_switch_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch config'],
        summary="Update the config for the specified Switch",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_switches', 'can_create_network_switches'], raise_exception=True))
    @validate_data(collection="SWI-CFG", resource_id_param="swi_id")
    def put(self, request, data, swi_id, version):
        """Update the config for the specified Switch"""
        data['SWI']['CFG Change Count'] += 1
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SWI-CFG", query={"_id": swi_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    swaggerSchema = get_schema('SWI-CFG')

    @extend_schema(

        operation_id="delete_one_switch_config",
        responses=None,
        tags=['switch config'],
        summary="Delete the config of the specified Switch",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_switches', raise_exception=True))
    def delete(self, request, swi_id, version):
        """Delete the config of the specified Switch"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="SWI-CFG", query={"_id": swi_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== Switch Configurations View ============
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-CFG')

    @extend_schema(
        operation_id="get_switch_configs",
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch config'],
        summary="Get the configs for all Switches",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    @validate_query_params(collection="SWI-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the configs for all Switches"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="SWI-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="SWI-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('SWI-CFG')

    @extend_schema(

        operation_id="post_switch_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch config'],
        summary="Create the provided Switch config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_switches', raise_exception=True))
    @validate_data(collection="SWI-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided Switch config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="SWI-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            swi_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={"message":f"Switch configuration with id {swi_id} already exists"})

        return response


# ==================================================
# ============= One Switch Logs View ===============
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-CFG')

    @extend_schema(

        operation_id="get_one_switch_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch config'],
        summary="Get the logs of the specified Switch between the start and end times",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the logs of the specified Switch between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'start-time' is required")
        # Validate Switch ID
        elif not validate_device_id(swi_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid Switch ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("SYSLOG-SWI-{}".format(swi_id.replace(":", "")))
            if end_time is None:
                res_data = list(collection.find({"time": {"$gte": start_time}},
                                                {"_id": 0, "device ID": 0}).limit(10000))
            else:
                res_data = list(collection.find({"time": {"$gte": start_time, "$lte": end_time}},
                                                {"_id": 0, "device ID": 0}).limit(10000))
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"Switch ID "+str(swi_id)+" has no logs documents"})

        return response

    @extend_schema(
        operation_id="delete_one_switch_logs",
        responses=None,
        tags=['switch logs'],
        summary="Delete the Logs of the specified Switch",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_switches', raise_exception=True))
    def delete(self, request, swi_id, version):
        """Delete the Logs of the specified Switch"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("SYSLOG-SWI-{}".format(swi_id.replace(":", "")))
        collection.drop()

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== One Switch Automation State View ========
# ==================================================
class OneAutomationState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="get_one_switch_automation_state",
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation state'],
        summary="Get the Automation State of the specified Switch",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the Automation State of the specified Switch"""
        # Validate Switch ID
        if not validate_device_id(swi_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid Switch ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-AUTO-STATE", query={"_id": swi_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"Switch ID "+str(swi_id)+" has no automation state document"})

        return response

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="delete_one_switch_automation_state",
        responses=None,
        tags=['switch automation state'],
        summary="Delete the specified Switch Automation State",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_switches', raise_exception=True))
    def delete(self, request, swi_id, version):
        """Delete the specified Switch Automation State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="SWI-AUTO-STATE", query={"_id": swi_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== Switch Automation States View =========
# ==================================================
class AutomationStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="get_switch_automation_states",
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation state'],
        summary="Get the Automation States of all Switches",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    # TODO @validate_query_params(collection="SWI-AUTO-STATE")
    def get(self, request, version):
        """Get the Automation States of all Switches"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="SWI-AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ======== One Switch Automation Config View =======
# ==================================================
class OneAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="get_one_switch_automation_config",
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation config'],
        summary="Get the specified Switch Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified Switch Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"CFG ID "+str(cfg_id)+" has no automation config document"})

        return response

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="put_one_switch_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation config'],
        summary="Update the config for the specified Switch Automation Config",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_switches', 'can_create_network_switches'], raise_exception=True))
    @validate_data(collection="SWI-AUTO-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified Switch Automation Config"""
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Request body must be of format '{ data: <SWI-AUTO-CFG> }'"})
        else:
            data['_id'] = cfg_id
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": cfg_id}, new_document=data)
            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="delete_one_switch_automation_config",
        responses=None,
        tags=['switch automation config'],
        summary="Delete the specified Switch Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_switches', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified Switch Automation Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= Switch Automation Configs View =========
# ==================================================
class AutomationConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="get_switch_automation_configs",
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation config'],
        summary="Get the Automation Configs of all Switches",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_switches', raise_exception=True))
    # TODO @validate_query_params(collection="SWI-AUTO-CFG")
    def get(self, request, version):
        """Get the Automation Configs of all Switches"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="SWI-AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="post_switch_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation config'],
        summary="Create the provided Switch Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_switches', raise_exception=True))
    # TODO @validate_data(collection="SWI-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided Switch Automation Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            swi_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={"message":f"Switch configuration with id {swi_id} already exists"})

        return response


# ==================================================
# ====== Global Switch Automation Config View ======
# ==================================================
class GlobalAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="get_global_switch_automation_config",
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation config'],
        summary="Get the Global Switch Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the Global Switch Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": "Global"})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(

        operation_id="put_global_switch_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation config'],
        summary="Update the Global Switch Automation Config",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_automation', 'can_create_automation'], raise_exception=True))
    @validate_data(collection="SWI-AUTO-CFG", resource_id_param=None)
    def put(self, request, data, version):
        """Update the Global Switch Automation Config"""
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Request body must be of format '{ data: <SWI-AUTO-CFG> }'"})
        else:
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="SWI-AUTO-CFG", query={"_id": "Global"},
                                                                 new_document=data)
            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    swaggerSchema = get_schema('SWI-AUTO-STATE')

    @extend_schema(
        operation_id="post_global_swi_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['switch automation config'],
        summary="Create the provided Switch Automation Global Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_automation', raise_exception=True))
    # TODO @validate_data(collection="SWI-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided Switch Automation Global Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={"message":f"Switch Automation configuration with id Global already exists"})

        return response

    @extend_schema(

        operation_id="delete_global_switch_automation_config",
        parameters=[
            OpenApiParameter(name="step", description="Automation step to delete template from",
                             type=OpenApiTypes.STR, required=True),
            OpenApiParameter(name="name", description="Automation template to delete from the specified step",
                             type=OpenApiTypes.STR, required=True)
        ],
        responses=None,
        tags=['switch automation config'],
        summary="Delete the Global Switch Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, version):
        """Delete the Global Switch Automation Config"""
        step = request.GET.get('step', None)
        name = request.GET.get('name', None)

        # Return missing parameter response if step or name are undefined
        if step is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'step' is required")
        elif name is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'name' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("SWI-AUTO-CFG")
            try:
                if step.upper() == "IDENTIFY":
                    collection.update_one({'_id': "Global"}, {"$pull": {"IDENTIFY.Mapping": {"Description": name}}})
                else:
                    collection.update_one({'_id': "Global"}, {"$unset": {f"{step.upper()}.{name}": ""}})
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

        return response
