"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import PonManagerApiResponse, validate_query_params, validate_data, get_nested_value, permission_required_any_of, permission_required
from utils.serializers import schema, get_schema




# ==================================================
# ========== All SLA Configurations View ===========
# ==================================================
class Slas(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SLA-CFG')

    @extend_schema(
        operation_id="get_sla_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['sla'],
        summary="Get all SLA configurations",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_slas', raise_exception=True))
    @validate_query_params(collection="SLA-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """ Get all SLA configurations """
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="SLA-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="SLA-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('SLA-CFG')

    @extend_schema(
        operation_id="post_sla_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['sla'],
        summary="Create the provided SLA config",
        description=" "
    )
    @method_decorator(permission_required('can_create_global_config_slas', raise_exception=True))
    @validate_data(collection="SLA-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided SLA config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="SLA-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            sla_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={"message":f"SLA configuration with id {sla_id} already exists"})

        return response


# ==================================================
# =========== One SLA Configuration View ===========
# ==================================================
class OneSla(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SLA-CFG')

    @extend_schema(
        operation_id="get_sla_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['sla'],
        summary="Get the specified SLA configuration",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_slas', raise_exception=True))
    def get(self, request, sla_id, version):
        """ Get the specified SLA configuration """
        res_data = database_manager.find_one(database_id=request.session.get("database"), collection="SLA-CFG", query={"_id": sla_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"SLA ID "+str(sla_id)+" has no configuration documents"})

        return response

    swaggerSchema = get_schema('SLA-CFG')

    @extend_schema(
        operation_id="put_sla_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['sla'],
        summary="Update the provided SLA configuration",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_global_config_slas', 'can_create_global_config_slas'], raise_exception=True))
    @validate_data(collection="SLA-CFG", resource_id_param="sla_id")
    def put(self, request, data, sla_id, version):
        """ Update the provided SLA configuration """
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SLA-CFG", query={"_id": sla_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_sla_config",
        responses=None,
        tags=['sla'],
        summary="Delete the specified SLA configuration",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_slas', raise_exception=True))
    def delete(self, request, sla_id, version):
        """ Delete the specified SLA configuration """
        database_manager.delete_one(database_id=request.session.get("database"), collection="SLA-CFG", query={"_id": sla_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
