"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, inline_serializer
from drf_spectacular.types import OpenApiTypes
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import get_nested_value, PonManagerApiResponse, permission_required_any_of, validate_data, permission_required
from utils.serializers import schema, get_schema


# ==================================================
# ========= One PON Automation State View ==========
# ==================================================


class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_state",
        responses=None,
        tags=['automation']  # , 'state', 'get'
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the state for the specified PON Automation"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="AUTO-STATE", query={"_id": pon_auto_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details="PON Automation state with ID "+str(pon_auto_id)+" was not found")

        return response

    @extend_schema(
        operation_id="delete_one_pon_automation_state",
        responses=None,
        tags=['automation']
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the state of the specified PON Automation"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="AUTO-STATE", query={"_id": pon_auto_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== PON Automation States View ============
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_pon_automation_states",
        responses=None,
        tags=['automation']  # , 'state', 'get'
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the states for all PON Automation services"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ===== One PON Automation Configuration View ======
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_config",
        responses=None,
        tags=['automation']  # , 'config', 'get'
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the config for the specified PON Automation"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="AUTO-CFG", query={"_id": pon_auto_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details="PON Automation config with ID "+str(pon_auto_id)+" was not found")

        return response

    @extend_schema(
        operation_id="put_one_pon_automation_config",
        request={},
        responses=None,
        tags=['automation']  # , 'config', 'put'
    )
    @method_decorator(permission_required_any_of(['can_update_automation', 'can_create_automation'], raise_exception=True))
    @validate_data(collection="AUTO-CFG", resource_id_param="pon_auto_id")
    def put(self, request, data, pon_auto_id, version):
        """Update the config for the specified PON Automation"""
        data['AUTO']['CFG Change Count'] += 1
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="AUTO-CFG", query={"_id": pon_auto_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_pon_automation_config",
        responses=None,
        tags=['automation']
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the config of the specified PON Automation"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="AUTO-CFG", query={"_id": pon_auto_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ====== PON Automation Configurations View ========
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_pon_automation_configs",
        responses=None,
        tags=['automation']
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the configs for all PON Automation services"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_pon_automation_config",
        request={},
        responses=None,
        tags=['automation']  # , 'config', 'post'
    )
    @method_decorator(permission_required('can_create_automation', raise_exception=True))
    def post(self, request, version):
        """Create the provided PON Automation config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="AUTO-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            pon_auto_id = get_nested_value(request.data, ["data", "_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"PON Automation configuration with id {pon_auto_id} already exists")

        return response


# ==================================================
# ======= One PON Automation Statistics View =======
# ==================================================
class Statistics(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_stats",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses=None,
        tags=['automation']  # , 'stats', 'get'
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the statistics of the specified PON Automation between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'start-time' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("STATS-AUTO")

            try:
                if end_time is None:
                    res_data = list(collection.find({
                        "$and": [
                            {"device ID": pon_auto_id},
                            {"valid": True},
                            {"time": {"$gte": start_time}},
                        ]
                    }, {"_id": 0, "device ID": 0}).limit(10000))
                else:
                    res_data = list(collection.find({
                        "$and": [
                            {"device ID": pon_auto_id},
                            {"valid": True},
                            {"time": {"$gte": start_time, "$lte": end_time}}
                        ]
                    }, {"_id": 0, "device ID": 0}).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            # Add PON Automation ID to response format for easier handling in UI
            for block in res_data:
                block['mac_address'] = pon_auto_id
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details="PON Automation statistics with ID "+str(pon_auto_id)+" were not found")

        return response

    @extend_schema(
        operation_id="delete_one_pon_automation_stats",
        responses=None,
        tags=['automation']  # , 'stats', 'delete'
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the Statistics of the specified PON Automation"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("STATS-AUTO-{}".format(pon_auto_id))
        collection.drop()

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= One PON Automation Logs View ===========
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses=None,
        tags=['automation']  # , 'logs', 'get'
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the logs of the specified PON Automation between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'start-time' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("SYSLOG-AUTO")

            try:
                if end_time is None:
                    res_data = list(collection.find({
                        "$and": [
                            {"device ID": pon_auto_id},
                            {"valid": True},
                            {"time": {"$gte": start_time}},
                        ]
                    }, {"_id": 0, "device ID": 0}).limit(10000))
                else:
                    res_data = list(collection.find({
                        "$and": [
                            {"device ID": pon_auto_id},
                            {"valid": True},
                            {"time": {"$gte": start_time, "$lte": end_time}}
                        ]
                    }, {"_id": 0, "device ID": 0}).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message": "PON Automation logs with ID "+str(pon_auto_id)+" were not found"})

        return response

    @extend_schema(
        operation_id="delete_one_pon_automation_logs",
        responses=None,
        tags=['automation']  # , 'logs', 'delete'
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the Logs of the specified PON Automation"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("SYSLOG-AUTO-{}".format(pon_auto_id))
        collection.drop()

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
