"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""
from api.settings import log_file_path, IN_PRODUCTION
from log.PonManagerLogger import pon_manager_logger
import logging

_SIMPLE_FORMAT = "%(asctime)s.%(msecs)03d %(levelname)s %(message)s"
_DATE_FORMAT = "%Y-%m-%d %H:%M:%S"
_LOG_FILENAME = "ponMgrTrace.log"

LOGGING_FORMATTER = logging.Formatter(fmt=_SIMPLE_FORMAT, datefmt=_DATE_FORMAT)


class TracebackLogger:
    """
    Debug logger object for PON Manager API
    """

    def __init__(self):
        if IN_PRODUCTION:
            try:
                handler = logging.FileHandler(log_file_path.replace('ponMgr.log','ponMgrTrace.log'))
            except Exception as err:
                handler = None
                pon_manager_logger.error(f'Failure to initialize trace log handler. {err}')

        else:
            handler = logging.StreamHandler()

        if handler:
            handler.setFormatter(LOGGING_FORMATTER)
            self._traceback_logger = logging.getLogger('trace')
            self._traceback_logger.setLevel(logging.DEBUG)
            self._traceback_logger.addHandler(handler)
        else:
            self._traceback_logger = None

    def add_handler(self, log_handler):
        """ Adds a new handler to the logger

        :param log_handler: the log handler to add to the logger
        """
        if self._traceback_logger:
            self._traceback_logger.addHandler(log_handler)

    def debug(self, message):
        """ Log an DEBUG level message to all active handlers for this logger """
        if self._traceback_logger:
            self._traceback_logger.debug(message)

    def info(self, message):
        """ Log an INFO level message to all active handlers for this logger """
        if self._traceback_logger:
            self._traceback_logger.info(message)

    def warning(self, message):
        """ Log an WARNING level message to all active handlers for this logger """
        if self._traceback_logger:
            self._traceback_logger.warning(message)

    def error(self, message):
        """ Log an ERROR level message to all active handlers for this logger """
        if self._traceback_logger:
            self._traceback_logger.error(message)

    def critical(self, message):
        """ Log an CRITICAL level message to all active handlers for this logger """
        if self._traceback_logger:
            self._traceback_logger.critical(message)

traceback_logger = TracebackLogger()
