"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import datetime
import logging
import re
import socket

import pymongo
from log.PonManagerLogger import LOGGING_FORMATTER, pon_manager_logger


class DatabaseLogHandler(logging.Handler):
    """
    Logging Handler for writing messages to the users database
    """

    def __init__(self, user_database, level=logging.INFO):
        logging.Handler.__init__(self)
        self.setLevel(level)
        self.setFormatter(LOGGING_FORMATTER)
        self._user_database = user_database
        self.hostname = socket.getfqdn()

        pon_manager_logger.info("Database log handler is ready")

    def emit(self, record):
        """ Called automatically when the logger is told to log a message

        :param record: The logging module record
        """
        try:
            time = str(datetime.datetime.now())
            record_msg = str(record.__dict__['msg'])

            # Parse the message string to determine if there is a data change from a PUT/POST, if its a GET, or a system message
            if "GET" in record_msg or "PUT" in record_msg or "POST" in record_msg or "DELETE" in record_msg:
                # Strip message contents to build database document
                method_index = -1
                method_length = 0
                if "GET" in record_msg:
                    method_index = record_msg.index("GET")
                    method_length = 3
                if "PUT" in record_msg:
                    method_index = record_msg.index("PUT")
                    method_length = 3
                if "POST" in record_msg:
                    method_index = record_msg.index("POST")
                    method_length = 4
                if "DELETE" in record_msg:
                    method_index = record_msg.index("DELETE")
                    method_length = 6

                ip_and_email = record_msg[:method_index].split(" ")
                request = record_msg[method_index:]
                ip_addr = ip_and_email[0]
                user = ip_and_email[1]
                method = request[:method_length]
                status_index = request.index("status: ")
                # length of 'status: ' and a 3 digit number for indexes
                status = request[status_index + 8:status_index + 11].strip()
                url = request[method_length + 1:status_index].strip()
                message = re.sub('([0-9]+[.]{1}[0-9]{1,2}(ms){1})', '', request[status_index + 12:].strip())

                # Request log format
                log_document = {
                    "_id": f"{user} @ {time}",
                    "Time": time,
                    "Level": record.__dict__["levelname"],
                    "User": user,
                    "IP": ip_addr,
                    "Hostname": self.hostname,
                    "URL": url,
                    "Method": method,
                    "Message": message,
                    "Status": int(status)
                }
            else:
                # System message format
                log_document = {
                    "_id": f"PON Manager @ {time}",
                    "Time": time,
                    "Level": record.__dict__["levelname"],
                    "User": "PON Manager",
                    "Message": record.__dict__['msg']
                }

            collection = self._user_database.get_collection("SYSLOG-ACTIONS").with_options(write_concern=pymongo.write_concern.WriteConcern(w=0))
            collection.insert_one(log_document)
        except Exception as e:
            print(f"MongoDB error: {e}")
