"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import copy
import sys
import traceback

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from django.contrib import messages
from pymongo.errors import ConfigurationError, DuplicateKeyError
from rest_framework.fields import BooleanField, CharField, ListField
from rest_framework.generics import GenericAPIView
from rest_framework.views import APIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from rest_framework import status

import DBdjango

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import CharFieldSerializer, RequestSerializer
from utils.tools import PonManagerApiResponse, get_nested_value, validate_data, permission_required_any_of, \
    permission_required
from utils.set_keyring_passwd import set_keyring
from utils.keyring_helper import delete_keyring_username
from utils.serializers import schema, get_schema, get_unversioned_schema


# ==================================================
# ============= Databases Status View ==============
# ==================================================
class ConnectionStatus(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_unversioned_schema('MANAGE-DATABASE')

    @extend_schema(
        operation_id="get_database_statuses",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database status'],
        summary="Get the alive status of the databases",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_databases', raise_exception=True))
    def get(self, request, version):
        """Get the alive status for all databases"""
        if version == "v1":
            # v1 code as of R3.0.0
            all_databases = database_manager.get_all_databases()
            res_data = {}

            for key in all_databases.keys():
                res_data[key] = all_databases[key][1].is_alive
        elif version == 'v2':
            # v2 code as of R3.1.0
            all_databases = database_manager.get_all_databases()
            res_data = {}

            for key in all_databases.keys():
                res_data[key] = {"is_alive": all_databases[key][1].is_alive, "details": database_manager.mongo_server_get_status(key)}
        else:
            # v3 code as of R5.1.0
            all_databases = database_manager.get_all_databases()
            res_data = {}

            for key in list(all_databases.keys()):
                if key in all_databases:
                    db_version = database_manager.mongo_server_get_version(key)
                    res_data[key] = {"is_alive": all_databases[key][1].is_alive, "details": database_manager.mongo_server_get_status(key), "version": db_version}

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============ One Database Status View ============
# ==================================================
class OneConnectionStatus(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_unversioned_schema('MANAGE-DATABASE')

    @extend_schema(
        operation_id="get_database_status",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database status'],
        summary="Get the alive status for the given database",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_databases', raise_exception=True))
    def get(self, request, database_id, version):
        """Get the alive status for the given database"""
        if version == "v1":
            # v1 code as of R3.0.0
            try:
                res_data = {"is_alive": database_manager.mongo_server_is_active(database_id)}
            except KeyError:
                res_data = {"is_alive": False}
        elif version == 'v2':
            # v2 code as of R3.1.0
            try:
                res_data = {"is_alive": database_manager.mongo_server_is_active(database_id),
                            "details": database_manager.mongo_server_get_status(database_id)}
            except KeyError:
                res_data = {"is_alive": False, "status": "Not found"}
        else:
            # v3 code as of R5.1.0
            try:
                db_version = database_manager.mongo_server_get_version(database_id)
                res_data = {"is_alive": database_manager.mongo_server_is_active(database_id), "details": database_manager.mongo_server_get_status(database_id), "version": db_version}
            except KeyError:
                res_data = {"is_alive": False, "status": "Not found"}

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# =============== Database Stats View ==============
# ==================================================
class Stats(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_unversioned_schema('MANAGE-DATABASE')

    @extend_schema(
        operation_id="check_database_stats",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database stats'],
        summary="Get the stats for the active database",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_databases', raise_exception=True))
    def get(self, request, database_id, version):
        """Get the stats for the active database"""
        try:
            res_data = database_manager.get_database_stats(database_id)
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        except KeyError as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response = PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

        return response

    @extend_schema(
        operation_id="delete_database_stats",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            405: ResponseExample(405),
            500: ResponseExample(500),
        },
        tags=['database', 'delete']
    )
    @method_decorator(permission_required('can_delete_global_config_databases', raise_exception=True))
    def delete(self, request, database_id, version):
        """Delete the given database statistics"""
        try:
            database_manager.clear_database_stats(database_id)
        except KeyError as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            return PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

# ==================================================
# ============= Manage Databases View ==============
# ==================================================
class Manage(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_unversioned_schema('MANAGE-DATABASE')

    @extend_schema(
        operation_id="get_databases",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database'],
        summary="Get the connection parameters for all active databases",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_databases', raise_exception=True))
    def get(self, request, version):
        """Get the connection parameters for all active databases"""
        res_data = database_manager.get_databases_json()

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============== Select Database View ==============
# ==================================================
class Select(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_unversioned_schema('MANAGE-DATABASE')

    @extend_schema(
        operation_id="get_selected_database",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database selection'],
        summary="Get the users active database selection",
        description=" "
    )
    def get(self, request, version):
        """Get the users active database selection"""
        try:
            database_id = database_manager.get_users_selected_database(request.user.email)
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=database_id)
        except KeyError as err:
            traceback.print_exc(file=sys.stdout)
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response = PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR, details="User does not have a selected database ID or the database ID does not exist")

        return response

    @extend_schema(
        operation_id="select_database",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database selection'],
        summary="Change the users active database selection",
        description=" "
    )
    def put(self, request, version):
        """Change the users active database selection"""
        db_value = get_nested_value(request.data, ['data'])
        if db_value is None or not isinstance(db_value, str):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details="Request body must be of format '{ data: <Database ID> }'")
        else:
            update_result = database_manager.set_users_selected_database(user_email=request.user.email,
                                                                         database_id=db_value)
            old_db_id = request.session.get("database", "")
            request.session.update({"database": db_value})

            # User was not found. Does not include message as to reason for error to avoid giving information on existing users
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=db_value, old_data=old_db_id)

        return response


# ==================================================
# =========== SessionOnly Database View ============
# ==================================================
class SessionOnly(LoginRequiredMixin, APIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_unversioned_schema('MANAGE-DATABASE')

    @extend_schema(
        operation_id="get_selected_database_by_session",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database session'],
        summary="Get the active database selection for the session",
        description=" "
    )
    def get(self, request, version):
        """
        Get the active database selection for the session.
        Used so 1 user can have multiple session with different databases.
        The client should be able to get the db from the session or cookie but this is left in to avoid versioning.
        """
        try:
            database_id = request.session.get("database", "")
            if database_id == "":
                database_id = database_manager.get_users_selected_database(request.user.email)
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=database_id)
        except KeyError as err:
            traceback.print_exc(file=sys.stdout)
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response = PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR, details="User does not have a selected database ID or the database ID does not exist")

        return response

    @extend_schema(
        operation_id="select_database_for_session",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database session'],
        summary="Change the active database selection for the session",
        description=" "
    )
    def put(self, request, version):
        """Change the active database selection for the session"""
        database_key = get_nested_value(request.data, ['data'])
        if database_key is None or not isinstance(database_key, str):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details="Request body must be of format '{ data: <Database ID> }'")
        else:
            old_database_id = request.session.get("database", "")
            request.session.update({"database": database_key})
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=database_key, old_data=old_database_id)

        return response


# ==================================================
# ==== Manage One Database Schema Definition =======
# ==================================================
MANAGE_DATABASE_SCHEMA = {
    "properties": {
        "auth_db": {
            "type": "string",
            "maxlength": 63
        },
        "auth_enable": {
            "type": "boolean"
        },
        "ca_cert_path": {
            "type": "string"
        },
        "db_uri": {
            "type": "string"
        },
        "dns_srv": {
            "type": "boolean"
        },
        "host": {
            "type": "string",
            "anyOf": [
                {"format": "hostname"},
                {"format": "ipv4"},
                {"format": "ipv6"},
                {"const": ""}
            ],
        },
        "name": {
            "type": "string",
            "pattern": "^([^\/\\. \';$*<>:|?]+)$",
            "maxlength": 63
        },
        "password": {
            "type": "string"
        },
        # TCP Port Number (0 - 65535)
        "port": {
            "type": "string",
            "pattern": "^([0-9]|[1-9][0-9]{1,3}|[1-5][0-9]{4}|6[0-4][0-9]{3}|65[0-4][0-9]{2}|655[0-2][0-9]|6553[0-5])$"
        },
        "replica_set_enable": {
            "type": "boolean"
        },
        "replica_set_hosts": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "replica_set_name": {
            "type": "string"
        },
        "tls_enable": {
            "type": "boolean"
        },
        "username": {
            "type": "string"
        }
    }
}


# ==================================================
# ============ Manage One Database View ============
# ==================================================
class ManageOne(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_unversioned_schema('MANAGE-DATABASE')

    @extend_schema(
        operation_id="get_one_database",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database'],
        summary="Get the connection parameters for the specified database",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_databases', raise_exception=True))
    def get(self, request, database_id, version):
        """Get the connection parameters for the specified database"""
        try:
            res_data = database_manager.get_databases_json()[database_id]
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        except KeyError as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND)

        return response

    @extend_schema(
        operation_id="put_one_database",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database'],
        summary="Update the connection parameters for the specified database",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_global_config_databases', 'can_create_global_config_databases'],
                                   raise_exception=True))
    @validate_data(collection="DATABASES", resource_id_param=None, schema=MANAGE_DATABASE_SCHEMA)
    def put(self, request, data, database_id, version):
        """Update the connection parameters for the specified database"""
        try:
            if version == "v1":
                old_data = None
                databases_list = database_manager.get_all_databases()
                if database_id in databases_list.keys():
                    old_data = database_manager.get_databases_json()[database_id]
                    old_data["password"] = "*****"

                edit_result = database_manager.edit_database(database_id, data)

                no_password_new_data = copy.deepcopy(data)
                no_password_new_data["password"] = "*****"

                if edit_result == "Created":
                    response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=no_password_new_data,
                                                     old_data=old_data)
                else:
                    response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=no_password_new_data,
                                                     old_data=old_data)
            else:
                old_data = None
                databases_list = database_manager.get_all_databases()
                if database_id in databases_list.keys():
                    old_data = database_manager.get_databases_json()[database_id]
                    old_data["password"] = "*****"

                # If using keyring, set the mongo password with the keyring and do not store it
                if data["auth_enable"] and data["password"] != "":
                    try:
                        user_db_password_opts = database_manager.get_user_database_password_opts()
                        if "type" in user_db_password_opts and "keyring_path" in user_db_password_opts and "keyring_key_path" in user_db_password_opts and \
                                user_db_password_opts["type"] == "keyring":
                            set_keyring(user_db_password_opts["keyring_path"],
                                        user_db_password_opts["keyring_key_path"],
                                        data["password"], False, usernames_to_add=[f"databases.{database_id}.password"])
                            data["password"] = ""
                    except ValueError as e:
                        messages.error(self.request, f"{type(e)}")
                        messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                        return PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR, details=str(e))
                    except Exception as err:
                        messages.error(self.request, f"{type(err)}")
                        messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                        return PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

                edit_result = database_manager.edit_database(database_id, data)

                no_password_new_data = copy.deepcopy(data)
                no_password_new_data["password"] = "*****"

                if edit_result == "Created":
                    response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=no_password_new_data,
                                                     old_data=old_data)
                else:
                    response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=no_password_new_data,
                                                     old_data=old_data)

        except (ConfigurationError, KeyError) as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details=str(e))

        return response

    @extend_schema(
        operation_id="post_one_database",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['database'],
        summary="Create a new connection to the given database",
        description=" "
    )
    @method_decorator(permission_required('can_create_global_config_databases', raise_exception=True))
    @validate_data(collection="DATABASES", resource_id_param=None, schema=MANAGE_DATABASE_SCHEMA)
    def post(self, request, data, database_id, version):
        """Create a new connection to the given database"""
        try:
            if version == "v1":
                database_manager.add_database(database_id, data)
                no_password_new_data = copy.deepcopy(data)
                no_password_new_data["password"] = "*****"
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=no_password_new_data)

            else:  #v2 adds support for keyring password option
                # If using keyring, set the mongo password with the keyring and do not store it
                if data["auth_enable"] and data["password"] != "":
                    try:
                        user_db_password_opts = database_manager.get_user_database_password_opts()
                        if "type" in user_db_password_opts and "keyring_path" in user_db_password_opts and "keyring_key_path" in user_db_password_opts and \
                                user_db_password_opts["type"] == "keyring":
                            set_keyring(user_db_password_opts["keyring_path"],
                                        user_db_password_opts["keyring_key_path"],
                                        data["password"], False, usernames_to_add=[f"databases.{database_id}.password"])
                            data["password"] = ""
                    except ValueError as e:
                        messages.error(self.request, f"{type(e)}")
                        messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                        return PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR, details=str(e))
                    except Exception as e:
                        messages.error(self.request, f"{type(e)}")
                        messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                        return PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

                database_manager.add_database(database_id, data)
                no_password_new_data = copy.deepcopy(data)
                no_password_new_data["password"] = "*****"
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=no_password_new_data)
        except DuplicateKeyError as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details="All database IDs must be unique.")
        except (ConfigurationError, KeyError) as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details=str(e))

        return response

    @extend_schema(
        operation_id="delete_one_database",
        responses=None,
        tags=['database'],
        summary="Delete the given database connection",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_databases', raise_exception=True))
    def delete(self, request, database_id, version):
        """Delete the given database connection"""
        if database_id == 'Default':
            return PonManagerApiResponse(status=status.HTTP_405_METHOD_NOT_ALLOWED)
        else:
            user_db_password_opts = database_manager.get_user_database_password_opts()
            if "type" in user_db_password_opts and "keyring_path" in user_db_password_opts and "keyring_key_path" in user_db_password_opts and \
                    user_db_password_opts["type"] == "keyring":
                try:
                    delete_keyring_username(user_db_password_opts["keyring_path"],
                                            user_db_password_opts["keyring_key_path"],
                                            f'databases.{database_id}.password')
                except:
                    pass

            database_manager.remove_database(database_id)

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
