"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import datetime
import pymongo

MAX_SAMPLES = 16

class DatabaseStatsWriter:
    """
    For writing database stats to the user database
    """

    def __init__(self):
        self._user_database = ""
        self.recovery_failure_count = {}

    def set_user_database(self, user_database):
        self._user_database = user_database

    def write_statistics(self, database_id, connection_stats=None, server_stats=None, command_stats=None):
        try:
            time = str(datetime.datetime.now())
            base_document = {
                "_id": database_id,
                "Connections": {
                    "Count Average Samples": [],
                    "Count Current Samples": [],
                    "Count Max": 0,
                },
                "Commands": {},
                "Servers": [],
                "Time": time
            }

            servers_sorted = self.list_to_dict(server_stats, "Host")
            servers_sorted = dict(sorted(servers_sorted.items()))
            servers_sorted = self.dict_to_list(servers_sorted, "Host")

            update_document = { "$set": {}, "$min": {}, "$max": {}, "$inc": {}, "$push": {} }
            update_document['$set']['Time'] = time

            # Add connection stats to update doc
            if connection_stats:
                average_connections = connection_stats['Average Count']
                current_connections = connection_stats['Count']
                update_document['$push']['Connections.Count Average Samples'] = { "$each": [average_connections], "$slice" : -MAX_SAMPLES}
                update_document["$push"]["Connections.Count Current Samples"] = { "$each": [current_connections], "$slice" : -MAX_SAMPLES}
                update_document['$max']['Connections.Count Max'] = max(average_connections, current_connections)

            # Add server stats to update doc
            if server_stats:
                for index, server in enumerate(servers_sorted):
                    update_document['$set'][f'Servers.{index}.Host'] = servers_sorted[index]['Host']
                    update_document['$set'][f'Servers.{index}.Status'] = servers_sorted[index]['Status']
                    update_document["$inc"][f"Servers.{index}.Failure Count"] = servers_sorted[index]['Failure Count']
                    if 'Latency' in server and 'Type' in server:
                        update_document["$push"][f"Servers.{index}.Latency Samples"] = { "$each": [servers_sorted[index]["Latency"]], "$slice" : -MAX_SAMPLES}
                        update_document['$set'][f'Servers.{index}.Type'] = servers_sorted[index]['Type']
                    else:
                        update_document["$push"][f"Servers.{index}.Latency Samples"] = { "$each": [0], "$slice" : -MAX_SAMPLES}
                        update_document['$set'][f'Servers.{index}.Type'] = "Unknown"

            # Add command stats to update doc
            if command_stats:
                for command_name in command_stats:
                    average_latency = command_stats[command_name]['Average Latency']
                    update_document["$push"][f"Commands.{command_name}.Average Latency Samples"] = { "$each": [average_latency], "$slice" : -MAX_SAMPLES}
                    update_document["$min"][f"Commands.{command_name}.Min Latency"] = command_stats[command_name]['Min Latency']
                    update_document["$max"][f"Commands.{command_name}.Max Latency"] = command_stats[command_name]['Max Latency']
                    update_document["$inc"][f"Commands.{command_name}.Success Count"] = command_stats[command_name]['Success Count']
                    update_document["$inc"][f"Commands.{command_name}.Failure Count"] = command_stats[command_name]['Failure Count']

            # Write stats to DATABASE-STATE
            collection = self._user_database.get_collection("DATABASE-STATE").with_options(
                write_concern=pymongo.write_concern.WriteConcern(w=0))
            result = collection.find_one(filter={"_id": database_id})
            if result is None:
                collection.insert_one(base_document)
            collection.update_one(filter={"_id": database_id}, update=update_document, upsert=True)

            if database_id in self.recovery_failure_count and self.recovery_failure_count[database_id] > 0:
                previous_count = self.recovery_failure_count[database_id]
                self.recovery_failure_count[database_id] = previous_count - 1

        except pymongo.errors.OperationFailure as e:
            if database_id in self.recovery_failure_count and self.recovery_failure_count[database_id] == 0:
                print(f"MongoDB error: {e}")
            else:
                self.recovery_failure_count[database_id] = 3

        except Exception as e:
            print(f"MongoDB error: {e}")

    def list_to_dict(self, _list, key):
        """ Convert a list of dictionaries to a dictionary indexed by the specified list key """
        return {x[key]: x for x in _list}

    def dict_to_list(self, _dict, key):
        """ Convert a dictionary to a list of dictionaries """
        return [(lambda d: d.update({key: k}) or d)(v) for (k, v) in _dict.items()]

# Create the database stats writer object
database_stats_writer = DatabaseStatsWriter()