"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import collections
from pymongo import monitoring


class CustomConnectionPoolLogger(monitoring.ConnectionPoolListener):
    """A simple listener that logs server connection pool events.
    Listens for :class:`~pymongo.monitoring.PoolCreatedEvent`,
    :class:`~pymongo.monitoring.PoolClearedEvent`,
    :class:`~pymongo.monitoring.PoolClosedEvent`,
    :~pymongo.monitoring.class:`ConnectionCreatedEvent`,
    :class:`~pymongo.monitoring.ConnectionReadyEvent`,
    :class:`~pymongo.monitoring.ConnectionClosedEvent`,
    :class:`~pymongo.monitoring.ConnectionCheckOutStartedEvent`,
    :class:`~pymongo.monitoring.ConnectionCheckOutFailedEvent`,
    :class:`~pymongo.monitoring.ConnectionCheckedOutEvent`,
    and :class:`~pymongo.monitoring.ConnectionCheckedInEvent`
    events and logs them at the `INFO` severity level using :mod:`logging`.
    """

    def __init__(self):
        self.count = 0
        self.max_count = 0
        self.average_count = 0
        self.average_count_sample_set = collections.deque(14400 * [0],
                                                          14400)  # Entry created every ~1 second. Set holds average for 4 hour period

    def calculate_average_connection_count(self):
        """ Calculates average connection count  """
        self.average_count_sample_set.pop()
        self.average_count_sample_set.appendleft(self.count)
        non_zero_sample_set_count = len(self.average_count_sample_set) - self.average_count_sample_set.count(0)
        if non_zero_sample_set_count > 0:
            self.average_count = round(int((sum(self.average_count_sample_set) / non_zero_sample_set_count)), 0)

    def connection_created(self, event: monitoring.ConnectionCreatedEvent) -> None:
        """Listener for created DB connections. """
        self.count += 1
        if self.count > self.max_count:
            self.max_count = self.count

    def connection_closed(self, event: monitoring.ConnectionClosedEvent) -> None:
        """Listener for closed DB connections. """
        self.count -= 1

    def get_statistics(self):
        """ Gets all connection pool statistics
        Returns:
            dict: All connection pool statistics in DB format
        """
        self.calculate_average_connection_count()
        return {
            'Count': self.count,
            'Max Count': self.max_count,
            'Average Count': self.average_count,
        }

    def connection_ready(self, event: monitoring.ConnectionReadyEvent) -> None:
        """ Not currently used. """
        pass

    def connection_check_out_started(self, event: monitoring.ConnectionCheckOutStartedEvent) -> None:
        """ Not currently used. """
        pass

    def connection_check_out_failed(self, event: monitoring.ConnectionCheckOutFailedEvent) -> None:
        """ Not currently used. """
        pass

    def connection_checked_out(self, event: monitoring.ConnectionCheckedOutEvent) -> None:
        """ Not currently used. """
        pass

    def connection_checked_in(self, event: monitoring.ConnectionCheckedInEvent) -> None:
        """ Not currently used. """
        pass

    def pool_created(self, event: monitoring.PoolCreatedEvent) -> None:
        """ Not currently used. """
        pass

    def pool_ready(self, event):
        """ Not currently used. """
        pass

    def pool_cleared(self, event: monitoring.PoolClearedEvent) -> None:
        """ Not currently used. """
        pass

    def pool_closed(self, event: monitoring.PoolClosedEvent) -> None:
        """ Not currently used. """
        pass