"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
======================================================

Django project configurations
"""

import json
import os

import mongoengine
from utils.keyring_helper import get_mongo_password

from utils.mongo_uri import get_mongo_uri

from utils.database_init import DatabaseInit

APPEND_SLASH = False
ALLOWED_HOSTS = ['*']
USE_X_FORWARDED_HOST = True

IN_PRODUCTION = True
BRAND = 'tibit'
PONMGR_OPT_PATH = f'/opt/{BRAND}/ponmgr'
PONMGR_ETC_PATH = f'/etc/{BRAND}/ponmgr'

# Check if in development mode
try:
    with open(f"{PONMGR_OPT_PATH}/api/api/settings.py") as json_file:
        IN_PRODUCTION = True
except FileNotFoundError:
    IN_PRODUCTION = False

DEBUG = not IN_PRODUCTION

# Set CSRF_TRUSTED_ORIGINS for dev environment
if not IN_PRODUCTION:
    CSRF_TRUSTED_ORIGINS = ['http://localhost:4200', "https://localhost:4200"]

# Grab secret key from user_database.json
try:
    with open(f"{PONMGR_OPT_PATH}/api/user_database.json" if IN_PRODUCTION else "user_database.json") as json_file:
        user_db_data = json.load(json_file)
        if user_db_data and "django_key" in user_db_data:
            SECRET_KEY = user_db_data["django_key"]
except FileNotFoundError:
    SECRET_KEY = ''

INSTALLED_APPS = [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.staticfiles',
    'django.contrib.messages',
    'rest_framework',
    'corsheaders',
    'rest_framework_swagger',
    'drf_spectacular',
    'drf_spectacular_sidecar',
    'databases',
    'users',
    'controllers',
    'cascading',
    'switches',
    'olts',
    'onus',
    'dpg',
    'cpes',
    'files',
    'slas',
    'service_configs',
    'downstream_maps',
    'pon_automation',
    'ponmgr',
    'docs',
    'DBdjango.apps.DiscoverConfig',
    'tasks',
]

MIDDLEWARE = [
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'corsheaders.middleware.CorsMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
    'utils.middleware.set_headers',
    'utils.middleware.handle_server_errors',
    'utils.middleware.update_session',
    'utils.middleware.get_user_db_from_cookie',
    'utils.middleware.log_request',
]

# Retrieving logging configuration json file
try:
    if IN_PRODUCTION:
        with open("/var/www/html/api/logging_configuration.json") as json_file:
            LOGGING_OPTIONS = json.load(json_file)
    else:
        with open("logging_configuration.json") as json_file:
            LOGGING_OPTIONS = json.load(json_file)
except:
    # Set default values in case of file failure
    LOGGING_OPTIONS = {
        "Directory": f"/var/log/{BRAND}",
        "FileCount": 3,
        "FileSize": 1024000
    }

if LOGGING_OPTIONS['Directory'].endswith("/"):
    log_file_path = LOGGING_OPTIONS['Directory'] + 'ponMgr.log'
else:
    log_file_path = LOGGING_OPTIONS['Directory'] + '/ponMgr.log'

# Settings here are only to disable default django logging
LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'filters': {
        'require_debug_true': {
            '()': 'django.utils.log.RequireDebugTrue',
        }
    },
    'handlers': {
        # Used to silence unwanted logs
        'null': {
            'class': 'logging.NullHandler',
        },
    },
    'loggers': {
        'django': {
            'propagate': False
        },
        # Silence default error logs to use custom logging
        'django.request': {
            'handlers': ['null'],
            'propagate': False
        },
        # Silence logs when running via runserver
        'django.server': {
            'filters': ['require_debug_true'],
            'propagate': False
        }
    }
}

AUTHENTICATION_BACKENDS = [
    # Mongoengine Backend
    'backends.mongoengine.MongoBackend',
    # Djongo Backend
    # 'django.contrib.auth.backends.ModelBackend',
    'backends.radius.RADIUSBackend'
]

REST_FRAMEWORK = {
    'DEFAULT_AUTHENTICATION_CLASSES': ['rest_framework.authentication.SessionAuthentication'],
    'DEFAULT_SCHEMA_CLASS': 'drf_spectacular.openapi.AutoSchema',
    'DEFAULT_VERSIONING_CLASS': 'rest_framework.versioning.URLPathVersioning',
    'ALLOWED_VERSIONS': 'v1|v2|v3',
    'DEFAULT_VERSION': 'v3'
}

# Used for generating OpenAPI yaml Swagger and ReDoc UIs

# Static folder for swagger frontend files
BASE_DIR = os.path.dirname(os.path.realpath(__file__))
STATIC_ROOT = os.path.join(BASE_DIR, 'static')
STATIC_URL = '/static/'

# https://drf-spectacular.readthedocs.io/en/latest/settings.html#example-swaggerui-settings
SPECTACULAR_SETTINGS = {
    'SWAGGER_UI_DIST': 'SIDECAR',  # shorthand to use the sidecar instead of cdn
    'SWAGGER_UI_FAVICON_HREF': 'SIDECAR',
    'TITLE': 'PON Manager REST API',
    'DESCRIPTION': 'REST API for interfacing with PON Controller databases',
    'VERSION': 'R5.2.0',
    'COMPONENT_SPLIT_REQUEST': True,
    'SCHEMA_PATH_PREFIX_INSERT': '/api',
    'SWAGGER_UI_SETTINGS': {  # Only applies to Swagger UI, not ReDoc
        'filter': True,  # Adds filter input
        'docExpansion': 'none',  # Collapses the tag entries by default (full or none)
        'deepLinking': True,  # Retains users spot on the page, allows link sharing to a specific tag/operation
        'tagsSorter': 'alpha',  # Sorts the tags alphabetically
        'tryItOutEnabled': True,  # Allows for API calls to be made from Swagger UI
        'defaultModelsExpandDepth': '-1',  # Removes the "Schemas" section
        'persistAuthorization': False,  # Remain logged in on reload
        "displayOperationId": False,  # List the operation ID next to endpoint name
    },
    'PREPROCESSING_HOOKS': [
        'utils.drf_hooks.exclude_private_endpoints',
        'drf_spectacular.hooks.preprocess_exclude_path_format'
    ],
}

# Swagger settings development settings
if not IN_PRODUCTION:
    SPECTACULAR_SETTINGS["SWAGGER_UI_DIST"] = "https://cdn.jsdelivr.net/npm/swagger-ui-dist@latest"
    SPECTACULAR_SETTINGS["SWAGGER_UI_FAVICON_HREF"] = "https://cdn.jsdelivr.net/npm/swagger-ui-dist@latest/favicon-32x32.png"
# If in development mode, overwrite settings above.
# NOTE: None of this section needs to be modified if enabling non-secure HTTP


# Swagger api documentation for version 2.2
# https://django-rest-swagger.readthedocs.io/en/stable-0.3.x/settings.html
SWAGGER_SETTINGS = {
    'api_version': '0.1',
    'JSON_EDITOR': True,
    'enabled_methods': [
        'get',
        'post',
        'put',
        'patch',
        'delete'
    ],
}

ROOT_URLCONF = 'api.urls'

TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [
            '/var/www/html',
            '/var/www/ponmgr.localhost/public_html'
        ],
        'APP_DIRS': True,
        'OPTIONS': {
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

WSGI_APPLICATION = 'api.wsgi.application'

# Retrieving database json files
if IN_PRODUCTION:
    database_options = DatabaseInit(user_database="/var/www/html/api/user_database.json")

    with open("/var/www/html/api/recovery_email_configuration.json") as json_file:
        recovery_email_options = json.load(json_file)
else:
    database_options = DatabaseInit(user_database="user_database.json")
    with open("recovery_email_configuration.json") as json_file:
        recovery_email_options = json.load(json_file)

user_database_info = database_options.user_database_fields

# MongoEngine Settings
SESSION_ENGINE = 'backends.mongo_sessions'
SESSION_SERIALIZER = 'backends.mongo_sessions.BSONSerializer'
mongo_uri = get_mongo_uri(user_database_info)
if "compression" in user_database_info and user_database_info["compression"]:
    mongoengine.connect(host=mongo_uri, compressors='snappy')
else:
    mongoengine.connect(host=mongo_uri)

# Djongo Settings
# DATABASES = {'default':
#                  {'ENGINE': ''}
#              }
#
# DATABASES = {
#     'default': {
#         'ENGINE': 'backends.radius.RADIUSBackend',
#         'NAME': user_database_info["name"]
#     }
# }
#
# DATABASES = {
#     'default': {
#         'ENGINE': 'djongo',
#         'NAME': user_database_info["name"]
#     }
# }
#
# replica_set_enabled = user_database_info["replica_set_enable"]
# if replica_set_enabled:
#     DATABASES['default']['HOST'] = ",".join(user_database_info["replica_set_hosts"])
#     DATABASES['default']['REPLICASET'] = user_database_info["replica_set_name"]
# else:
#     DATABASES['default']['HOST'] = user_database_info["host"]
#     DATABASES['default']['PORT'] = int(user_database_info["port"])
#
# auth_enabled = user_database_info["auth_enable"]
# if auth_enabled:
#     DATABASES['default']['USER'] = user_database_info["username"]
#     DATABASES['default']['PASSWORD'] = get_mongo_password(user_database_info)
#     DATABASES['default']['AUTH_SOURCE'] = user_database_info["auth_db"]
#
# tls_enable = user_database_info["tls_enable"]
# if tls_enable:
#     DATABASES['default']['SSL'] = True
#     DATABASES['default']['SSL_CA_CERTS'] = user_database_info["ca_cert_path"]
# Djongo Settings End

AUTH_PASSWORD_VALIDATORS = [
    {
        'NAME': 'django.contrib.auth.password_validation.UserAttributeSimilarityValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.MinimumLengthValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.CommonPasswordValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.NumericPasswordValidator',
    },
]

# Internationalization
# https://docs.djangoproject.com/en/1.9/topics/i18n/
LANGUAGE_CODE = 'en-us'
TIME_ZONE = 'UTC'
USE_I18N = True
USE_L10N = True
USE_TZ = True

# SMTP server config for recovery email
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = recovery_email_options['host']
EMAIL_PORT = recovery_email_options['port']
EMAIL_USE_TLS = recovery_email_options['use_tls'].lower() == "true"
EMAIL_HOST_USER = recovery_email_options['user']
EMAIL_HOST_PASSWORD = get_mongo_password({
    'password': recovery_email_options['password'],
    'auth_enable': user_database_info['auth_enable'],
    'password_opts': user_database_info['password_opts']
}, 'recovery_email.password')

# Session Settings
SESSION_SAVE_EVERY_REQUEST = True
SESSION_COOKIE_AGE = 1200  # Default 20 minutes
SESSION_COOKIE_SAMESITE = 'Strict'
SESSION_COOKIE_NAME = '__Host-sessionid'
SESSION_COOKIE_SECURE = True

# CSRF Settings
CSRF_COOKIE_SAMESITE = 'Strict'
CSRF_COOKIE_NAME = '__Host-csrftoken'
CSRF_COOKIE_SECURE = True

# Swagger api documentation for version 2.2
# https://django-rest-swagger.readthedocs.io/en/stable-0.3.x/settings.html
SWAGGER_SETTINGS = {
    'api_version': '0.1',
    'JSON_EDITOR': True,
    'enabled_methods': [
        'get',
        'post',
        'put',
        'patch',
        'delete'
    ],
}

MESSAGE_STORAGE = "django.contrib.messages.storage.session.SessionStorage"
