import copy
from typing import Union
from mongoengine import Document, fields, DynamicDocument
from backends.mongo_sessions import MongoSession
from django.utils import timezone
from django.contrib.auth.hashers import make_password
from bson.objectid import ObjectId

class User(Document):
    meta = {'collection': 'auth_user'}
    _id = fields.ObjectIdField()
    uid = fields.IntField(unique=True, db_field='id')
    email = fields.StringField(required=True, unique=True)
    password = fields.StringField(required=True)
    username = fields.StringField(required=True, unique=True)
    first_name = fields.StringField(max_length=50)
    last_name = fields.StringField(max_length=50)
    last_login = fields.DateTimeField(default=None, null=True)
    is_superuser = fields.BooleanField(default=False)
    is_staff = fields.BooleanField(default=False)
    is_active = fields.BooleanField(default=False)
    date_joined = fields.DateTimeField()
    active_database = fields.StringField(default='Default')
    database_type = fields.StringField()
    backend = fields.StringField(null=True)
    groups = []

    def __init__(self, *args, **values):
        super().__init__(*args, **values)
        self.groups = GroupManager(parent_instance=self)
        if self.backend and 'radius' not in self.backend:
            self.backend = 'backends.mongoengine.MongoBackend'

    def save(self, is_initial_user=False, **kwargs):
        # Generate the unique _id before save on new users. MongoEngine treats _id as the primary key when saving
        if not self._id:
            self._id = ObjectId()
        if self.database_type == 'radius':
            self.password = '' #Save will fail with null value

        super().save(**kwargs)

        if self.groups.has_modified:
            #Adding groups
            if len(self.groups.ids_to_add) > 0:
                new_group_relations = []
                try:
                    highest_user_group_relation_id = UserGroupRelation.objects().order_by('-ugid').first().ugid
                except AttributeError:
                    if is_initial_user:
                        highest_user_group_relation_id = 1

                for one_based_index, group_id in enumerate(self.groups.ids_to_add, start=1):
                    # Create user group relationship objects and append to a list
                    new_group_relations.append(UserGroupRelation(user_id=self.uid, group_id=group_id, ugid=highest_user_group_relation_id + one_based_index))

                # bulk save new user group relationships
                if len(new_group_relations) > 0:
                    UserGroupRelation.objects.insert(new_group_relations)

            #Removing groups
            if len(self.groups.ids_to_remove) > 0:
                UserGroupRelation.objects(group_id__in=self.groups.ids_to_remove, user_id=self.uid).delete()

            self.groups.reset_modification_flags()


    def delete(self, signal_kwargs=None, **write_concern):
        # Delete references to groups
        UserGroupRelation.objects(user_id=self.uid).delete()

        # Remove sessions that belong to the user
        session_ids_to_delete = []
        session_objects = MongoSession.objects()
        for session_object in session_objects:
            decoded_session_data = session_object.get_decoded()
            if '_auth_user_id' in decoded_session_data and decoded_session_data['_auth_user_id'] == self.uid:
                session_ids_to_delete.append(session_object._id)
        MongoSession.objects(_id__in=session_ids_to_delete).delete()

        User.objects(_id=self._id).delete() # super delete will remove all users because it uses query {pk: none}

    @classmethod #Called before init
    def create_user(cls, username, password, email=None, is_initial_user=False, database_type='local'):
        return cls._create_user(username, password, email, is_initial_user, database_type)

    @classmethod
    def _create_user(cls, username, password, email, is_initial_user, database_type):
        now = timezone.now()

        # Lowercase email for normalization
        if email is not None:
            email = email.lower()

        # Lowercase username for normalization
        if username is not None:
            username = username.lower()

        # Instantiate the new user
        if database_type == 'radius':
            user = cls(username=username, email=email, date_joined=now, database_type='radius', backend='backends.radius.RADIUSBackend')
        else:
            user = cls(username=username, email=email, date_joined=now)

        user.set_password(password)

        try:
            # ID is one higher than the highest
            user.uid = User.objects().order_by('-uid').first().uid + 1
        except AttributeError:
            if is_initial_user:
                user.uid = 1
                user.groups = GroupManager(parent_instance=user)
        user.save()
        return user

    def set_password(self, password):
        self.password = make_password(password)
        #TODO: should this save user?
        return self

    def is_authenticated(self):
        if self.id != None:
            return True
        else:
            return False

    def __str__(self):
        return self.username


class Group(Document):
    meta = {'collection': 'auth_group'}
    _id = fields.ObjectIdField()
    gid = fields.IntField(unique=True, db_field='id', required=True)
    name = fields.StringField(required=True, unique=True)
    user_session_expiry_age_timeout = fields.IntField(required=True, db_field='User Session Expiry Age Timeout')
    user_session_expiry_age_timeout_override = fields.BooleanField(required=True, db_field='User Session Expiry Age Timeout Override')
    user_set = None
    permissions = None

    def __init__(self, *args, **values):
        super().__init__(*args, **values)
        self.user_set = UserManager(parent_instance=self)
        self.permissions = PermissionManager(parent_instance=self)

    def save(self, **kwargs):
        # Generate the unique _id before save. MongoEngine treats _id as the primary key when saving
        if not self._id:
            self._id = ObjectId()

        if not self.gid:
            highest_group_id = Group.objects().order_by('-gid').first().gid
            self.gid = highest_group_id + 1

        if self.user_session_expiry_age_timeout is None:
            try:
                self.user_session_expiry_age_timeout = Settings.objects.get(_id='Default')['User Session Expiry Age Timeout']  # db.get_global_session_time()[1]['timeout']
            except Exception as Err:
                self.user_session_expiry_age_timeout = 20 #Default value

        if self.user_session_expiry_age_timeout_override is None:
            self.user_session_expiry_age_timeout_override = False

        super().save(**kwargs)

        if self.user_set.has_modified:
            if len(self.user_set.ids_to_add) > 0:
                new_group_relations = []
                highest_user_group_relation_id = UserGroupRelation.objects().order_by('-ugid').first().ugid

                for one_based_index, user_id in enumerate(self.user_set.ids_to_add, start=1):
                    # Create user group relationship objects and append to a list
                    new_group_relations.append(UserGroupRelation(user_id=user_id, group_id=self.gid,
                                                                 ugid=highest_user_group_relation_id + one_based_index))
                # bulk save new user group relationships
                if len(new_group_relations) > 0:
                    UserGroupRelation.objects.insert(new_group_relations)

            if len(self.user_set.ids_to_remove) > 0:
                UserGroupRelation.objects(user_id__in=self.user_set.ids_to_remove, group_id=self.gid).delete()

        if self.permissions.has_modified:
            if len(self.permissions.ids_to_add) > 0:
                new_group_permission_relations = []
                highest_group_permission_relation_id = GroupPermissionRelation.objects().order_by('-gpid').first().gpid

                for one_based_index, permission_id in enumerate(self.permissions.ids_to_add, start=1):
                    # Create user group relationship objects and append to a list
                    new_group_permission_relations.append(GroupPermissionRelation(permission_id=permission_id, group_id=self.gid,
                                                                 gpid=highest_group_permission_relation_id + one_based_index))
                # bulk save new user group relationships
                if len(new_group_permission_relations) > 0:
                    GroupPermissionRelation.objects.insert(new_group_permission_relations)

            if len(self.permissions.ids_to_remove) > 0:
                GroupPermissionRelation.objects(permission_id__in=self.permissions.ids_to_remove, group_id=self.gid).delete()

            self.permissions.reset_modification_flags()

    def delete(self, signal_kwargs=None, **write_concern):
        # Delete references to users
        UserGroupRelation.objects(group_id=self.gid).delete()

        # Delete references to permissions
        GroupPermissionRelation.objects(group_id=self.gid).delete()

        # Delete self
        Group.objects(_id=self._id).delete() # super delete will remove all users because it uses query {pk: none}


class UserGroupRelation(Document):
    meta = {'collection': 'auth_user_groups'}
    _id = fields.ObjectIdField()
    ugid = fields.IntField(unique=True, db_field='id', required=True)
    user_id = fields.IntField(required=True, unique_with='group_id')
    group_id = fields.IntField(required=True)


class GroupPermissionRelation(Document):
    meta = {'collection': 'auth_group_permissions'}
    _id = fields.ObjectIdField()
    gpid = fields.IntField(unique=True, db_field='id', required=True)
    permission_id = fields.IntField(required=True, unique_with='group_id')
    group_id = fields.IntField(required=True)


class Permission(Document):
    meta = {'collection': 'auth_permission'}
    _id = fields.ObjectIdField()
    permission_id = fields.IntField(unique=True, db_field='id')
    name = fields.StringField(required=True)
    content_type_id = fields.IntField(required=True)
    codename = fields.StringField(required=True)
    content_type = ''

    def __init__(self, *args, **values):
        super().__init__(*args, **values)
        self.content_type = ContentType.objects.get(ctid=self.content_type_id).ctid

    def __str__(self):
        return self.name


class ContentType(Document):
    meta = {'collection': 'django_content_type'}
    _id = fields.ObjectIdField()
    ctid = fields.IntField(unique=True, db_field='id')
    app_label = fields.StringField(required=True)
    model = fields.StringField(required=True)


class Settings(DynamicDocument): # Use only for read
    meta = {'collection': 'PONMGR-CFG'}
    _id = fields.StringField()


class RelationalManager:

    _data = []
    parent_reference: Group = None
    model_reference = None
    has_modified = False
    has_queried_for_data = False
    ids_to_add = []
    ids_to_remove = []
    relation_to_parent_class = None
    parent_primary_key = ''
    primary_key = ''
    parent_primary_key_in_relation = ''
    primary_key_in_relation = ''

    def __init__(self, parent_instance=None, data_list=None):
        """
        Used to read/write relational objects from a parent object.
        Ex: Adding groups to a user object.

        Args:
            **parent_instance (Object): Reference to 'self' for the relational manager to access
            **data_list (list <Objects>): If already retrieved, pass in the relational objects
        """
        self.has_modified = False
        self.ids_to_add = []
        self.ids_to_remove = []
        self.parent_reference = parent_instance
        self.model_reference = None
        self.relation_to_parent_class = None

        if data_list and len(data_list) != 0:
            self.has_queried_for_data = True
            self._data = copy.deepcopy(data_list)
        else:
            self.has_queried_for_data = False
            self._data = []

    def reset_modification_flags(self):
        self.has_modified = False
        self.ids_to_add = []
        self.ids_to_remove = []

    def clear(self):
        self._data = []

    def add(self, instance_to_add: Union[Permission,Group,User]):
        if not self.has_queried_for_data:
            self.query_for_data()

        existing_primary_keys = [getattr(obj_instance, self.primary_key) for obj_instance in self._data]

        if getattr(instance_to_add, self.primary_key) not in existing_primary_keys:
            self._data.append(instance_to_add)
            self.ids_to_add.append(getattr(instance_to_add, self.primary_key))
            self.has_modified = True

    def remove(self, instance: Union[Permission,Group,User]):
        if not self.has_queried_for_data:
            self.query_for_data()

        existing_primary_keys = [getattr(obj_instance, self.primary_key) for obj_instance in self._data]

        if getattr(instance, self.primary_key) in existing_primary_keys:
            index_to_remove = [index for (index, stored_obj) in enumerate(self._data) if
                               getattr(stored_obj, self.primary_key) == getattr(instance, self.primary_key)]
            if len(index_to_remove) > 0:
                del self._data[index_to_remove[0]]
            self.ids_to_remove.append(getattr(instance, self.primary_key))
            self.has_modified = True

    def query_for_data(self):
        relation_objects = self.relation_to_parent_class.objects(**{self.parent_primary_key_in_relation: getattr(self.parent_reference, self.parent_primary_key)})
        ids = [getattr(relation_instance, self.primary_key_in_relation) for relation_instance in relation_objects]
        self._data = list(self.model_reference.objects(**{f'{self.primary_key}__in': ids}))
        self.has_queried_for_data = True

    def names(self):
        if not self.has_queried_for_data:
            self.query_for_data()
        return [obj_instance.name for obj_instance in self._data]

    def all(self):
        if not self.has_queried_for_data:
            self.query_for_data()
        return copy.deepcopy(self._data)

    def count(self):
        if not self.has_queried_for_data:
            self.query_for_data()
        return len(self._data)

    def __str__(self):
        if not self.has_queried_for_data:
            self.query_for_data()
        return str(self._data)

class UserManager(RelationalManager):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.relation_to_parent_class = UserGroupRelation
        self.model_reference = User
        self.parent_primary_key = 'gid'
        self.primary_key = 'uid'
        self.parent_primary_key_in_relation = 'group_id'
        self.primary_key_in_relation = 'user_id'

    def names(self):
        if not self.has_queried_for_data:
            self.query_for_data()
        return [user_instance.email for user_instance in self._data]

class GroupManager(RelationalManager):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.relation_to_parent_class = UserGroupRelation
        self.model_reference = Group
        self.parent_primary_key = 'uid'
        self.primary_key = 'gid'
        self.parent_primary_key_in_relation = 'user_id'
        self.primary_key_in_relation = 'group_id'

class PermissionManager(RelationalManager):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.relation_to_parent_class = GroupPermissionRelation
        self.model_reference = Permission
        self.parent_primary_key = 'gid'
        self.primary_key = 'permission_id'
        self.parent_primary_key_in_relation = 'group_id'
        self.primary_key_in_relation = 'permission_id'