#!/bin/bash
# Copyright (c) 2025, Ciena Corporation

#
# Instructions for repackaging MCMS binary packages with user defined branding files.
# This procedure replaces files for a more customized look and feel.
#

USAGE="$(basename "$0") [-h] [-b <path>] [-o <path>] [-l] [-e] <in-pkg>

REQUIRED:
in-pkg:         Filepath to the .deb package to be rebranded.

OPTIONAL:
-h|--help:      Display this help message.
-b|--branding:  Path to the branding directory.
-o|--out-pkg:   The output .deb file name.
-x|--lock:      Move custom controls (logo lock) to /opt/{BRAND}/ponmgr/ponmgr/branding/
-l|--list:      List the rebranded package contents and find any file names, paths, or text mentioning 'tibit', 'mcms', or 'ciena'.
                (if no branding directory is given, --list will work on the in-pkg)
-e|--extract:   Extract the contents of the in-pkg. The extracted files will be placed in a new directory 'build/EXTRACT'

This process replaces files for a more customized look and feel.
If required, custom database seed files should be placed in the branding directory at /opt/{BRAND}/ponmgr/api/databaseSeedFiles/

Examples:
./rebrand.sh -b ./brand_files -o mcms-rebranded.deb -l ./mcms-package-original.deb
./rebrand.sh -e ./mcms-package.deb

To facilitate the process there must be a file 'rebrand.txt' at the root of the branding directory.
The file should contain instructions for removing and renaming files and directories as well as replacing brand-specific strings in files.

Instructions should follow the format below:
remove  <file>
rename  <file>  <new file>
replace <string to replace>   <new string>   <file>

Examples:
remove   opt/tibit/ponmgr/examples
rename   etc/tibit   etc/ciena
replace  /etc/tibit/netconf/   /etc/ciena/netconf/   DEBIAN/conffiles
"

# Exit handling
function error_check {
    if [ "$?" == "0" ]; then
        echo
        echo -e "\e[32m${SUCCESS_MSG}\e[0m"
        echo
    else
        echo
        echo -e "\e[31m${ERR_MSG}\e[0m"
        echo
    fi
}
trap error_check EXIT
set -e

function remove_file {
    TARGET=$1
    if [ -e "${TARGET}" ]; then
        rm -rf "${TARGET}"
        echo "Removed file ${TARGET}."
    else
        echo -e "\e[31m[ERR]\e[0m File not found: ${TARGET}"
    fi
}

function rename_file {
    SRC=$1
    DST=$2
    if [ -e "${SRC}" ]; then
        mv "${SRC}" "${DST}"
        echo "Renamed file ${SRC} -> ${DST}."
    else
        echo -e "\e[31m[ERR]\e[0m File not found: ${SRC}"
    fi
}

function replace_strings {
    FROM=$1
    TO=$2
    FILE=$3
    if [ -f "${FILE}" ]; then
        sed -i "s|${FROM}|${TO}|g" "${FILE}"
        echo "Replaced string '${FROM}' with '${TO}' in file ${FILE}."
    else
        echo -e "\e[31m[ERR]\e[0m File not found: ${FILE}"
    fi
}

function list_files {
    echo
    echo "Files in .deb package:"
    dpkg-deb -c "$1"

    # Extract the contents for listing
    extract_files "$1" "${BUILD_ROOT}/LIST"

    # Find tibit references in files
    echo
    echo "Finding files with remaining tibit/ciena/mcms references..."
    pushd "${BUILD_ROOT}/LIST" > /dev/null

    echo
    echo "References in file names/paths:"
    find . | grep -i 'ciena\|tibit\|mcms' || echo "NONE"

    echo
    echo "References in file contents:"
    find . -type f -print0 | xargs -0 grep -l -I -i 'ciena\|tibit\|mcms' || echo "NONE"

    popd > /dev/null
    rm -rf "${BUILD_ROOT}/LIST"
    echo "Done."
}

function extract_files {
    echo
    echo "Extracting contents from $(basename $1)..."

    # If the target directory already exists, remove it
    rm -rf "$2"
    # Extract the contents and Debian control files from the target .deb file
    dpkg-deb -x "$1" "$2"
    dpkg-deb -e "$1" "$2/DEBIAN"
    echo "Done."
}

BRAND_DIR=""
IN_PKG=""
OUT_PKG=""
LOCK_LOGOS=false
LIST_FILES=false
EXTRACT_FILES=false
SUCCESS_MSG="Operation completed successfully"
ERR_MSG="Operation failed"

# Parse command line arguments
while (( $# )); do
    case $1 in
        -h|--help)
            echo "$USAGE"
            SUCCESS_MSG=""
            exit 0
            ;;
        -b|--branding)
            BRAND_DIR=$2
            shift 2
            ;;
        -o|--out-pkg)
            OUT_PKG=$2
            shift 2
            ;;
        -x|--lock)
            LOCK_LOGOS=true
            shift
            ;;
        -l|--list)
            LIST_FILES=true
            shift
            ;;
        -e|--extract)
            EXTRACT_FILES=true
            shift
            ;;
        --) # end argument parsing
            shift
            break
            ;;
        -*|--*=) # unsupported flags
            echo "$USAGE"
            echo
            echo "ERROR: Unknown argument: $1" >&2
            exit 1
            ;;
        *) # store the required in-pkg argument
            IN_PKG=$1
            shift
            ;;
    esac
done

# Check that the 'dpkg-deb' utility exists
DPKGDEB=$(which dpkg-deb)
if [ -z "$DPKGDEB" ]; then
    echo
    echo "ERROR: Missing 'dpkg-deb' utility"
    exit 1
fi

# Check that the 'fakeroot' utility exists
FAKEROOT=$(which fakeroot)
if [ -z "$FAKEROOT" ]; then
    echo
    echo "ERROR: Missing 'fakeroot' utility"
    exit 1
fi

# Ensure in-pkg is valid
if [ -z "$IN_PKG" ]; then
    echo "$USAGE"
    echo
    echo "ERROR: Missing <in-pkg> argument"
    exit 1
elif [ ! -f "$IN_PKG" ]; then
    echo
    echo "ERROR: <in-pkg> file not found: $IN_PKG"
    exit 1
elif [ "${IN_PKG: -4}" != ".deb" ]; then
    echo
    echo "ERROR: <in-pkg> is not a .deb file: $IN_PKG"
    exit 1
fi

# Determine the project root path and enter build dir
PROJROOT="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
BUILD_ROOT=${PROJROOT}/build
mkdir -p "${BUILD_ROOT}"

# Handle special operations when NOT rebranding
if [ -z "$BRAND_DIR" ]; then

    if [ $EXTRACT_FILES == false ] && [ $LIST_FILES == false ]; then
        echo "$USAGE"
        echo
        echo "ERROR: Missing argument -b|--branding"
        exit 1
    fi

    # Extract the files from the in-pkg if requested
    if [ $EXTRACT_FILES == true ]; then
        extract_files "${IN_PKG}" "${BUILD_ROOT}/EXTRACT"
    fi

    # List .deb files from the in-pkg if requested
    if [ $LIST_FILES == true ]; then
        list_files "${IN_PKG}"
    fi

    exit 0
fi

# Update the exit messages (By this point, we know a rebrand is being attempted)
SUCCESS_MSG="Rebranding completed successfully"
ERR_MSG="Rebranding failed"

# Extract the files from the in-pkg if requested
# (why someone would need to do this while also rebranding a package? no clue, but its possible to try so here it is)
if [ $EXTRACT_FILES == true ]; then
    extract_files "${IN_PKG}" "${BUILD_ROOT}/EXTRACT"
fi

# Ensure the branding directory exists
if [ ! -d "$BRAND_DIR" ]; then
    echo
    echo "ERROR: Branding directory not found: $BRAND_DIR"
    exit 1
fi

# If no specified output file name, output will be same as input with "-rebranded" added
if [ -z "$OUT_PKG" ]; then
    OUT_PKG="${IN_PKG%.*}-rebranded.deb"
fi

OUT_PKG_NAME=$(basename $OUT_PKG .deb)
IN_PKG_ROOT=${BUILD_ROOT}/$(basename $IN_PKG .deb)
OUT_PKG_ROOT=${BUILD_ROOT}/${OUT_PKG_NAME}

# Extract the contents from the original package .deb file
extract_files "${IN_PKG}" "${IN_PKG_ROOT}"

# Rename the working directory to the rebranded package name
if [ "$IN_PKG_ROOT" != "$OUT_PKG_ROOT" ]; then
    rm -rf "${OUT_PKG_ROOT}"
    mv "${IN_PKG_ROOT}" "${OUT_PKG_ROOT}"
fi

# Move custom controls (logo lock) to designated file
if [ $LOCK_LOGOS == true ]; then
    CNTRL_FILE=opt/tibit/ponmgr/ponmgr/assets/branding/customControls/custom-controls.json
    echo
    echo "Locking logos. Copying ${OUT_PKG_NAME}/${CNTRL_FILE}"
    cp "${OUT_PKG_ROOT}/${CNTRL_FILE}" "${OUT_PKG_ROOT}/opt/tibit/ponmgr/ponmgr/assets/branding/"
    echo "Done."
fi

# Read the branding instructions file if it exists
BRANDING_FILE="$(realpath ${BRAND_DIR})/rebrand.txt"
if [ -f "$BRANDING_FILE" ]; then

    # Parse rebrand.txt and execute instructions
    echo
    echo "Executing rebrand.txt instructions..."
    pushd "${OUT_PKG_ROOT}" > /dev/null

    while IFS= read -r line || [[ -n "$line" ]]; do
        if [ -n "$line" ] && [ "${line:0:1}" != "#" ]; then
            eval "cmd=($line)"

            # Remove unwanted files and directories
            if [ "${cmd[0]}" == "remove" ]; then
                remove_file "${cmd[1]}"

            # Rename existing files and directories
            elif [ "${cmd[0]}" == "rename" ]; then
                rename_file "${cmd[1]}" "${cmd[2]}"

            # Replace brand names in files using sed
            elif [ "${cmd[0]}" == "replace" ]; then
                replace_strings "${cmd[1]}" "${cmd[2]}" "${cmd[3]}"

            fi
        fi
    done < "$BRANDING_FILE"
    popd > /dev/null
    echo "Done."
else
    echo
    echo "Rebranding file rebrand.txt could not be found, rebranding will use overlay files only."
fi

# Overlay the brand directory into the working directory
echo
echo "Overlaying branded files..."
( cd "${BRAND_DIR}" && tar --exclude="rebrand.txt" -cf - . ) | ( cd "${OUT_PKG_ROOT}"; tar xvf - )
echo "Done."

# Create the new Debian package with the rebranded files
echo
echo "Creating rebranded .deb package..."
rm -f "${BUILD_ROOT}/${OUT_PKG_NAME}.deb"
fakeroot dpkg-deb -Z xz -b "${OUT_PKG_ROOT}" "${BUILD_ROOT}/${OUT_PKG_NAME}.deb"
echo "Done."

# Report 'tibit'/'ciena'/'mcms' in filenames and file contents if requested
if [ $LIST_FILES == true ]; then
    list_files "${BUILD_ROOT}/${OUT_PKG_NAME}.deb"
    exit 0
fi
