#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" Utility functions used by the MCMS REST API example scripts.

The API Utilities module contains common utility functions that
are used accross the REST API example scripts.

"""

import json
import re
from typing import Any, List, Optional


def json_dumps(dict_: dict, indent=4, sort_keys=True):
    """ Convert a JSON dictionary to a formatted JSON string.

    Args:
        dict_: Dictionary containing the JSON data to be converted to a string.
        indent: Set the indent level when converting to a string.
        sort_keys: Sort the dictionary entries by key during the conversion.

    Returns:
        string: The formatted string representing the JSON data.
    """
    return json.dumps(dict_, indent=indent, sort_keys=sort_keys)

def natural_sort_key(key):
    """ Key for sorting a list in the way that humans expect. """
    convert = lambda text: int(text) if text.isdigit() else text
    return [convert(c) for c in re.split('([0-9]+)', key)]

def dict_read(dict_: dict, path: str, sep: str = '.', default: Optional[Any] = None) -> Any:
    """ Read the value of an attribute in a nested dictionary

    Args:
        dict_: Dictionary to be read from.
        path: The path to the field to be retrieved fromthe nested dictionary.
        sep: Separator used to split the fields in the path.
        default: Default value to use when the field identified by path does not exist.

    Returns:
        val: The value read from the dictionary

    """
    val = dict_
    for elem in path.split(sep):
        try:
            val = val[elem]
        except KeyError:
            val = default
            break

    return val

def list_to_dict(list_: List[dict], key: str) -> dict:
    """ Convert a list of dictionaries to a dictionary

    Args:
        list_: The list to be converted to a dictionary.
        key: The field in the dictionary to use as the dictionary key.

    Returns:
        dict_: The dictionary representation of the list.
    """
    return {x[key]: x for x in list_}

def dict_to_list(dict_: dict, key: str) -> List[dict]:
    """ Convert a dictionary to a list of dictionaries

    Args:
        list_: The dictionary to be converted to a list.
        key: The field in the dictionary to use as the dictionary key.

    Returns:
        list_: The list representation of the dictionary.
    """
    return [(lambda d: d.update({key : k}) or d)(v) for (k, v) in dict_.items()]

def slot12_8ns_to_us(slot12_8ns: float) -> float:
    """ Convert a slots (12.8 ns) to microseconds

    Args:
        slot12_8ns: Value in slot units, 12.8 ns

    Returns:
        list_: Slot value converted to microseconds.
    """
    return round(((slot12_8ns * 12.8) / 1000), 3)
