#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2020-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" Get ONU Firmware Information.

This Tibit YANG Example script retrieves firmware information for an ONU

Example:

  ./firmware_upgrade/get_onu_firmware.py --onu ALPHe30cadcf


usage: get_onu_firmware.py [--help] [-h HOST] --onu ONU [-w PASSWD] [-p PORT]
                           [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
from lxml import etree
import os
import sys
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    options = {
        "{{ONU}}" : args.onu
    }

    # Send a Netconf <get> request to retreive the ONU configuration
    ONU_CFG = '''
    <rpc
        xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db"
        message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:pon>
                    <tibitcntlr:onus>
                        <tibitcntlr:onu>
                            <name>{{ONU}}</name>
                            <tibitcntlr:onu/>
                        </tibitcntlr:onu>
                    </tibitcntlr:onus>
                </tibitcntlr:pon>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=ONU_CFG, options=options, message="/tibit-pon-controller-db::pon/tibit-pon-controller-db:onus/tibit-pon-controller-db:onu/tibit-pon-controller-db:onu")
    
    # Parse the Netconf response and retrieve the firmware bank pointer from the XML response data.
    onu_cfg_fw_bank_ptr = 65535
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'tibitcntlr' : "urn:com:tibitcom:ns:yang:controller:db",
            }
        root = etree.fromstring(rsp_xml)
        for fw_bank_ptr in root.findall("nc:data/tibitcntlr:pon/tibitcntlr:onus/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-ptr", namespaces=NSMAP):
            onu_cfg_fw_bank_ptr = int(fw_bank_ptr.text)

    # Parse the Netconf response and retrieve the firmware image file names for all banks from the XML response data.
    onu_cfg_fw_bank_files = ["", ""]
    if rsp_xml:
        for fw_bank_file in root.findall("nc:data/tibitcntlr:pon/tibitcntlr:onus/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-files", namespaces=NSMAP):
            fw_bank_file_id = fw_bank_file.find("tibitcntlr:id", namespaces=NSMAP)
            if fw_bank_file_id is not None:
                fw_bank_file_id = int(fw_bank_file_id.text)
                fw_bank_file_value = fw_bank_file.find("tibitcntlr:file", namespaces=NSMAP)
                if fw_bank_file_value is not None:
                    fw_bank_file_value = fw_bank_file_value.text
                    if fw_bank_file_value:
                        onu_cfg_fw_bank_files[fw_bank_file_id] = fw_bank_file_value

    # Parse the Netconf response and retrieve the firmware image versions for all banks from the XML response data.
    onu_cfg_fw_bank_versions = ["", ""]
    if rsp_xml:
        for fw_bank_version in root.findall("nc:data/tibitcntlr:pon/tibitcntlr:onus/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-versions", namespaces=NSMAP):
            fw_bank_version_id = fw_bank_version.find("tibitcntlr:id", namespaces=NSMAP)
            if fw_bank_version_id is not None:
                fw_bank_version_id = int(fw_bank_version_id.text)
                fw_bank_version_value = fw_bank_version.find("tibitcntlr:version", namespaces=NSMAP)
                if fw_bank_version_value is not None:
                    fw_bank_version_value = fw_bank_version_value.text
                    if fw_bank_version_value:
                        onu_cfg_fw_bank_versions[fw_bank_version_id] = fw_bank_version_value

    # Send a Netconf <get> request to retreive the state for an ONU
    ONU_STATE = '''
    <rpc
        xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db"
        message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:pon-state>
                    <tibitcntlr:onu-state>
                        <tibitcntlr:onu>
                            <name>{{ONU}}</name>
                            <tibitcntlr:onu/>
                        </tibitcntlr:onu>
                    </tibitcntlr:onu-state>
                </tibitcntlr:pon-state>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=ONU_STATE, options=options, message="/tibit-pon-controller-db::pon-state/tibit-pon-controller-db:onu-state/tibit-pon-controller-db:onu/tibit-pon-controller-db:onu")

    # Parse the Netconf response and retrieve the current firmware version from the XML response data.
    onu_state_fw_version = None
    if rsp_xml:
        root = etree.fromstring(rsp_xml)
        for fw_version in root.findall("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-version", namespaces=NSMAP):
            onu_state_fw_version = fw_version.text

    # Parse the Netconf response and retrieve the firmware bank pointer from the XML response data.
    onu_state_fw_bank_ptr = 65535
    if rsp_xml:
        for fw_bank_ptr in root.findall("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-ptr", namespaces=NSMAP):
            onu_state_fw_bank_ptr = int(fw_bank_ptr.text)

    # Parse the Netconf response and retrieve the firmware image versions for all banks from the XML response data.
    onu_state_fw_bank_versions = ["", ""]
    if rsp_xml:
        for fw_bank_version in root.findall("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-versions", namespaces=NSMAP):
            fw_bank_version_id = fw_bank_version.find("tibitcntlr:id", namespaces=NSMAP)
            if fw_bank_version_id is not None:
                fw_bank_version_id = int(fw_bank_version_id.text)
                fw_bank_version_value = fw_bank_version.find("tibitcntlr:version", namespaces=NSMAP)
                if fw_bank_version_value is not None:
                    fw_bank_version_value = fw_bank_version_value.text
                    if fw_bank_version_value:
                        onu_state_fw_bank_versions[fw_bank_version_id] = fw_bank_version_value

    # Parse the Netconf response and retrieve the last firmware upgrade status from the XML response data.
    onu_state_fw_upgrade_status = {}
    if rsp_xml:
        for field in root.findall("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-upgrade-status/*", namespaces=NSMAP):
            # Strip the XML namespace from the tag
            tag = field.tag.split("}")[1][0:]
            onu_state_fw_upgrade_status[tag] = field.text

    # Display the firmare information for the ONU
    print(f"\nFirmware Information for ONU {args.onu}:")
    print()
    # Current version information from state
    print("Current Version:")
    active_bank_str = ["",""]
    if onu_state_fw_bank_ptr < 2:
        active_bank_str[onu_state_fw_bank_ptr] = "(active)"
    print(f"  Bank 0:  {onu_state_fw_bank_versions[0]} {active_bank_str[0]}")
    print(f"  Bank 1:  {onu_state_fw_bank_versions[1]} {active_bank_str[1]}")
    print()
    # Configured versions
    configured_bank_str = ["",""]
    if onu_cfg_fw_bank_ptr < 2:
        if onu_cfg_fw_bank_versions[onu_cfg_fw_bank_ptr]:
            configured_bank_str[onu_cfg_fw_bank_ptr] = "(selected)"
    version_str = "Version"
    file_str = "File"
    if onu_cfg_fw_bank_ptr != 65535 and \
        onu_cfg_fw_bank_versions[0] and onu_cfg_fw_bank_versions[1] and \
        onu_cfg_fw_bank_files[0] and onu_cfg_fw_bank_files[1]:
        # Upgrade is "enabled"
        upgrade_status_str = ""
    else:
        # Upgrade is "disabled"
        upgrade_status_str = " (Upgrade disabled)"
    print(f"Configuration{upgrade_status_str}:")
    print(f"           {version_str:<{20}s} {file_str}")
    print(f"  Bank 0:  {onu_cfg_fw_bank_versions[0]:<{20}s} {onu_cfg_fw_bank_files[0]} {configured_bank_str[0]}")
    print(f"  Bank 1:  {onu_cfg_fw_bank_versions[1]:<{20}s} {onu_cfg_fw_bank_files[1]} {configured_bank_str[1]}")
    print()
    # Last upgrade status from state
    if onu_state_fw_upgrade_status:
        print("Last Firmware Upgrade Status:")
        print(f"  Status:     {onu_state_fw_upgrade_status['status']}")
        print(f"  Bank:       {onu_state_fw_upgrade_status['bank']}")
        print(f"  File:       {onu_state_fw_upgrade_status['file']}")
        print(f"  Progress:   {onu_state_fw_upgrade_status['progress']} %")
    else:
        print("Last Firmware Upgrade Status: None.")