#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (C) 2015 - 2022 by Tibit Communications, Inc.                  #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#   MicroPlug PON Manager                                                  #
#                                                                          #
# --------------------------------------------------------------------------#
MANAGER_VERSION = "R2.1.0"

# IMPORTS
from subprocess import check_call, CalledProcessError
import os
import sys

# Variables
required_python_version = (3, 6, 7)  # Currently required minimum python version


#########################################
# Check Python Version
# python_version - (TUPLE) Minimum python version required
#########################################
def check_python_version(python_version):
    if sys.version_info >= python_version:
        return True
    else:
        raise Exception(f"Must use Python Version: ".join(required_python_version + ". You have: " + {python_version}))


#########################################
# Check Root Permissions
# Verify that root permissons were given
#########################################
def check_root_permissions():

    if os.geteuid() != 0:
        raise Exception("Must use root permissions. (Sudo)")
    else:
        return True


#########################################
# Disabling Pon Manager Apache Sites
#########################################
def disable_pon_manager_apache():
    try:
        check_call(['a2dissite', 'tibitdev-web'],
                   stderr=open(os.devnull, 'wb'))
    except CalledProcessError as err:
        print("--Web Site Already Disabled")

    try:
        check_call(['a2dissite', 'tibitdev-api'],
                   stderr=open(os.devnull, 'wb'))
    except CalledProcessError as err:
        print("--API Site Already Disabled")

#########################################
# Removing Pon Manager Apache Sites
#########################################
def remove_pon_manager_apache_configs():
    try:
        check_call(['rm', '/etc/apache2/sites-available/tibitdev-web.conf'],
                   stderr=open(os.devnull, 'wb'))
    except CalledProcessError as err:
        print("--Web Config Already Removed")

    try:
        check_call(['rm', '/etc/apache2/sites-available/tibitdev-api.conf'],
                   stderr=open(os.devnull, 'wb'))
    except CalledProcessError as err:
        print("--API Config Already Removed")

#########################################
# Disabling all Pon Manager documents
#########################################
def remove_pon_manager_files():
    try:
        check_call(['rm', '-r', '/var/www/html/'],
                   stderr=open(os.devnull, 'wb'))
    except CalledProcessError as err:
        print("--Pon Manager Files Already Removed")



#############################################
# Entry Point
#############################################

if __name__ == '__main__':

    try:
        # Checking Python Version
        check_python_version(required_python_version)

        # checking for root permissions
        check_root_permissions()

        # Disabling Pon Manager Apache Sites
        disable_pon_manager_apache()

        # Removing Pon Manager Apache Sites
        remove_pon_manager_apache_configs()

        # Disabling all Pon Manager documents
        remove_pon_manager_files()

    except Exception as err:
        print(err)
        exit()
