#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
#--------------------------------------------------------------------------#
""" List ONU devices for an OLT.

This MCMS REST API example script lists all ONU devices for a specific OLT,
along with ONU registration status and PON related status for the ONU.

Example:

  ./get_onu_status.py --url http://10.2.10.29/api --user <email> --password <password> --olt e8:b4:70:70:0c:9c


usage: get_onu_status.py [-d DATABASE] [-f FILTER] [-h] [-l URL] --olt OLT
                         [-p PASSWORD] [-u USER] [-v]

optional arguments:
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  -f FILTER, --filter FILTER
                        Filter by ONU state (e.g., deregistered, disabled,
                        disallowed-admin, disallowed-error, dying-gasp,
                        registered, unprovisioned, unspecified) (default:
                        None)
  -h, --help            Show this help message and exit.
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  --olt OLT             OLT MAC Address (e.g., e8:b4:70:70:0c:9c) (default:
                        None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
from api_client import ApiClient
from api_utilities import dict_read


def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("-f", "--filter", action="store", dest="filter", default=None, required=False, help="Filter by ONU state (e.g., Dregistered, Disabled, Disallowed-Admin, Disallowed-Error, Dying-Gasp, Registered, Unprovisioned, Unspecified)")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., e8:b4:70:70:0c:9c)")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # If a filter was specified, only display ONUs with a state that matches the filter
    if args.filter:
        filtered_onu_states = [args.filter]
    else:
        filtered_onu_states = [
            # There are situations where the ONU can appear in 'unprovisioned' or 'unspecified'
            # in addition to other states. As a result, processes these states first, then
            # overwrite later with the actual state.
            "Unprovisioned",
            "Unspecified",
            # ONUs appear in the remaining states only once.
            "Deregistered",
            "Disabled",
            "Disallowed Admin",
            "Disallowed Error",
            "Dying Gasp",
            "Registered",
        ]

    # Get the OLT-STATE.ONU States for the specified OLT
    onu_status = {}
    status, olt_state = api_client.request("GET", f"/v1/olts/states/{args.olt}/")
    if status == 200 and olt_state:
        for olt_onu_state in filtered_onu_states:
            for onu_ssn in olt_state['ONU States'][olt_onu_state]:
                onu_status[onu_ssn] = {}
                onu_status[onu_ssn]['id'] = onu_ssn
                onu_status[onu_ssn]['status'] = olt_onu_state

    # Get ONU state data
    for onu_ssn in onu_status.keys():
        status, onu_state = api_client.request("GET", f"/v1/onus/states/{onu_ssn}/")
        if status == 200 and onu_state:
            onu_status[onu_ssn]['onu_id'] = dict_read(onu_state, 'ONU.ALLOC ID (OMCC)', default=0)
            onu_status[onu_ssn]['fiber_distance'] = dict_read(onu_state, 'STATS.OLT-PON.Fiber Distance', default=0.0)
            onu_status[onu_ssn]['tx_power'] = dict_read(onu_state, 'STATS.OLT-PON.TX Optical Level', default=0.0)
            onu_status[onu_ssn]['rx_power'] = dict_read(onu_state, 'STATS.OLT-PON.RX Optical Level', default=0.0)
            onu_status[onu_ssn]['downstream_rate'] = 0
            onu_status[onu_ssn]['upstream_rate'] = 0
            for port_num in range(0, 8):
                onu_status[onu_ssn]['downstream_rate'] += dict_read(onu_state, f'STATS.OLT-PON Service {port_num}.TX Rate bps', default=0)
                onu_status[onu_ssn]['upstream_rate'] += dict_read(onu_state, f'STATS.OLT-PON Service {port_num}.RX Rate bps', default=0)
            #ber_value = dict_read(onu_state, 'STATS.OLT-PON.RX Pre-FEC BER', default=0.0)
            #onu_status[onu_ssn]['pre_fec_ber'] = 0.0 if ber_value == 0 else 1e-12 / ber_value
            onu_status[onu_ssn]['pre_fec_ber'] = dict_read(onu_state, 'STATS.OLT-PON.RX Pre-FEC BER', default=0.0) / 1e12
            onu_status[onu_ssn]['post_fec_ber'] = dict_read(onu_state, 'STATS.OLT-PON.RX Post-FEC BER', default=0.0) / 1e12
        else:
            onu_status[onu_ssn]['status'] = "Provisioned"
            onu_status[onu_ssn]['onu_id'] = ""
            onu_status[onu_ssn]['fiber_distance'] = ""
            onu_status[onu_ssn]['tx_power'] = ""
            onu_status[onu_ssn]['rx_power'] = ""
            onu_status[onu_ssn]['downstream_rate'] = ""
            onu_status[onu_ssn]['upstream_rate'] = ""
            onu_status[onu_ssn]['pre_fec_ber'] = ""
            onu_status[onu_ssn]['post_fec_ber'] = ""

    # Logout of the web server to terminate the session
    api_client.logout()

    # Display the ONU registration status of for the OLT  20000.000
    print(f"\nONU Devices on OLT {args.olt}:")
    print("{:<17}  {:<6}  {:<12}  {:<9}  {:<8}  {:<8}  {:<12}  {:<12}  {:<8}  {:<8}".format(
        '  Serial Num', 'ONU ID', 'Status', ' Distance', 'Tx Power', 'Rx Power', '  Downstream', '  Upstream ', ' Pre-FEC', 'Post-FEC'))
    print("{:<17}  {:<6}  {:<12}  {:<9}  {:<8}  {:<8}  {:<12}  {:<12}  {:<8}  {:<8}".format(
        '', '', '', ' (km)', '(dbm)', '(dbm)', '  Rate (bps)', '  Rate (bps)', ' BER', 'BER'))
    for onu_ssn, status in sorted(onu_status.items()):
        print("{:<17}  {:>6}  {:<12}  {:>9}  {:>8}  {:>8}  {:>12}  {:>12}  {:>8.3}  {:>8.3}".format(
            status['id'],
            status['onu_id'],
            status['status'],
            status['fiber_distance'],
            status['tx_power'],
            status['rx_power'],
            status['downstream_rate'],
            status['upstream_rate'],
            status['pre_fec_ber'],
            status['post_fec_ber']))

if __name__ == '__main__':
    main()
