#!/usr/bin/env bash
# --------------------------------------------------------------------------#
#  Copyright (C) 2021 by Tibit Communications, Inc.                         #
#  All rights reserved.                                                     #
#                                                                           #
#     _______ ____  _ ______                                                #
#    /_  __(_) __ )(_)_  __/                                                #
#     / / / / __  / / / /                                                   #
#    / / / / /_/ / / / /                                                    #
#   /_/ /_/_____/_/ /_/                                                     #
#                                                                           #
#                                                                           #
#  Distributed as Tibit-Customer confidential                               #
#                                                                           #
# --------------------------------------------------------------------------#

TMPFILE=$(mktemp /tmp/tibit_nc_rpc.XXXXXX)

# Usage
usage () {
    cat <<HELP_USAGE

    $0 config-file

    Load configuration from an XML file into the running configuration datastore.

    Options:
      -h|--help       Display this help and usage text.
      -p|--port       TCP port number used to connect to the NETCONF Server (default: 830).
      -s|--server     Hostname or IP address used to connect to the NETCONF Server (default: localhost).
      -u|--user       User used to connect to the NETCONF Server

    Example:
      $0  running_config.xml

HELP_USAGE

exit 0
}

# Defaults
HOST=localhost
PORT=830

# Parse command line arguments
PARAMS=""
while (( "$#" )); do
    case "$1" in
        -s|--server)
            HOST=$2
            shift 2
            ;;
        -p|--port)
            PORT=$2
            shift 2
            ;;
        -u|--user)
            USER=$2
            shift 2
            ;;
        -h|--help)
            usage
            ;;
        --) # end argument parsing
            shift
            break
            ;;
        -*|--*=) # unsupported option
            echo "Error: Unsupported option $1" >&2
            usage
            ;;
        *) # preserve positional arguments
            PARAMS="$PARAMS $1"
            shift
            ;;
    esac
done
# set positional arguments in their proper place
eval set -- "$PARAMS"

[ -z $1 ] && { usage; }

CONFIG_FILE=$1

if [ -z "$USER" ]
then
   LOGIN=""
else
   LOGIN="--login $USER"
fi

#
# Load XML from the file to the candidate configuration
#
echo "Loading configuration from '${CONFIG_FILE}' into the running configuration datastore."
(
PATH=${PATH}:/opt/tibit/netconf/bin netopeer2-cli <<COMMANDINPUT
connect --host ${HOST} --port ${PORT} ${LOGIN}
edit-config --target running --config=${CONFIG_FILE} --defop merge --test test-then-set --error rollback
COMMANDINPUT
)

rm -rf ${TMPFILE}
