#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

""" Configure an Add CTag Service for an ONU.

This Tibit YANG Example script configures an "Add CTag Service" for an ONU,
where the OLT adds an outer STag and the ONU adds an inner CTag. This example
also configures SLA and DHCP Relay Option 82 for the ONU.

Example - Configure ONU ALPHe30cadcf with outer tag 200 and inner tag 25:

  ./add_ctag_service/edit_config_add_ctag_svc.py \
      --olt 70:b3:d5:52:37:24 \
      --onu ALPHe30cadcf \
      --olt_tag 200 \
      --onu_tag 25 \
      --sla Max \
      --opt82-circuit-id "tibit pon 1/0/1:vlan200.25" \
      --opt82-remote-id "ALPHe30cadcf"


usage: edit_config_add_ctag_svc.py [--help] [-h HOST] --olt OLT
                                   [--olt_tag OLT_TAG] --onu ONU --onu_tag
                                   ONU_TAG [--option_82 OPTION_82]
                                   [--opt82-circuit-id CIRCUIT_ID]
                                   [--opt82-remote-id REMOTE_ID] [-w PASSWD]
                                   [-p PORT] --sla SLA [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --olt OLT             OLT MAC Address (e.g., 70:b3:d5:52:37:24) (default:
                        None)
  --olt_tag OLT_TAG     Tag to be added by the OLT (default: 0)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  --onu_tag ONU_TAG     Tag to be added by the ONU (default: None)
  --option_82 OPTION_82
                        Option 82 Value (hex string) (default: None)
  --opt82-circuit-id CIRCUIT_ID
                        DHCP Relay Option 82, Circuit ID (default: )
  --opt82-remote-id REMOTE_ID
                        DHCP Relay Option 82, Remote ID (default: )
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  --sla SLA             SLA (default: None)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import itertools
from lxml import etree
import os
import sys
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver

def get_onu_ids(nc, options):
    """
    Get a dictionary of ONU ID values configured in ONU Inventory for an OLT.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        onu_ids: A dictionary mapping ONU serial number to ONU ID configured in ONU inventory
    """

    # Send a Netconf <get> request to retrieve the ONU Inventory for an OLT
    OLT_CFG_ONU_INVENTORY = '''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:olt xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                <tibitcntlr:olt>
                    <name>{{OLT}}</name>
                    <tibitcntlr:onus/>
                </tibitcntlr:olt>
                </tibitcntlr:olt>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=OLT_CFG_ONU_INVENTORY, options=options, message="/tibit-pon-controller-db:olt/tibit-pon-controller-db:olt[name={{OLT}}]/tibit-pon-controller-db:onus")

    # Parse the Netconf response and build a dictionary mapping ONU serial number
    # to ONU IDs from the XML response data.
    NSMAP = {
        'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
        'tibit' : "urn:com:tibitcom:ns:yang:controller:db",
        }
    root = etree.fromstring(rsp_xml)
    onu_ids = {}
    for onu in root.findall(f"nc:data/tibit:olt/tibit:olt/tibit:onus", namespaces=NSMAP):
        onu_serial_number = onu.find("tibit:id", namespaces=NSMAP)
        if onu_serial_number is not None:
            onu_serial_number = onu_serial_number.text
            onu_id = onu.find("tibit:alloc-id-omcc", namespaces=NSMAP)
            if onu_id is not None:
                onu_id = onu_id.text
                onu_ids[onu_serial_number] = int(onu_id)

    return onu_ids

def get_next_onu_id(nc, options):
    """
    Get the next free ONU ID that has _not_ been configured in the ONU Inventory.

    If the list of empty, return the first valid ONU ID value '1'. Otherwise find the first
    unused value starting from from 1 to 128.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        next_onu_id: The next available ONU ID value or 'None'.
    """
    next_onu_id = None
    onu_ids = get_onu_ids(nc, options)
    if len(onu_ids.values()) == 0:
        next_onu_id = 1
    else:
        for value in range(1,129):
            if value not in onu_ids.values():
                next_onu_id = value
                break

    return next_onu_id

def get_onu_id_from_serial_number(nc, serial_number, options):
    """
    Get the configured ONU ID value for an ONU by serial number.

    Args:
        nc: Netconf driver object.
        serial_number: ONU Vendor-specific serial number
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        onu_id: The ONU ID configured for this ONU or 'None'.
    """
    onu_ids = get_onu_ids(nc, options)
    onu_id = None
    if serial_number in onu_ids:
        onu_id = onu_ids[serial_number]

    return onu_id

def get_alloc_ids(nc, options):
    """
    Get a dictionary of Alloc ID values per ONU configured in ONU Inventory for an OLT.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        alloc_ids: A dictionary of a list Alloc IDs for each ONU configured in ONU inventory
    """

    # Send a Netconf <get> request to retrieve the ONU Inventory for an OLT
    OLT_CFG_ONU_INVENTORY = '''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:olt xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                <tibitcntlr:olt>
                    <name>{{OLT}}</name>
                    <tibitcntlr:onus/>
                </tibitcntlr:olt>
                </tibitcntlr:olt>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=OLT_CFG_ONU_INVENTORY, options=options, message="/tibit-pon-controller-db:olt/tibit-pon-controller-db:olt[name={{OLT}}]/tibit-pon-controller-db:onus")

    # Parse the Netconf response and build a list of Alloc IDs from the XML response data.
    # Each ONU can be configured with multiple Alloc IDs. Therefore, build a dictionary with the
    # format { ONU : [] } that contains a list of Alloc IDs for each ONU entry in the dictionary.
    NSMAP = {
        'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
        'tibit' : "urn:com:tibitcom:ns:yang:controller:db",
        }
    root = etree.fromstring(rsp_xml)
    alloc_ids = {}
    for onu in root.findall(f"nc:data/tibit:olt/tibit:olt/tibit:onus", namespaces=NSMAP):
        onu_serial_number = onu.find("tibit:id", namespaces=NSMAP)
        if onu_serial_number is not None:
            onu_serial_number = onu_serial_number.text
            alloc_ids[onu_serial_number] = []
            for alloc_id in onu.findall("tibit:olt-service/tibit:unicast-id", namespaces=NSMAP):
                alloc_id = alloc_id.text
                alloc_ids[onu_serial_number].append(int(alloc_id))

    return alloc_ids

def get_next_alloc_id(nc, options):
    """
    Get the next free Alloc ID that has _not_ been configured in the ONU Inventory.

    If the list of empty, return the first valid Alloc ID value '1154'. Otherwise find the first
    unused value starting from from 1154 to 1534.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        next_alloc_id: The next available Alloc ID value or 'None'.
    """
    next_alloc_id = None
    alloc_ids = get_alloc_ids(nc, options)
    alloc_ids = alloc_ids.values()
    alloc_ids = list(itertools.chain.from_iterable(alloc_ids))
    if len(alloc_ids) == 0:
        next_alloc_id = 1154
    else:
        # Need to exclude values reserved by HW: 0x4FF, 0x5FF (already excluded from the range above)
        restricted_values = [0x4FF, 0x5FF]
        for value in range(1154,1535):
            if value not in alloc_ids:
                if value not in restricted_values:
                    next_alloc_id = value
                    break

    return next_alloc_id

def get_alloc_id_from_serial_number(nc, serial_number, options):
    """
    Get the configured Alloc ID value for an ONU by serial number.

    Note, this routine assumes only one Alloc ID is configured for each ONU. As a result,
    only the first Alloc ID value is returned.

    Args:
        nc: Netconf driver object.
        serial_number: ONU Vendor-specific serial number
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        alloc_id: The Alloc ID configured for this ONU or 'None'.
    """
    alloc_ids = get_alloc_ids(nc, options)
    alloc_id = None
    if serial_number in alloc_ids:
        if len(alloc_ids[serial_number]) > 0:
            alloc_id = alloc_ids[serial_number][0]

    return alloc_id

def get_nni_networks(nc, options):
    """
    Get a list of NNI Networks configured in NNI Inventory for an OLT.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        nni_networks: A dictionary of a list Alloc IDs for each ONU configured in ONU inventory
    """

    # Send a Netconf <get> request to retrieve the ONU Inventory for an OLT
    OLT_CFG_NNI_NETWORKS = '''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:olt xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                <tibitcntlr:olt>
                    <name>{{OLT}}</name>
                    <tibitcntlr:nni-networks/>
                </tibitcntlr:olt>
                </tibitcntlr:olt>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=OLT_CFG_NNI_NETWORKS, options=options, message="/tibit-pon-controller-db:olt/tibit-pon-controller-db:olt[name={{OLT}}]/tibit-pon-controller-db:nni-networks")

    # Parse the Netconf response and build a list of NNI Networks from the XML response data.
    NSMAP = {
        'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
        'tibit' : "urn:com:tibitcom:ns:yang:controller:db",
        }
    root = etree.fromstring(rsp_xml)
    nni_networks = []
    for nni_network in root.findall(f"nc:data/tibit:olt/tibit:olt/tibit:nni-networks", namespaces=NSMAP):
        nni_network_id = nni_network.find("tibit:id", namespaces=NSMAP)
        if nni_network_id is not None:
            nni_network_tags = nni_network.find("tibit:tags", namespaces=NSMAP)
            if nni_network_tags is not None:
                nni_networks.insert(int(nni_network_id.text), nni_network_tags.text)

    return nni_networks

def get_next_nni_network_id(nc, options):
    """
    Get the next free NNI Network index (id) that has _not_ been configured in the NNI Inventory.

    If the list of empty, return the first valid index value = '0'. Otherwise, find the first
    unused value starting from from '0'.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        next_nni_network_id: The next available NNI Network index (id) value or 'None'.
    """
    next_nni_network_id = None
    nni_networks = get_nni_networks(nc, options)
    if len(nni_networks) == 0:
        next_nni_network_id = 0
    else:
        next_nni_network_id = len(nni_networks)

    return next_nni_network_id

def get_nni_network_id_from_tags(nc, tags, options):
    """
    Get the next free NNI Network index (id) value for the specified VLAN tags.

    Args:
        nc: Netconf driver object.
        tags: VLAN tags with the format 'sX.cY.cZ'
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        nni_network_id: The NNI Network index (id) value for the specified VLAN tags or 'None'.
    """
    nni_network_id = None
    nni_networks = get_nni_networks(nc, options)
    if tags in nni_networks:
        nni_network_id = nni_networks.index(tags)

    return nni_network_id

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., 70:b3:d5:52:37:24)")
    parser.add_argument(      "--olt_tag", action="store", dest="olt_tag", default=None, required=True, help="Tag to be added by the OLT")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument(      "--onu_tag", action="store", dest="onu_tag", default=None, required=True, help="Tag to be added by the ONU")
    parser.add_argument(      "--option_82", action="store", dest="option_82", default=None, required=False, help="Option 82 Value (hex string)")
    parser.add_argument(      "--opt82-circuit-id", action="store", dest="circuit_id", default="", required=False, help="DHCP Relay Option 82, Circuit ID")
    parser.add_argument(      "--opt82-remote-id", action="store", dest="remote_id", default="", required=False, help="DHCP Relay Option 82, Remote ID")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument(      "--sla", action="store", dest="sla", default=None, required=True, help="SLA")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    # The variables {{VAR}} be substituted in the Netconf requests from the .xml files sent below
    options = {
        "{{CIRCUIT_ID}}" : args.circuit_id,
        "{{OLT}}" : args.olt,
        "{{OLT_TAG}}" : args.olt_tag,
        "{{ONU}}" : args.onu,
        "{{ONU_TAG}}" : args.onu_tag,
        "{{REMOTE_ID}}" : args.remote_id,
        "{{SLA}}" : args.sla,
    }

    # Get the ONU ID for this ONU
    onu_id = get_onu_id_from_serial_number(nc, args.onu, options)
    if onu_id is None:
        onu_id = get_next_onu_id(nc, options)
    options['{{ONU_ID}}'] = onu_id

    # Get the Alloc ID for this ONU
    alloc_id = get_alloc_id_from_serial_number(nc, args.onu, options)
    if alloc_id is None:
        alloc_id = get_next_alloc_id(nc, options)
    options['{{ALLOC_ID}}'] = alloc_id

    # Get the NNI Network Inventory table entry id (key)
    nni_network_id = get_nni_network_id_from_tags(nc, f"s{args.olt_tag}.c{args.onu_tag}.c0", options)
    if nni_network_id is None:
        nni_network_id = get_next_nni_network_id(nc, options)
    options['{{NNI_NETWORK_ID}}'] = nni_network_id

    # Configure the service VLANs, SLA, and Option 82
    nc.edit_config(filename="1-onu-cfg.xml", options=options)
    # Configure the ONU ID and Alloc ID in the ONU inventory
    nc.edit_config(filename="2-olt-cfg-onu-inventory.xml", options=options)
    # Configure the NNI Network in the NNI inventory
    nc.edit_config(filename="3-olt-cfg-nni-inventory.xml", options=options)

    # Display a summary of what was configured
    print(f"\nProvisioned ONU {options['{{ONU}}']}")
    print(f" OLT VLAN Tag:  {options['{{OLT_TAG}}']}")
    print(f" ONU VLAN Tag:  {options['{{ONU_TAG}}']}")
    print(f" PON Alloc ID:  {options['{{ALLOC_ID}}']}")
    print(f" PON ONU ID:    {options['{{ONU_ID}}']}")
    print(f" SLA:           {options['{{SLA}}']}")
    print(f" Option 82:")
    print(f"  Circuit ID:   {options['{{CIRCUIT_ID}}']}")
    print(f"  Remote ID:    {options['{{REMOTE_ID}}']}")
