#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import sys
import os
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver


if __name__ == '__main__':

    # Example: ./config_add_ctag_svc.py --onu ALPHe30cadcf --olt_tag 200 --onu_tag 2

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt_port", action="store", dest="olt_port", default=None, required=True, help="OLT Port number. This could be a logical port number or a physical port number representing the switch port (e.g., LLDP switch port ID)")
    parser.add_argument(      "--olt_tag", action="store", dest="olt_tag", default=None, required=True, help="Tag to be added by the OLT")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-t", "--traffic_descriptor_profile", action="store", dest="traffic_descriptor_profile", default="Max", required=False, help="Configure the traffic descriptor profile name (e.g., SLA) for this service.")
    parser.add_argument(      "--uni", action="store", dest="uni_port", default='1', required=False, help="UNI port number 1..5")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    # The variables {{VAR}} be substituted in the Netconf requests from the
    # .xml files sent below
    options = {
        "{{OLT_PORT}}" : args.olt_port,
        "{{OLT_TAG}}" : args.olt_tag,
        "{{ONU}}" : args.onu,
        "{{TRAFFIC_DESCRIPTOR_PROFILE}}" : args.traffic_descriptor_profile,
        "{{UNI_PORT}}" : args.uni_port,
    }

    nc.edit_config(filename="1-onu-interfaces.xml", options=options)
    nc.edit_config(filename="2-link-table.xml", options=options)
    nc.edit_config(filename="3-onu-vlan-sub-interfaces.xml", options=options)
    nc.edit_config(filename="4-gemports.xml", options=options)
    nc.edit_config(filename="5-olt-1to1-forwarding.xml", options=options)
