#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

from collections import defaultdict
import math
import re
from lxml import etree

def get_dict_from_etree(root):
    """ Used to convert the etree representation of the rpc reply xml
    into a python readable dictionary object

    Args:
        root (lxml.etree.Element): The root Element of the etree object representing
        the xml from the rpc reply

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    def _get_dict_from_etree(root):
        if root.tag:
            tag = root.tag.split("}")[1][0:]
        else:
            tag = None
        dict_ = {tag: {} if root.attrib else None}
        children = list(root)
        if children:
            def_dict = defaultdict(list)
            for child_dict in map(get_dict_from_etree, children):
                for k, v in child_dict.items():
                    def_dict[k].append(v)
            dict_ = {tag: {k: v[0] if len(v) == 1 else v
                        for k, v in def_dict.items()}}

        if root.text:
            text = root.text.strip()
            if children or root.attrib:
                if text:
                    dict_[tag]['#text'] = text
            else:
                dict_[tag] = text
        return dict_

    dict_ = _get_dict_from_etree(root)
    if 'rpc-reply' in dict_ and 'data' in dict_['rpc-reply']:
        dict_ = dict_['rpc-reply']['data']
    else:
        dict = {}

    return dict_


def nat_sort(keys):
    """ A natural sorting algorithm

    Args:
        keys (list or str): A string or list of strings that map to dict keys
    Returns:
        lambda: Returns the lambda function to be used as a sort
        function key

    """
    # Make sure keys is a list
    if not isinstance(keys, list):
        keys = [keys]
    # Converts text digits to digits or skips
    convert = lambda text: int(text) if text.isdigit() else text
    # Splits off the numbers and converts to ints
    alphanum_key = lambda key: [convert(c) for c in re.split('([0-9]+)', key)]
    # Passes the list of variables held in the dictionary key or keys specified by the user
    # and returns the resulting organizational function
    return lambda final: alphanum_key(final) if isinstance(final, str) \
        else tuple(alphanum_key(final[key]) for key in keys)


def mw_to_dbm(mW):
    """ Used to convert milliWatts into decibel-milliwatts

    Args:
        value (int): The current value to be rounded (in mW)

    Returns:
        float: The dBm representation of the value in float form

    """
    if mW == 0:
        mW = 0.025
    return 10.0 * math.log10(mW)


def parse_port_number_from_name(name):
    """ Parse a port number from the end of a UNI Port or TCONT name

    For example,
        onu-ALPHe30cadcf-eth.5     =>  parses the '5' from the UNI Port name
        vani-ALPHe30cadcf-tcont.1  =>  parses the '1' from the TCONT name

    Args:
        name (str): UNI Port or TCONT name

    Returns:
        int: The port number of instance number parsed from the end of the name

    """
    try:
        port_num = re.match('.*?([0-9]+)$', name).group(1)
        port_num = int(port_num)
    except:
        port_num = 0
    return port_num