#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# --------------------------------------------------------------------------#

"""Get the list of Tibit MicroPlug OLT devices using BBF Yang Models.

This BBF Yang example script retrieves all OLTs and their current state.

Example:

  ./bbf_list_olts.py


usage: bbf_list_olts.py [--help] [-h HOST] [-w PASSWD]
                                 [-p PORT] [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys

from lxml import etree

from netconf_driver import NetconfDriver
from netconf_utils import get_dict_from_etree, nat_sort


def get_ietf_hardware_by_name(nc, name):
    """ Get an ietf-hardware entry by componet name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The detected name of the desired hardware component

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f''' <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:bbf-hw-xcvr="urn:bbf:yang:bbf-hardware-transceivers"
                                     xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name>{name}</hw:name>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                 </rpc>'''
    xml = nc.get(data_xml=data_xml, message=f"/hw:hardware/hw:component[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        data = data['hardware']['component']
    else:
        data = None
    return data


def get_ietf_hardware_names(nc):
    """ Get a list of all ietf-hardware entry names

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    hardware_names = []
    data_xml=f'''<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name/>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                </rpc>'''
    xml = nc.get(data_xml=data_xml, message="nc:data/hw:hardware/hw:component/hw:name")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        for name in data['hardware']['component']:
            hardware_names.append(name['name'])
    return hardware_names


def get_ietf_interface_state_by_name(nc, name):
    """ Get an IETF interface-state entry by interface name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name of the desired /ietf:interfaces-state/ietf:interface

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces-state>
                          <if:interface>
                            <if:name>
                                {name}
                            </if:name>
                          </if:interface>
                        </if:interfaces-state>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces-state/if:interface[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces-state' in data and 'interface' in data['interfaces-state']:
        data = data['interfaces-state']['interface']
    else:
        data = None
    return data


def get_tibit_bbf_olt_interface_map(nc, device_id):
    """ Get the Tibit YANG BBF OLT interface mapping entry by OLT device id (MAC Address)

    Args:
        nc (obj): Netconf connection to use for this request.
        device_id (str): The device-id used for the interface that appears in the device-id field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                <get>
                  <filter type="subtree">
                    <tibit-bbf-if:interfaces xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces">
                      <tibit-bbf-if:olt-interface-map>
                        <tibit-bbf-if:olt>
                          <tibit-bbf-if:device-id>{device_id}</tibit-bbf-if:device-id>
                        </tibit-bbf-if:olt>
                      </tibit-bbf-if:olt-interface-map>
                    </tibit-bbf-if:interfaces>
                  </filter>
                </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map[device_id={device_id}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'olt-interface-map' in data['interfaces'] and 'olt' in data['interfaces']['olt-interface-map']:
        data = data['interfaces']['olt-interface-map']['olt']
    else:
        data = None
    return data


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False,
                        help="NETCONF Server IP address or hostname.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False,
                        help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False,
                        help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False,
                        help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Create a connection to the Netconf server.
    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    #
    # Gather information via BBF YANG for displaying the list of OLTs.
    #
    olts = []

    # Get all of the hardware components
    hardware_names = get_ietf_hardware_names(nc)

    # Iterate through the hardware components and find the ones marked OLT
    for hardware_name in hardware_names:
        # Check to see if OLT is not in the component name and skip it
        if 'OLT' not in hardware_name:
            continue

        hardware_interface = get_ietf_hardware_by_name(nc, hardware_name)
        if not hardware_interface:
            continue

        # Initialize the entry with the olt-mac, switch-port and firmware revision until later
        entry = {
          'olt-mac': hardware_name.split(" ")[2],
          'switch-port': hardware_name.split(" ")[3].strip("[]"),
          'firmware-rev': str(hardware_interface['firmware-rev']),
          'oper-status': '-',
          'onus': '-',
        }

        # Look up the PON channel termination interface name
        olt_interface_map = get_tibit_bbf_olt_interface_map(nc, entry["olt-mac"])
        if olt_interface_map:
            # If the OLT is mapped get its channel-termination-ref
            channel_termination_name = olt_interface_map['pon']['channel-termination-ref']

            # Get the PON Port State
            olt_interface_state = get_ietf_interface_state_by_name(nc, channel_termination_name)
            if olt_interface_state:
              # Get the operational status of the OLT
              entry['oper-status'] = olt_interface_state['oper-status']

              # Get the status of ONUs attached to the OLTs
              if 'channel-termination' in olt_interface_state:
                if 'onus-present-on-local-channel-termination' in olt_interface_state['channel-termination']:
                    if isinstance(olt_interface_state['channel-termination']['onus-present-on-local-channel-termination']['onu'], list):
                      onus = olt_interface_state['channel-termination']['onus-present-on-local-channel-termination']['onu']
                    else:
                      onus = [olt_interface_state['channel-termination']['onus-present-on-local-channel-termination']['onu']]
                    if onus:
                        online = 0
                        total = 0
                        for onu in onus:
                            total += 1
                            if str(onu['onu-presence-state'].split(':')[1]) == "onu-present-and-on-intended-channel-termination":
                                online += 1
                        entry["onus"] = f"{online}/{total}"

        # Add entry to the list of OLTs
        olts.append(entry)

    #
    # Display the list of OLTs
    #

    # Print table header row
    print("")
    print(f"{'OLT': <24}"
          f"{'Switch Port': <16}"
          f"{'Operational Status': <24}"
          f"{'Firmware Version': <24}"
          f"{'ONUs (online/total)': <18}")

    # Display OLT entries sorted by MAC address
    for olt in sorted(olts,key=nat_sort('olt-mac')):
        print(f"{olt['olt-mac']: <24}"
              f"{olt['switch-port']: <16}"
              f"{olt['oper-status']: <24}"
              f"{olt['firmware-rev']: <24}"
              f"{olt['onus']: <18}")
