#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# --------------------------------------------------------------------------#

""" Get ONU information via BBF Yang models.

This BBF YANG Example script retrieves all information for an ONU.

Example:

  ./bbf_get_onu.py --onu TBIT00000001


usage: bbf_get_onu.py [--help] [-h HOST] --onu ONU [-w PASSWD]
                                 [-p PORT] [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --onu ONU             ONU Serial Number or v-ANI name (e.g., TBITc84c00df or vani-TBITc84c00df) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys

from lxml import etree

from netconf_driver import NetconfDriver
from netconf_utils import get_dict_from_etree, nat_sort, mw_to_dbm, parse_port_number_from_name


def get_ietf_interface_by_name(nc, name, if_type=None):
    """ Get an ietf-interface entry by name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name used for the interface that appears in the name field.
        type (str): Optional interface type filter.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    if if_type:
        if_type_filter = f"<type>{if_type}</type>"
    else:
        if_type_filter = ""
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-xponift="urn:bbf:yang:bbf-xpon-if-type"
                    xmlns:bbfift="urn:bbf:yang:bbf-if-type"
                    xmlns:ianaift="urn:ietf:params:xml:ns:yang:iana-if-type"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces>
                          <if:interface>
                            <if:name>
                                {name}
                            </if:name>
                            {if_type_filter}
                          </if:interface>
                        </if:interfaces>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces/if:interface[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'interface' in data['interfaces']:
        data = data['interfaces']['interface']
    else:
        data = None
    return data


def get_ietf_interface_by_vani_ssn(nc, ssn):
    """ Get a V-ANI interface entry by ONU Serial Number by filtering on the
    expected-serial-number field.

    Args:
        nc (obj): Netconf connection to use for this request.
        ssn (str): The expected serial number of the desired vani interface.
        This is located in the interface's v-ani section under the
        expected-serial-number field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-xponvani="urn:bbf:yang:bbf-xponvani"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces>
                          <if:interface>
                            <bbf-xponvani:v-ani>
                              <expected-serial-number>
                              {ssn}
                              </expected-serial-number>
                            </bbf-xponvani:v-ani>
                          </if:interface>
                        </if:interfaces>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces/if:interface/bbf-xponvani:v-ani[expected-serial-number={ssn}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'interface' in data['interfaces']:
        data = data['interfaces']['interface']
    else:
        data = None
    return data


def set_bbf_interface_performance_enable(nc, name, enable=False):
    """ Set the performance monitoring enable or disable collection of performance data.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name used for the interface that appears in the name field.
        enable (str): Sets the enable field for performance allowing performance stats to be
        collected.

    Returns:

    """
    # Convert the Python boolean to Netconf
    nc_enable = 'true' if enable else 'false'
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                  <edit-config>
                    <target>
                      <running/>
                    </target>
                    <config>
                      <interfaces xmlns="urn:ietf:params:xml:ns:yang:ietf-interfaces">
                        <interface>
                          <name>{name}</name>
                          <performance xmlns="urn:bbf:yang:bbf-interfaces-performance-management" nc:operation="merge" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                            <enable>{nc_enable}</enable>
                          </performance>
                        </interface>
                      </interfaces>
                    </config>
                  </edit-config>
                </rpc>"""
    nc.edit_config(data_xml=data_xml, message=f"/if:interfaces/if:interface[name={name}]/performance = {nc_enable}")


def get_ietf_interface_state_by_name(nc, name, with_stats=False):
    """ Get an IETF interface-state entry by name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name of the desired /ietf:interfaces-state/ietf:interface

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-xponift="urn:bbf:yang:bbf-xpon-if-type"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces-state>
                          <if:interface>
                            <if:name>
                                {name}
                            </if:name>
                            <if:oper-status/>
                            <if:last-change/>
                            <if:phys-address/>
                            <if:speed/>
                            <v-ani xmlns="urn:bbf:yang:bbf-xponvani"/>
                            <performance xmlns="urn:bbf:yang:bbf-interfaces-performance-management">
                            <intervals-15min>
                              <current>
                                <xpon xmlns="urn:bbf:yang:bbf-xpon-performance-management">
                                  <power-monitoring>
                                    <tx-optical-power-level/>
                                  </power-monitoring>
                                </xpon>
                              </current>
                            </intervals-15min>
                          </performance>
                          </if:interface>
                        </if:interfaces-state>
                      </filter>
                    </get>
                </rpc>"""
    # If requested, temporarily enable performance management to retrieve statistics
    if with_stats:
        set_bbf_interface_performance_enable(nc, name, enable=True)
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces-state/if:interface[name={name}]")
    # Disable stats after retrieveing counters
    if with_stats:
        set_bbf_interface_performance_enable(nc, name, enable=False)
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces-state' in data and 'interface' in data['interfaces-state']:
        data = data['interfaces-state']['interface']
    else:
        data = None
    return data


def get_ietf_interfaces_by_sub_interface(nc, name):
    """ Get an ietf-interface entry by BBF lower layer sub-interface name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name used for the interface that appears in the interface field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    interfaces = []
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-subif="urn:bbf:yang:bbf-sub-interfaces"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces>
                          <if:interface>
                            <bbf-subif:subif-lower-layer>
                              <bbf-subif:interface>
                                {name}
                              </bbf-subif:interface>
                            </bbf-subif:subif-lower-layer>
                          </if:interface>
                        </if:interfaces>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces/if:interface/bbf-subif:subif-lower-layer[interface={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'interface' in data['interfaces']:
        if isinstance(data['interfaces']['interface'], list):
            for interface in data['interfaces']['interface']:
                interfaces.append(interface)
        else:
            interfaces.append(data['interfaces']['interface'])
    return interfaces


def get_ietf_hardware_names(nc):
    """  Get a list of all ietf-hardware entry names

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """

    data_xml=f'''<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name/>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                </rpc>'''
    xml = nc.get(data_xml=data_xml, message="nc:data/hw:hardware/hw:component/hw:name")
    hardware_names = []
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        for name in data['hardware']['component']:
            hardware_names.append(name['name'])
    return hardware_names


def get_ietf_hardware_by_name(nc, name):
    """ Get an ietf-hardware entry by name

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The detected name of the desired hardware component

    Returns:
        dict: It returns a python dict representation of the xml data.

    """

    data_xml=f''' <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:bbf-hw-xcvr="urn:bbf:yang:bbf-hardware-transceivers"
                                     xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name>{name}</hw:name>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                 </rpc>'''
    xml = nc.get(data_xml=data_xml, message=f"/hw:hardware/hw:component[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        data = data['hardware']['component']
    else:
        data = None
    return data


def get_bbf_hardware_rssi_by_onu(nc, ssn):
    """ Get a bbf-hardware transceiver RSSI value by ONU Serial Number

    Args:
        nc (obj): Netconf connection to use for this request.
        ssn (str): The detected serial number of the desired hardware component
        located under /bbf-hw-xcvr:transceiver-link/bbf-hw-xcvr:diagnostics/
        /bbf-hw-xcvr-xpon:rssi-onu

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f''' <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:bbf-hw-xcvr="urn:bbf:yang:bbf-hardware-transceivers"
                                     xmlns:bbf-hw-xcvr-xpon="urn:bbf:yang:bbf-hardware-transceivers-xpon"
                                     xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <bbf-hw-xcvr:transceiver-link>
                              <bbf-hw-xcvr:diagnostics>
                                <bbf-hw-xcvr-xpon:rssi-onu>
                                  <bbf-hw-xcvr-xpon:detected-serial-number>
                                    {ssn}
                                  </bbf-hw-xcvr-xpon:detected-serial-number>
                                </bbf-hw-xcvr-xpon:rssi-onu>
                              </bbf-hw-xcvr:diagnostics>
                            </bbf-hw-xcvr:transceiver-link>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                 </rpc>'''
    xml = nc.get(data_xml=data_xml,
                 message=f"/hw:hardware/hw:component/bbf-hw-xcvr:transceiver-link/bbf-hw-xcvr:diagnostics"
                    f"/bbf-hw-xcvr-xpon:rssi-onu[detected-serial-number={ssn}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware'] \
        and 'transceiver-link' in data['hardware']['component'] \
        and 'diagnostics' in  data['hardware']['component']['transceiver-link'] \
        and 'rssi-onu' in data['hardware']['component']['transceiver-link']['diagnostics']:
        data = data['hardware']['component']['transceiver-link']['diagnostics']['rssi-onu']
    else:
        data = None
    return data


def get_bbf_oltvenet_names_by_vani(nc, name):
    """ Get a list of OLT V-ENET interface names by V-ANI interface name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The field value of the lower-layer-interface for the desired
        olt-v-enet interface

    Returns:
        list: It returns a list of oltvenet names.

    """
    oltvenet_names = []
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-xponvani="urn:bbf:yang:bbf-xponvani"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces>
                          <if:interface>
                            <bbf-xponvani:olt-v-enet>
                                <bbf-xponvani:lower-layer-interface>
                                    {name}
                                </bbf-xponvani:lower-layer-interface>
                             </bbf-xponvani:olt-v-enet>
                          </if:interface>
                        </if:interfaces>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message="/if:interfaces-state/if:interface/if:name")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'interface' in data['interfaces']:
        if isinstance(data['interfaces']['interface'], list):
            for interface in data['interfaces']['interface']:
                oltvenet_names.append(interface['name'])
        else:
            oltvenet_names.append(data['interfaces']['interface']['name'])
    return oltvenet_names


def get_bbf_link_table_by_to_interface(nc, to_interface_name):
    """ Get is BBF YANG link-table entry by the to-interface.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The to-interface field value of the desired link table entry

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    from_interface_name = None
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:bbf-lt="urn:bbf:yang:bbf-link-table"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <bbf-lt:link-table>
                          <bbf-lt:link-table>
                            <bbf-lt:to-interface>
                                {to_interface_name}
                            </bbf-lt:to-interface>
                          </bbf-lt:link-table>
                        </bbf-lt:link-table>
                      </filter>
                  </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/bbf-lt:link-table/bbf-lt:link-table[to-interface={to_interface_name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'link-table' in data and 'link-table' in data['link-table']:
        from_interface_name = data['link-table']['link-table']['from-interface']
    return from_interface_name


def get_tibit_bbf_olt_interface_map_by_channel_termination(nc, channel_termination_ref):
    """ Get a Tibit YANG BBF OLT interface mapping entry by PON Channel Termination reference.

    Args:
        nc (obj): Netconf connection to use for this request.
        device_id (str): The device-id used for the interface that appears in the device-id field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                <get>
                  <filter type="subtree">
                    <tibit-bbf-if:interfaces xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces">
                      <tibit-bbf-if:olt-interface-map>
                        <tibit-bbf-if:olt>
                          <tibit-bbf-if:pon>
                            <channel-termination-ref>{channel_termination_ref}</channel-termination-ref>
                          </tibit-bbf-if:pon>
                        </tibit-bbf-if:olt>
                      </tibit-bbf-if:olt-interface-map>
                    </tibit-bbf-if:interfaces>
                  </filter>
                </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map/tibit-bbf-if:pon[channel-termination-ref={channel_termination_ref}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'olt-interface-map' in data['interfaces'] and 'olt' in data['interfaces']['olt-interface-map']:
        data = data['interfaces']['olt-interface-map']['olt']
    else:
        data = None
    return data


def get_bbf_tconts_by_vani(nc, name):
    """ Get a list of TCONTs by V-ANI interface name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The interface-reference field value of the desired tcont entry
        (corresponds to the vani name)

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    tconts = []
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                   <get>
                       <filter type="subtree">
                        <bbf-xpongemtcont:xpongemtcont xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont">
                          <bbf-xpongemtcont:tconts>
                            <bbf-xpongemtcont:tcont>
                             <bbf-xpongemtcont:interface-reference>
                                {name}
                             </bbf-xpongemtcont:interface-reference>
                            </bbf-xpongemtcont:tcont>
                          </bbf-xpongemtcont:tconts>
                        </bbf-xpongemtcont:xpongemtcont>
                      </filter>
                      </get>
                    </rpc>"""
    xml = nc.get(data_xml=data_xml, message="f/bbf-xpongemtcont:xpongemtcont/bbf-xpongemtcont:tconts/bbf-xpongemtcont:tcont[interface-reference={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'xpongemtcont' in data and 'tconts' in data['xpongemtcont'] and 'tcont' in data['xpongemtcont']['tconts']:
        if isinstance(data['xpongemtcont']['tconts']['tcont'], list):
            for tcont in data['xpongemtcont']['tconts']['tcont']:
                tconts.append(tcont)
        else:
            tconts.append(data['xpongemtcont']['tconts']['tcont'])
    return tconts


def get_bbf_tcont_state_by_name(nc, name):
    """ Get a tcont-state entry by name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The itcont-ref field value of the desired tcont entry
        (corresponds to a gemport name)

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                   <get>
                      <filter type="subtree">
                        <bbf-xpongemtcont:xpongemtcont-state xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont">
                          <bbf-xpongemtcont:tconts>
                            <bbf-xpongemtcont:tcont>
                              <bbf-xpongemtcont:name>
                                {name}
                              </bbf-xpongemtcont:name>
                            </bbf-xpongemtcont:tcont>
                          </bbf-xpongemtcont:tconts>
                        </bbf-xpongemtcont:xpongemtcont-state>
                      </filter>
                    </get>
                    </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/bbf-xpongemtcont:xpongemtcont-state/bbf-xpongemtcont:tconts/bbf-xpongemtcont:tcont[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'xpongemtcont-state' in data and 'tconts' in data['xpongemtcont-state'] and 'tcont' in data['xpongemtcont-state']['tconts']:
        data = data['xpongemtcont-state']['tconts']['tcont']
    else:
        data = None
    return data


def get_bbf_gemports_by_tcont(nc, name):
    """ Get a gemport entry by TCONT reference.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The itcont-ref field value of the desired tcont entry
        (corresponds to a gemport name)

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    gemports = []
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                   <get>
                      <filter type="subtree">
                        <bbf-xpongemtcont:xpongemtcont xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont">
                          <bbf-xpongemtcont:gemports>
                            <bbf-xpongemtcont:gemport>
                              <bbf-xpongemtcont:tcont-ref>
                                {name}
                              </bbf-xpongemtcont:tcont-ref>
                            </bbf-xpongemtcont:gemport>
                          </bbf-xpongemtcont:gemports>
                        </bbf-xpongemtcont:xpongemtcont>
                      </filter>
                    </get>
                    </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/bbf-xpongemtcont:xpongemtcont/bbf-xpongemtcont:gemports/bbf-xpongemtcont:gemport[tcont-reference={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'xpongemtcont' in data and 'gemports' in data['xpongemtcont'] and 'gemport' in data['xpongemtcont']['gemports']:
        if isinstance(data['xpongemtcont']['gemports']['gemport'], list):
            for gemport in data['xpongemtcont']['gemports']['gemport']:
                gemports.append(gemport)
        else:
            gemports.append(data['xpongemtcont']['gemports']['gemport'])
    return gemports


def get_bbf_gemport_state_by_name(nc, name):
    """ Get a gemport-state entry by GemPort name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The itcont-ref field value of the desired tcont entry
        (corresponds to a gemport name)

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                   <get>
                      <filter type="subtree">
                        <bbf-xpongemtcont:xpongemtcont-state xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont">
                          <bbf-xpongemtcont:gemports>
                            <bbf-xpongemtcont:gemport>
                              <bbf-xpongemtcont:name>
                                {name}
                              </bbf-xpongemtcont:name>
                            </bbf-xpongemtcont:gemport>
                          </bbf-xpongemtcont:gemports>
                        </bbf-xpongemtcont:xpongemtcont-state>
                      </filter>
                    </get>
                    </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/bbf-xpongemtcont:xpongemtcont-state/bbf-xpongemtcont:gemports/bbf-xpongemtcont:gemport[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'xpongemtcont-state' in data and 'gemports' in data['xpongemtcont-state'] and 'gemport' in data['xpongemtcont-state']['gemports']:
        data = data['xpongemtcont-state']['gemports']['gemport']
    else:
        data = None
    return data


def get_bbf_forwarder_name_by_vlan_sub_interface(nc, name):
    """ Get a BBF L2 Forwarder name by the VLAN Sub-interface reference in the
    forwarder's port list.

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    message-id="">
                  <get>
                    <filter type="subtree">
                      <bbf-l2-fwd:forwarding xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding">
                        <bbf-l2-fwd:forwarders>
                          <bbf-l2-fwd:forwarder>
                            <bbf-l2-fwd:ports>
                              <bbf-l2-fwd:port>
                                <bbf-l2-fwd:sub-interface>
                                  {name}
                                </bbf-l2-fwd:sub-interface>
                              </bbf-l2-fwd:port>
                            </bbf-l2-fwd:ports>
                          </bbf-l2-fwd:forwarder>
                        </bbf-l2-fwd:forwarders>
                      </bbf-l2-fwd:forwarding>
                    </filter>
                  </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/bbf-l2-fwd:forwarding/bbf-l2-fwd:forwarders/bbf-l2-fwd:forwarder/bbf-l2-fwd:ports/bbf-l2-fwd:port[sub-interface={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'forwarding' in data and 'forwarders' in data['forwarding'] and 'forwarder' in data['forwarding']['forwarders']:
        data = data['forwarding']['forwarders']['forwarder']['name']
    else:
        data = None
    return data


def get_bbf_forwarder_by_name(nc, name):
    """ Get a BBF L2 Forwarder entry by name.

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    message-id="">
                  <get>
                    <filter type="subtree">
                      <bbf-l2-fwd:forwarding xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding">
                        <bbf-l2-fwd:forwarders>
                          <bbf-l2-fwd:forwarder>
                            <bbf-l2-fwd:name>
                              {name}
                            </bbf-l2-fwd:name>
                          </bbf-l2-fwd:forwarder>
                        </bbf-l2-fwd:forwarders>
                      </bbf-l2-fwd:forwarding>
                    </filter>
                  </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/bbf-l2-fwd:forwarding/bbf-l2-fwd:forwarders/bbf-l2-fwd:forwarder[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'forwarding' in data and 'forwarders' in data['forwarding'] and 'forwarder' in data['forwarding']['forwarders']:
        data = data['forwarding']['forwarders']['forwarder']
    else:
        data = None
    return data


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False,
                        help="NETCONF Server IP address or hostname.")
    parser.add_argument("--onu", action="store", dest="onu", default=0, required=True, help="onu device id")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False,
                        help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False,
                        help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False,
                        help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Create a connection to the Netconf server.
    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Initialize variables
    onu_tag = None
    olt_tag = None
    uni_oper_statuses = []
    uni_speeds = []
    ports = []

    # First, attempt to look up the ONU by Serial Number
    # Note: the args.onu argument can be a V-ANI name or ONU Serial number
    vani_interface = get_ietf_interface_by_vani_ssn(nc, args.onu)
    if not vani_interface:
        # If the lookup by SSN returns none, assume args.onu contains a V-ANI name.
        vani_interface = get_ietf_interface_by_name(nc, name=args.onu, if_type="bbf-xponift:v-ani")
    # If the value is neither the SSN nor the V-ANI name, exit with a failure
    if not vani_interface:
        print(f"ERROR: ONU {args.onu} no found.")
        sys.exit(1)

    # Save the V-ANI name and ONU Serial Number for later processing
    vani_name = vani_interface['name']
    onu_ssn = vani_interface['v-ani']['expected-serial-number']

    # Get the interface-state for the V-ANI interface
    vani_interface_st = get_ietf_interface_state_by_name(nc, vani_name, with_stats=True)
    if vani_interface_st and 'v-ani' in vani_interface_st:
        # Flatten v-ani info into the vani_interface_st for easier access
        vani_interface_st['onu-id'] = vani_interface_st['v-ani']['onu-id']
        vani_interface_st['detected-serial-number'] = vani_interface_st['v-ani']['onu-present-on-this-olt']['detected-serial-number']
        vani_interface_st['detected-registration-id'] = vani_interface_st['v-ani']['onu-present-on-this-olt']['detected-registration-id']
        vani_interface_st['onu-present-on-this-channel-pair'] = vani_interface_st['v-ani']['onu-present-on-this-olt']['onu-present-on-this-channel-pair']
        vani_interface_st['onu-present-on-this-channel-termination'] = vani_interface_st['v-ani']['onu-present-on-this-olt']['onu-present-on-this-channel-termination']
        # Convert meters to kilometers
        vani_interface_st['onu-fiber-distance'] = float(vani_interface_st['v-ani']['onu-present-on-this-olt']['onu-fiber-distance']) / 1000.0
        # Strip the name space from the presense state
        vani_interface_st['onu-presence-state'] = vani_interface_st['v-ani']['onu-presence-state'].split(":")[1]
    if not vani_interface_st:
        # Entry is missing for this device, fill in a blank entry for display below
        vani_interface_st = {
            'oper-status': "",
            'last-change': "",
            'phys-address': "",
            'onu-id': "",
            'detected-serial-number': "",
            'detected-registration-id': "",
            'onu-present-on-this-channel-pair': "",
            'onu-present-on-this-channel-termination': "",
            'onu-fiber-distance': "",
            'onu-presence-state': ""
        }
    if vani_interface_st and 'performance' in vani_interface_st and 'intervals-15min' in vani_interface_st['performance'] and 'current' in vani_interface_st['performance']['intervals-15min']:
        # Flatten performance info into the vani_interface_st for easier access
        vani_interface_st['tx-optical-power-level'] = float(vani_interface_st['performance']['intervals-15min']['current']['xpon']['power-monitoring']["tx-optical-power-level"])
        # Convert from 0.1 dbm to dbm
        vani_interface_st['tx-optical-power-level'] /= 10.0
    else:
        vani_interface_st['tx-optical-power-level'] = 0.0

    # Get the ietf-hardware entry for this ONU
    hardware_interface = get_ietf_hardware_by_name(nc, f"XGS-PON ONU {onu_ssn}")
    # Update the values used in the final output
    if hardware_interface:
        # Convert from 0.1 mW to mW
        if 'transceiver-link' in hardware_interface:
            tx_power_mW = float(hardware_interface['transceiver-link']['diagnostics']['tx-power']) / 10.0
            hardware_interface['tx-power'] = float(mw_to_dbm(tx_power_mW))
            # Convert from 0.1 mW to mW
            rx_power_mW = float(hardware_interface['transceiver-link']['diagnostics']['rx-power']) / 10.0
            hardware_interface['rx-power'] = float(mw_to_dbm(rx_power_mW))
        else:
            hardware_interface['tx-power'] = 0.0
            hardware_interface['rx-power'] = 0.0
    else:
        # Entry is missing for this device, fill in a blank entry for display below
        hardware_interface = {
            'description': "",
            'hardware-rev': "",
            'firmware-rev': "",
            'serial-num': "",
            'mfg-name': "",
            'model-name': "",
            'tx-power': 0.0,
            'rx-power': 0.0
        }

    # Get the bbf-hardware RSSI entry for this ONU
    hardware_rssi = get_bbf_hardware_rssi_by_onu(nc, onu_ssn)
    if hardware_rssi and 'rssi' in hardware_rssi:
        # Convert RSSI value to a float
        hardware_rssi['rssi'] = float(hardware_rssi['rssi'])
        # Convert from 0.1 dbm to dbm
        hardware_rssi['rssi'] /= 10.0
    else:
        hardware_rssi = {
            'rssi': 0.0
        }


    #
    # Gather OLT Information
    #

    # Get the Tibit OLT BBF Interface map for this ONU
    olt_interface_map = None
    if vani_interface_st and vani_interface_st['onu-present-on-this-channel-termination']:
        olt_interface_map = get_tibit_bbf_olt_interface_map_by_channel_termination(
            nc,
            vani_interface_st['onu-present-on-this-channel-termination']
        )

    # Get the OLT MAC Address
    olt_mac = ""
    if olt_interface_map:
        olt_mac = olt_interface_map['device-id']

    # Get the OLT Switch Port
    olt_switch_port = ""
    hardware_list = get_ietf_hardware_names(nc)
    for name in hardware_list:
        if "OLT" in name and olt_mac in name:
            # Parse the switch port out of the name
            olt_switch_port = name.split(" ")[3].strip("[]")

    #
    # Gather BBF Interface and Link Table information
    #

    # The bbf-link-table information is gathered as interfaces are mapped
    bbf_link_table = {}
    ani_name = get_bbf_link_table_by_to_interface(nc, vani_name)
    if ani_name:
        # Update the bbf-link-table with the ani -> v-ani mapping
        bbf_link_table[ani_name] = vani_name
    else:
        ani_name = ""
    oltvenet_names = get_bbf_oltvenet_names_by_vani(nc, vani_name)

    #
    # Gather ONU UNI Interface information
    #

    # Build a list of UNIs configured on this ONU based on the olt v-enets and the link table.
    uni_ports = {}
    for olt_venet_name in oltvenet_names:
        uni_name = get_bbf_link_table_by_to_interface(nc, olt_venet_name)
        if not uni_name:
            continue
        # Update the bbf-link-table with the olt-v-enet -> uni mapping
        bbf_link_table[olt_venet_name] = uni_name
        uni_port = get_ietf_interface_state_by_name(nc, uni_name)
        if uni_port:
            uni_port_num = parse_port_number_from_name(uni_port['name'])
            uni_ports[uni_port_num] = uni_port

    #
    # Gather OLT-Service Port information
    #
    olt_service_ports = {}
    tconts = get_bbf_tconts_by_vani(nc, vani_name)
    for tcont in tconts:
        olt_service_port_num = parse_port_number_from_name(tcont['name']) - 1
        olt_service_port = {
            'tcont': tcont['name'],
            'sla': tcont['traffic-descriptor-profile-ref']
        }

        # Get the Alloc ID from the TCONT state
        tcont_st = get_bbf_tcont_state_by_name(nc, tcont['name'])
        if tcont_st:
            olt_service_port['alloc-id'] = tcont_st['actual-alloc-id']
        else:
            olt_service_port['alloc-id'] = ""

        # Get the GemPort for this TCONT
        # Note: only handle a single gemport per tcont for now
        gemports = get_bbf_gemports_by_tcont(nc, tcont['name'])
        if gemports:
            olt_service_port['gemport'] = gemports[0]['name']
        else:
            olt_service_port['gemport'] = ""

        # Get the GemPort ID for this Service
        # Note: only handle a single gemport per tcont for now
        if gemports:
            gemport_st = get_bbf_gemport_state_by_name(nc, gemports[0]['name'])
            if gemport_st:
                olt_service_port['gemport-id'] = gemport_st['actual-gemport-id']
            else:
                olt_service_port['gemport-id'] = ""
        else:
            olt_service_port['gemport-id'] = ""

        # Get the PON-side VLAN Sub-interface from the olt-v-enet
        # NOTE: Currently the script assumes one olt-v-enet per ONU. This
        # will be enhanced in future versions of the script
        vlan_sub_interfaces = None
        if oltvenet_names:
            vlan_sub_interfaces = get_ietf_interfaces_by_sub_interface(nc, oltvenet_names[0])

        # Get the Forwarder name from the VLAN Sub-interface
        # Note: only handle a single vlan sub interface for now
        forwarder_name = None
        if vlan_sub_interfaces:
            forwarder_name = get_bbf_forwarder_name_by_vlan_sub_interface(nc, vlan_sub_interfaces[0]['name'])

        # Get the Forwarder
        forwarder = None
        if forwarder_name:
            forwarder = get_bbf_forwarder_by_name(nc, forwarder_name)

        # Find the NNI-side forwarder port and vlan-sub-interface, and parse
        # the VLAN Tags from the NNI-side VLAN Sub-interface
        olt_tag = ""
        onu_tag = ""
        if forwarder:
            # Get the list of forwarder ports
            if 'ports' in forwarder and 'port' in forwarder['ports']:
                if not isinstance(forwarder['ports']['port'], list):
                    forwarder_ports = [forwarder['ports']['port']]
                else:
                    forwarder_ports = forwarder['ports']['port']
            else:
                forwarder_ports = []

            # Gather VLAN tagging information from NNI-side forwarder ports
            for port in forwarder_ports:
                # Get the VLAN sub-interface for this forwarder port
                vlan_sub_interface = get_ietf_interface_by_name(nc, port['sub-interface'])

                if not vlan_sub_interface:
                    continue

                # Determine if this is an NNI or PON-side vlan_sub_interface
                if 'interface-usage' in vlan_sub_interface and 'interface-usage' in vlan_sub_interface['interface-usage']:
                    interface_usage = vlan_sub_interface['interface-usage']['interface-usage']
                else:
                    # Assume user-port (i.e., PON-side if not specified)
                    interface_usage = 'user-port'

                # Only process NNI-side interfaces in this loop
                if interface_usage != 'network-port':
                    continue

                # VLAN Tags can be parsed from the NNI vlan_sub_interface
                olt_tag = "untagged"
                onu_tag = "untagged"
                vlan_tags = vlan_sub_interface['inline-frame-processing']['ingress-rule']['rule']['flexible-match']['match-criteria']['tag']
                if isinstance(vlan_tags, dict):
                    vlan_tags = [vlan_tags]
                for index, tag in enumerate(vlan_tags):
                    # Determine the TPID for this tag
                    if tag['dot1q-tag']['tag-type'] == 'bbf-dot1qt:s-vlan':
                        tpid = 's-vlan'
                    else:
                        tpid = 'c-vlan'

                    # Assume the outer vlan is the OLT tag and the inner vlan is the ONU tag
                    if index == 0:
                        olt_tag = tpid + "/" + str(tag['dot1q-tag']['vlan-id'])
                    else:
                        onu_tag = tpid + "/" + str(tag['dot1q-tag']['vlan-id'])

                # Only handle one NNI Port
                break

        # Set the OLT and ONU Tags
        olt_service_port['forwarder'] = forwarder_name
        olt_service_port['olt-tag'] = olt_tag
        olt_service_port['onu-tag'] = onu_tag

        # Add the OLT-Service Port to the list
        olt_service_ports[olt_service_port_num] = olt_service_port


    #
    # Display ONU Information
    #

    print("")
    print("Inventory Info")
    print(f"  ONU:                       {onu_ssn}")
    print(f"  MAC Address:               {vani_interface_st['phys-address'] if 'phys-address' in vani_interface_st else ''}")
    print(f"  description:               {hardware_interface['description']}")
    print(f"  hardware revision:         {hardware_interface['hardware-rev']}")
    print(f"  firmware revision:         {hardware_interface['firmware-rev']}")
    print(f"  serial number:             {hardware_interface['serial-num']}")
    print(f"  manufacturers name:        {hardware_interface['mfg-name']}")
    print(f"  model name:                {hardware_interface['model-name']}")
    print("")
    print("Device Status")
    print(f"  operational status:        {vani_interface_st['oper-status']}")
    print(f"  last change:               {vani_interface_st['last-change']}")
    print(f"  onu id:                    {vani_interface_st['onu-id']}")
    print(f"  detected serial number:    {vani_interface_st['detected-serial-number']}")
    print(f"  detected registration id:  {vani_interface_st['detected-registration-id']}")
    print(f"  onu presence state:        {vani_interface_st['onu-presence-state']}")
    print("")
    print("OLT Device Info")
    print(f"  olt mac:                   {olt_mac}")
    print(f"  switch port:               {olt_switch_port}")
    print(f"  channel pair:              {vani_interface_st['onu-present-on-this-channel-pair']}")
    print(f"  channel termination:       {vani_interface_st['onu-present-on-this-channel-termination']}")
    print(f"  fiber distance:            {vani_interface_st['onu-fiber-distance']} km")
    print("")
    print("Optical Levels")
    print("              {:>7}      {:>7}".format('TX', 'RX'))
    print("  olt:        {:7.2f} dBm  {:7.2f} dBm".format(vani_interface_st['tx-optical-power-level'], hardware_rssi['rssi']))
    print("  onu:        {:7.2f} dBm  {:7.2f} dBm".format(hardware_interface['tx-power'], hardware_interface['rx-power']))
    print("")
    print("BBF Interfaces")
    print(f"  ani:                       {ani_name}")
    print(f"  v-ani:                     {vani_name}")
    for name in oltvenet_names:
        print(f"  olt-v-enet:                {name}")
    print("")
    print("BBF Link Table")
    # Print table header row
    print("  "
            f"{'From': <32}"
            f"{'To'}")
    # Display ONU entries sorted by ONU ID
    for from_name, to_name in bbf_link_table.items():
        print("  "
                f"{from_name: <32}"
                f"{to_name}")
    print("")
    print("ONU UNI Ports")
    # Print table header row
    print("  "
            f"{'Port': <12}"
            f"{'IETF Name': <32}"
            f"{'Oper Status': <16}"
            f"{'Speed'}")
    # Display ONU entries sorted by ONU ID
    for uni_port_num, uni_port in uni_ports.items():
        print("  "
                f"UNI-ETH {uni_port_num: <4}"
                f"{uni_port['name']: <32}"
                f"{uni_port['oper-status']: <16}"
                f"{uni_port['speed']}")
    print("")
    # Display OLT Service Ports
    for olt_service_port_num, olt_service_port in olt_service_ports.items():
        print(f"OLT-Service {olt_service_port_num}")
        print(f"  tcont:        {olt_service_port['tcont']}")
        print(f"  alloc-id:     {olt_service_port['alloc-id']}")
        print(f"  sla:          {olt_service_port['sla']}")
        print(f"  gemport:      {olt_service_port['gemport']}")
        print(f"  gemport-id:   {olt_service_port['gemport-id']}")
        print(f"  forwarder:    {olt_service_port['forwarder']}")
        print(f"  olt-tag:      {olt_service_port['olt-tag']}")
        print(f"  onu-tag:      {olt_service_port['onu-tag']}")
        print("")

