#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import os
import sys
import json
from lxml import etree
import ncclient
from ncclient import manager
from netconf_driver import NetconfDriver


#
# ietf-interfaces
#

def netconf_get_ietf_interface_names(nc, name_match=None, if_type=None):
    ietf_interface_names = []

    # if an interface type was specified, filter on type.
    if if_type:
        if_type_filter = f"<if:type>{if_type}</if:type>"
    else:
        if_type_filter = ""

    # Send a Netconf <get> request to retreive the ietf-interface names
    IETF_INTERFACE_NAMES = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
        xmlns:ianaift="urn:ietf:params:xml:ns:yang:iana-if-type"
        xmlns:bbf-xponift="urn:bbf:yang:bbf-xpon-if-type"
        xmlns:bbfift="urn:bbf:yang:bbf-if-type"
        message-id="1">
        <get>
            <filter type="subtree">
                <if:interfaces>
                    <if:interface>
                        <if:name/>
                        {if_type_filter}
                    </if:interface>
                </if:interfaces>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=IETF_INTERFACE_NAMES,
        message="/if:interfaces/if:interface/if:name")

    # import pdb; pdb.set_trace()

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'if' : "urn:ietf:params:xml:ns:yang:ietf-interfaces",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/if:interfaces/if:interface/if:name", namespaces=NSMAP):
            name = name.text
            if not name_match or name_match in name:
                ietf_interface_names.append(name)

    return ietf_interface_names

def netconf_get_onus_by_olt_port(nc, olt_port):
    onus = []

    # Send a Netconf <get> request to retreive the ietf-interface names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
        xmlns:bbf-xponvani="urn:bbf:yang:bbf-xponvani"
        message-id="1">
        <get>
            <filter type="subtree">
                <if:interfaces>
                    <if:interface>
                        <bbf-xponvani:v-ani>
                            <bbf-xponvani:channel-partition>channelpartition.{olt_port}</bbf-xponvani:channel-partition>
                            <bbf-xponvani:expected-serial-number/>
                        </bbf-xponvani:v-ani>
                    </if:interface>
                </if:interfaces>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/if:interfaces/if:interface/bbf-xponvani:v-ani/bbf-xponvani:expected-serial-number")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'if' : "urn:ietf:params:xml:ns:yang:ietf-interfaces",
            'bbf-xponvani' : "urn:bbf:yang:bbf-xponvani",
            }
        root = etree.fromstring(rsp_xml)
        for onu in root.findall(f"nc:data/if:interfaces/if:interface/bbf-xponvani:v-ani/bbf-xponvani:expected-serial-number", namespaces=NSMAP):
            onus.append(onu.text)

    return onus

def netconf_remove_ietf_interface_all(nc):
    nc.edit_config(
        message=f"Removing all from ietf-interface.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <if:interfaces nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_ietf_interface_by_name(nc, name):
    nc.edit_config(
        message=f"Removing ietf-interface {name}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <if:interfaces>
                        <if:interface nc:operation="remove">
                            <if:name>{name}</if:name>
                        </if:interface>
                    </if:interfaces>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_bbf_channel_group_pon_pools_by_name(nc, name):
    nc.edit_config(
        message=f"Removing bbf-xpon pon-pools for channel-group {name}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
            xmlns:bbf-xpon="urn:bbf:yang:bbf-xpon"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <if:interfaces>
                        <if:interface>
                            <if:name>{name}</if:name>
                            <bbf-xpon:channel-group>
                                <bbf-xpon:pon-pools>
                                    <bbf-xpon:pon-pool>
                                        <bbf-xpon:name>pool1</bbf-xpon:name>
                                        <bbf-xpon:channel-termination-ref nc:operation="remove"/>
                                    </bbf-xpon:pon-pool>
                                </bbf-xpon:pon-pools>
                            </bbf-xpon:channel-group>
                        </if:interface>
                    </if:interfaces>
                </config>
            </edit-config>
        </rpc>
        '''
    )


#
# bbf-l2-forwarding
#

def netconf_get_bbf_l2_forwarding_names(nc, name_match=None):
    names = []

    # Send a Netconf <get> request to retreive the names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding"
        message-id="1">
        <get>
            <filter type="subtree">
                <bbf-l2-fwd:forwarding>
                    <bbf-l2-fwd:forwarders>
                        <bbf-l2-fwd:forwarder>
                            <bbf-l2-fwd:name/>
                        </bbf-l2-fwd:forwarder>
                    </bbf-l2-fwd:forwarders>
                </bbf-l2-fwd:forwarding>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/bbf-l2-fwd:forwarding/bbf-l2-fwd:forwarders/bbf-l2-fwd:forwarder/bbf-l2-fwd:name")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'bbf-l2-fwd' : "urn:bbf:yang:bbf-l2-forwarding",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/bbf-l2-fwd:forwarding/bbf-l2-fwd:forwarders/bbf-l2-fwd:forwarder/bbf-l2-fwd:name", namespaces=NSMAP):
            name = name.text
            if not name_match or name_match in name:
                names.append(name)

    return names

def netconf_get_bbf_l2_forwarding_vlan_interface_names_from_forwarder(nc, forwarder_name):
    names = []

    # Send a Netconf <get> request to retreive the names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding"
        message-id="1">
        <get>
            <filter type="subtree">
                <bbf-l2-fwd:forwarding>
                    <bbf-l2-fwd:forwarders>
                        <bbf-l2-fwd:forwarder>
                            <bbf-l2-fwd:name>{forwarder_name}</bbf-l2-fwd:name>
                            <bbf-l2-fwd:ports>
                                <bbf-l2-fwd:port>
                                    <bbf-l2-fwd:sub-interface/>
                                </bbf-l2-fwd:port>
                            </bbf-l2-fwd:ports>
                        </bbf-l2-fwd:forwarder>
                    </bbf-l2-fwd:forwarders>
                </bbf-l2-fwd:forwarding>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/bbf-l2-fwd:forwarding/bbf-l2-fwd:forwarders/bbf-l2-fwd:forwarder/bbf-l2-fwd:ports/bbf-l2-fwd:port/bbf-l2-fwd:sub-interface")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'bbf-l2-fwd' : "urn:bbf:yang:bbf-l2-forwarding",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/bbf-l2-fwd:forwarding/bbf-l2-fwd:forwarders/bbf-l2-fwd:forwarder/bbf-l2-fwd:ports/bbf-l2-fwd:port/bbf-l2-fwd:sub-interface", namespaces=NSMAP):
            names.append(name.text)

    return names

def netconf_remove_bbf_l2_forwarding_all(nc):
    nc.edit_config(
        message="Removing all from bbf-l2-forwarding.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding"
            message-id="34566760">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-l2-fwd:forwarding nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_bbf_l2_forwarding_by_name(nc, name):
    nc.edit_config(
        message=f"Removing bbf-l2-forwarding forwarder {name}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-l2-fwd:forwarding>
                        <bbf-l2-fwd:forwarders>
                            <bbf-l2-fwd:forwarder nc:operation="remove">
                                <bbf-l2-fwd:name>{name}</bbf-l2-fwd:name>
                            </bbf-l2-fwd:forwarder>
                        </bbf-l2-fwd:forwarders>
                    </bbf-l2-fwd:forwarding>
                </config>
            </edit-config>
        </rpc>
        '''
    )


#
# bbf-xpongemtcont tconts
#

def netconf_get_bbf_xpongemtcont_tcont_names(nc, name_match=None):
    names = []

    # Send a Netconf <get> request to retreive the names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont"
        message-id="1">
        <get>
            <filter type="subtree">
                <bbf-xpongemtcont:xpongemtcont>
                    <bbf-xpongemtcont:tconts>
                        <bbf-xpongemtcont:tcont>
                            <bbf-xpongemtcont:name/>
                        </bbf-xpongemtcont:tcont>
                    </bbf-xpongemtcont:tconts>
                </bbf-xpongemtcont:xpongemtcont>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/bbf-xpongemtcont:xpongemtcont/bbf-xpongemtcont:tconts/bbf-xpongemtcont:tcont/bbf-xpongemtcont:name")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'bbf-xpongemtcont' : "urn:bbf:yang:bbf-xpongemtcont",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/bbf-xpongemtcont:xpongemtcont/bbf-xpongemtcont:tconts/bbf-xpongemtcont:tcont/bbf-xpongemtcont:name", namespaces=NSMAP):
            name = name.text
            if not name_match or name_match in name:
                names.append(name)

    return names

def netconf_remove_bbf_xpongemtcont_tcont_all(nc):
    nc.edit_config(
        message="Removing all from bbf-xpongemtcont tconts.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont"
            message-id="34566760">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-xpongemtcont:xpongemtcont>
                        <bbf-xpongemtcont:tconts nc:operation="replace"/>
                    </bbf-xpongemtcont:xpongemtcont>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_bbf_xpongemtcont_tcont_by_name(nc, name):
    nc.edit_config(
        message=f"Removing bbf-xpongemtcont tcont {name}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-xpongemtcont:xpongemtcont>
                        <bbf-xpongemtcont:tconts>
                            <bbf-xpongemtcont:tcont nc:operation="remove">
                                <bbf-xpongemtcont:name>{name}</bbf-xpongemtcont:name>
                            </bbf-xpongemtcont:tcont>
                        </bbf-xpongemtcont:tconts>
                    </bbf-xpongemtcont:xpongemtcont>
                </config>
            </edit-config>
        </rpc>
        '''
    )


#
# bbf-xpongemtcont gemports
#

def netconf_get_bbf_xpongemtcont_gemport_names(nc, name_match=None):
    names = []

    # Send a Netconf <get> request to retreive the names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont"
        message-id="1">
        <get>
            <filter type="subtree">
                <bbf-xpongemtcont:xpongemtcont>
                    <bbf-xpongemtcont:gemports>
                        <bbf-xpongemtcont:gemport>
                            <bbf-xpongemtcont:name/>
                        </bbf-xpongemtcont:gemport>
                    </bbf-xpongemtcont:gemports>
                </bbf-xpongemtcont:xpongemtcont>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/bbf-xpongemtcont:xpongemtcont/bbf-xpongemtcont:gemports/bbf-xpongemtcont:gemport/bbf-xpongemtcont:name")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'bbf-xpongemtcont' : "urn:bbf:yang:bbf-xpongemtcont",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/bbf-xpongemtcont:xpongemtcont/bbf-xpongemtcont:gemports/bbf-xpongemtcont:gemport/bbf-xpongemtcont:name", namespaces=NSMAP):
            name = name.text
            if not name_match or name_match in name:
                names.append(name)

    return names

def netconf_remove_bbf_xpongemtcont_gemport_all(nc):
    nc.edit_config(
        message="Removing all from bbf-xpongemtcont gemports.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont"
            message-id="34566760">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-xpongemtcont:xpongemtcont>
                        <bbf-xpongemtcont:gemports nc:operation="replace"/>
                    </bbf-xpongemtcont:xpongemtcont>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_bbf_xpongemtcont_gemport_by_name(nc, name):
    nc.edit_config(
        message=f"Removing bbf-xpongemtcont gemports {name}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-xpongemtcont:xpongemtcont>
                        <bbf-xpongemtcont:gemports>
                            <bbf-xpongemtcont:gemport nc:operation="remove">
                                <bbf-xpongemtcont:name>{name}</bbf-xpongemtcont:name>
                            </bbf-xpongemtcont:gemport>
                        </bbf-xpongemtcont:gemports>
                    </bbf-xpongemtcont:xpongemtcont>
                </config>
            </edit-config>
        </rpc>
        '''
    )


#
# bbf-link-table
#

def netconf_get_bbf_link_table_names(nc, name_match=None):
    names = []

    # Send a Netconf <get> request to retreive the names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:bbf-lt="urn:bbf:yang:bbf-link-table"
        message-id="1">
        <get>
            <filter type="subtree">
                <bbf-lt:link-table>
                    <bbf-lt:link-table>
                        <bbf-lt:from-interface/>
                    </bbf-lt:link-table>
                </bbf-lt:link-table>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/bbf-lt:link-table/bbf-lt:link-table/bbf-lt:from-interface")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'bbf-lt' : "urn:bbf:yang:bbf-link-table",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/bbf-lt:link-table/bbf-lt:link-table/bbf-lt:from-interface", namespaces=NSMAP):
            name = name.text
            if not name_match or name_match in name:
                names.append(name)

    return names

def netconf_remove_bbf_link_table_all(nc):
    nc.edit_config(
        message="Removing all from bbf-link-table.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-lt="urn:bbf:yang:bbf-link-table"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-lt:link-table nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_bbf_link_table_by_name(nc, name):
    nc.edit_config(
        message=f"Removing bbf-link-table entry {name}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-lt="urn:bbf:yang:bbf-link-table"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-lt:link-table>
                        <bbf-lt:link-table nc:operation="remove">
                            <bbf-lt:from-interface>{name}</bbf-lt:from-interface>
                        </bbf-lt:link-table>
                    </bbf-lt:link-table>
                </config>
            </edit-config>
        </rpc>
        '''
    )


#
# tibit-bbf-interfaces
#

def netconf_get_tibit_bbf_interface_olts(nc, olt_port):
    olts = []

    # Send a Netconf <get> request to retreive the names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces"
        message-id="1">
        <get>
            <filter type="subtree">
                <tibit-bbf-if:interfaces>
                    <tibit-bbf-if:olt-interface-map>
                        <tibit-bbf-if:olt>
                            <tibit-bbf-if:pon>
                                <tibit-bbf-if:channel-partition-ref>channelpartition.{olt_port}</tibit-bbf-if:channel-partition-ref>
                            </tibit-bbf-if:pon>
                        </tibit-bbf-if:olt>
                    </tibit-bbf-if:olt-interface-map>
                </tibit-bbf-if:interfaces>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map/tibit-bbf-if:olt/tibit-bbf-if:device-id")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'tibit-bbf-if' : "urn:com:tibitcom:ns:yang:bbf:interfaces",
            }
        root = etree.fromstring(rsp_xml)
        for olt in root.findall(f"nc:data/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map/tibit-bbf-if:olt/tibit-bbf-if:device-id", namespaces=NSMAP):
            olts.append(olt.text)

    return olts

def netconf_get_olt_port_from_mac_addr(nc, mac_addr):
    olt_port = None

    # Send a Netconf <get> request to retreive the ietf-interface names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces"
        message-id="1">
        <get>
            <filter type="subtree">
                <tibit-bbf-if:interfaces xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces">
                    <tibit-bbf-if:olt-interface-map>
                        <tibit-bbf-if:olt>
                            <tibit-bbf-if:device-id>{mac_addr}</tibit-bbf-if:device-id>
                        </tibit-bbf-if:olt>
                    </tibit-bbf-if:olt-interface-map>
                </tibit-bbf-if:interfaces>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map/tibit-bbf-if:olt/tibit-bbf-if:pon/tibit-bbf-if:channel-partition-ref")

    # import pdb; pdb.set_trace()

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'tibit-bbf-if' : "urn:com:tibitcom:ns:yang:bbf:interfaces",
            }
        root = etree.fromstring(rsp_xml)
        channel_partition = root.find(f"nc:data/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map/tibit-bbf-if:olt/tibit-bbf-if:pon/tibit-bbf-if:channel-partition-ref", namespaces=NSMAP)
        if channel_partition is not None:
            channel_partition = channel_partition.text
            if '.' in channel_partition:
                olt_port = '.'.join(channel_partition.split(".")[1:])

    return olt_port

def netconf_remove_tibit_bbf_interface_all(nc):
    nc.edit_config(
        message=f"Removing all from tibit-bbf-interfaces.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <tibit-bbf-if:interfaces nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_tibit_bbf_interface_by_olt(nc, mac_addr):
    nc.edit_config(
        message=f"Removing tibit-bbf-interface olt {mac_addr}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <tibit-bbf-if:interfaces>
                        <tibit-bbf-if:olt-interface-map>
                            <tibit-bbf-if:olt nc:operation="remove">
                                <tibit-bbf-if:device-id>{mac_addr}</tibit-bbf-if:device-id>
                            </tibit-bbf-if:olt>
                        </tibit-bbf-if:olt-interface-map>
                    </tibit-bbf-if:interfaces>
                </config>
            </edit-config>
        </rpc>
        '''
    )



#
# bbf-l2-dhcpv4-relay profiles
#

def netconf_remove_bbf_l2_dhcpv4_relay_profiles_all(nc):
    nc.edit_config(
        message="Removing all from bbf-l2-dhcpv4-relay profiles.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-l2-d4r="urn:bbf:yang:bbf-l2-dhcpv4-relay"
            message-id="34566760">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-l2-d4r:l2-dhcpv4-relay-profiles nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )



#
# bbf-qos-policies
#

def netconf_remove_bbf_qos_policy_profiles_all(nc):
    nc.edit_config(
        message="Removing all from bbf-qos-policies.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-qos-pol="urn:bbf:yang:bbf-qos-policies"
            message-id="34566760">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-qos-pol:policies nc:operation="replace"/>
                    <bbf-qos-pol:qos-policy-profiles nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )



#
# bbf-qos-classifiers
#

def netconf_remove_bbf_qos_classifiers_all(nc):
    nc.edit_config(
        message="Removing all from bbf-qos-classifiers.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-qos-cls="urn:bbf:yang:bbf-qos-classifiers"
            message-id="34566760">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-qos-cls:classifiers nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )



#
# bbf-subscriber-profiles
#

def netconf_get_bbf_subscriber_profile_names(nc, name_match=None):
    names = []

    # Send a Netconf <get> request to retreive the names
    GET_REQ = f'''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:bbf-subprof="urn:bbf:yang:bbf-subscriber-profiles"
        message-id="1">
        <get>
            <filter type="subtree">
                <bbf-subprof:subscriber-profiles>
                    <bbf-subprof:subscriber-profile>
                        <bbf-subprof:name/>
                    </bbf-subprof:subscriber-profile>
                </bbf-subprof:subscriber-profiles>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(
        data_xml=GET_REQ,
        message="/bbf-subprof:subscriber-profiles/bbf-subprof:subscriber-profile/bbf-subprof:name")

    # Parse the Netconf response and build the list of names.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'bbf-subprof' : "urn:bbf:yang:bbf-subscriber-profiles",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/bbf-subprof:subscriber-profiles/bbf-subprof:subscriber-profile/bbf-subprof:name", namespaces=NSMAP):
            name = name.text
            if not name_match or name_match in name:
                names.append(name)

    return names

def netconf_remove_bbf_subscriber_profile_all(nc):
    nc.edit_config(
        message="Removing all from bbf-subscriber-profiles.",
        data_xml='''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-subprof="urn:bbf:yang:bbf-subscriber-profiles"
            message-id="34566760">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                    <bbf-subprof:subscriber-profiles nc:operation="replace"/>
                </config>
            </edit-config>
        </rpc>
        '''
    )

def netconf_remove_bbf_subscriber_profile_by_name(nc, name):
    nc.edit_config(
        message=f"Removing bbf-subscriber-profile {name}.",
        data_xml=f'''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:bbf-subprof="urn:bbf:yang:bbf-subscriber-profiles"
            message-id="1">
            <edit-config>
                <target>
                    <running/>
                </target>
                <config>
                <bbf-subprof:subscriber-profiles>
                    <bbf-subprof:subscriber-profile nc:operation="remove">
                        <bbf-subprof:name>{name}</bbf-subprof:name>
                    </bbf-subprof:subscriber-profile>
                </bbf-subprof:subscriber-profiles>
                </config>
            </edit-config>
        </rpc>
        '''
    )


def netconf_remove_all_config(nc):
    # Delete all L2 Forwarding configuration
    print("\nCleaning up all L2 Forwarding.")
    netconf_remove_bbf_l2_forwarding_all(nc)

    # Delete all Gemport configuration
    print("\nCleaning up all Gemports.")
    netconf_remove_bbf_xpongemtcont_gemport_all(nc)

    # Delete all TCONT configuration
    print("\nCleaning up all TCONTs.")
    netconf_remove_bbf_xpongemtcont_tcont_all(nc)

    # Delete all Link Table configuration
    print("\nCleaning up all Link Table Entries.")
    netconf_remove_bbf_link_table_all(nc)

    # Delete all Tibit BBF Interface configuration
    print("\nCleaning up all Link Table Entries.")
    netconf_remove_tibit_bbf_interface_all(nc)

    # Delete all IETF Interface configuration
    print("\nCleaning up all IETF Interface Entries.")
    netconf_remove_ietf_interface_all(nc)

    # Delete QoS policy profiles
    print("\nCleaning up all QoS Policy Profiles.")
    netconf_remove_bbf_qos_policy_profiles_all(nc)

    # Delete QoS Classifiers
    print("\nCleaning up all QoS Classifiers.")
    netconf_remove_bbf_qos_classifiers_all(nc)

    # Delete all Subscriber Profiles
    print("\nCleaning up all Subscriber Profiles.")
    netconf_remove_bbf_subscriber_profile_all(nc)

    # Delete L2 DHCPv4 Relay Profiles
    print("\nCleaning up all L2 DHCPv4 Relay Profiles.")
    netconf_remove_bbf_l2_dhcpv4_relay_profiles_all(nc)

def netconf_remove_config_for_olt_port(nc, olt_port):
    # Clean up all ONUs under the OLT
    print(f"\nCleaning up all ONUs for OLT port {olt_port}.")
    onus = netconf_get_onus_by_olt_port(nc, olt_port)
    print(f"Removing ONUs: {onus}")
    for onu in onus:
        netconf_remove_config_for_onu(nc, onu)

    # Clean up Tibit BBF interface entry
    print(f"\nCleaning up the Tibit BBF interface entry.")
    olts = netconf_get_tibit_bbf_interface_olts(nc, olt_port)
    print(f"Removing OLTs: {olts}")
    for olt in olts:
        netconf_remove_tibit_bbf_interface_by_olt(nc, olt)

    # Clean up the PON Channel Group pon pools
    print(f"\nCleaning up the PON Channel Group pon pools.")
    entries = netconf_get_ietf_interface_names(nc, name_match=olt_port, if_type="bbf-xponift:channel-group")
    print(f"Removing pon-pools for ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_bbf_channel_group_pon_pools_by_name(nc, entry)

    # Clean up the PON Termination interface
    print(f"\nCleaning up the PON Channel Termination interface.")
    entries = netconf_get_ietf_interface_names(nc, name_match=olt_port, if_type="bbf-xponift:channel-termination")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Clean up the PON Pair interface
    print(f"\nCleaning up the PON Channel Pair interface.")
    entries = netconf_get_ietf_interface_names(nc, name_match=olt_port, if_type="bbf-xponift:channel-pair")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Clean up the PON Partition interface
    print(f"\nCleaning up the PON Channel Partition interface.")
    entries = netconf_get_ietf_interface_names(nc, name_match=olt_port, if_type="bbf-xponift:channel-partition")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Clean up the PON Channel Group interfaces
    print(f"\nCleaning up the PON Channel Group interface.")
    entries = netconf_get_ietf_interface_names(nc, name_match=olt_port, if_type="bbf-xponift:channel-group")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Clean up the NNI Uplink interface
    print(f"\nCleaning up the NNI Uplink interface.")
    entries = netconf_get_ietf_interface_names(nc, name_match=olt_port, if_type="ianaift:ethernetCsmacd")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)


def netconf_remove_all_onu_service_config(nc):
    # Delete all L2 Forwarding configuration
    print("\nCleaning up all L2 Forwarding.")
    netconf_remove_bbf_l2_forwarding_all(nc)

    # Delete all Subscriber Profiles
    print("\nCleaning up all Subscriber Profiles.")
    netconf_remove_bbf_subscriber_profile_all(nc)

    # Delete L2 DHCPv4 Relay Profiles
    print("\nCleaning up all L2 DHCPv4 Relay Profiles.")
    netconf_remove_bbf_l2_dhcpv4_relay_profiles_all(nc)

    # Delete all Gemport configurations
    print("\nCleaning up all Gemports.")
    netconf_remove_bbf_xpongemtcont_gemport_all(nc)

    # Delete all TCONTs configurations
    print("\nCleaning up all TCONTs.")
    netconf_remove_bbf_xpongemtcont_tcont_all(nc)

    # Delete all Link Table configuration
    print("\nCleaning up all Link Table Entries.")
    netconf_remove_bbf_link_table_all(nc)

    #
    # Delete all ONU-related interface configuration
    #

    # Remove all OLT and ONU VLAN subinterfaces (old format)
    print("\nCleaning up all VLAN subinterfaces (bbf-xponift).")
    entries = netconf_get_ietf_interface_names(nc, if_type="bbfift:vlan-sub-interface")
    print(f"Removing ietf-interface, vlan-sub-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove all OLT and ONU VLAN subinterfaces (new format)
    print("\nCleaning up all VLAN subinterfaces (bbf-xponift).")
    entries = netconf_get_ietf_interface_names(nc, if_type="bbf-xponift:vlan-sub-interface")
    print(f"Removing ietf-interface, vlan-sub-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU UNI interfaces
    print("\nCleaning up all ONU UNI interfaces.")
    entries = netconf_get_ietf_interface_names(nc, name_match="onu-", if_type="ianaift:ethernetCsmacd")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU OLT-V-ENET interfaces
    print("\nCleaning up all ONU OLT-V-ENET interfaces.")
    entries = netconf_get_ietf_interface_names(nc, if_type="bbf-xponift:olt-v-enet")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU ANI interfaces
    print("\nCleaning up all ONU ANI interfaces.")
    entries = netconf_get_ietf_interface_names(nc, if_type="bbf-xponift:ani")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU V-ANI interfaces
    print("\nCleaning up all ONU V-ANI interfaces.")
    entries = netconf_get_ietf_interface_names(nc, if_type="bbf-xponift:v-ani")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)


def netconf_remove_config_for_onu(nc, onu):
    # Get the list of L2 forwarding entries for this ONU
    print("\nCleaning up OLT Forwarding.")
    entries = netconf_get_bbf_l2_forwarding_names(nc, name_match=onu)
    print(f"Removing bbf-l2-forwarding entries: {entries}")

    # Before deleting the OLT forwarding entries, get the list of VLAN subinterfaces attached to forwarders.
    olt_vlan_interface_entries = []
    for entry in entries:
        olt_vlan_interface_entries += netconf_get_bbf_l2_forwarding_vlan_interface_names_from_forwarder(nc, entry)

    # Remove OLT L2 forwarding entries for this ONU
    for entry in entries:
        netconf_remove_bbf_l2_forwarding_by_name(nc, entry)

    # Remove OLT VLAN subinterfaces for this ONU
    print("\nCleaning up OLT VLAN subinterfaces.")
    print(f"Removing ietf-interface, vlan-sub-interface entries: {olt_vlan_interface_entries}")
    for entry in olt_vlan_interface_entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Delete all Subscriber Profiles
    print("\nCleaning up Subscriber Profiles.")
    entries = netconf_get_bbf_subscriber_profile_names(nc, name_match=onu)
    print(f"Removing bbf-subscriber-profiles: {entries}")
    for entry in entries:
        netconf_remove_bbf_subscriber_profile_by_name(nc, entry)

    # Remove Gemports
    print("\nCleaning up Gemports.")
    entries = netconf_get_bbf_xpongemtcont_gemport_names(nc, name_match=onu)
    print(f"Removing bbf-xpongemtcont gemports: {entries}")
    for entry in entries:
        netconf_remove_bbf_xpongemtcont_gemport_by_name(nc, entry)

    # Remove Tconts
    print("\nCleaning up TCONTs.")
    entries = netconf_get_bbf_xpongemtcont_tcont_names(nc, name_match=onu)
    print(f"Removing bbf-xpongemtcont tconts: {entries}")
    for entry in entries:
        netconf_remove_bbf_xpongemtcont_tcont_by_name(nc, entry)

    # Remove ONU VLAN subinterfaces (old format)
    print("\nCleaning up ONU VLAN subinterfaces (bbf-xponift).")
    entries = netconf_get_ietf_interface_names(nc, name_match=onu, if_type="bbfift:vlan-sub-interface")
    print(f"Removing ietf-interface, vlan-sub-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU VLAN subinterfaces (new format)
    print("\nCleaning up ONU VLAN subinterfaces (bbf-xponift).")
    entries = netconf_get_ietf_interface_names(nc, name_match=onu, if_type="bbf-xponift:vlan-sub-interface")
    print(f"Removing ietf-interface, vlan-sub-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove Link Table entries
    print("\nCleaning up Link Table.")
    entries = netconf_get_bbf_link_table_names(nc, name_match=onu)
    print(f"Removing bbf-link-table entries: {entries}")
    for entry in entries:
        netconf_remove_bbf_link_table_by_name(nc, entry)

    # Remove ONU UNI interfaces
    print("\nCleaning up ONU UNI interfaces.")
    entries = netconf_get_ietf_interface_names(nc, name_match=onu, if_type="ianaift:ethernetCsmacd")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU OLT-V-ENET interfaces
    print("\nCleaning up ONU OLT-V-ENET interfaces.")
    entries = netconf_get_ietf_interface_names(nc, name_match=onu, if_type="bbf-xponift:olt-v-enet")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU ANI interfaces
    print("\nCleaning up ONU ANI interfaces.")
    entries = netconf_get_ietf_interface_names(nc, name_match=onu, if_type="bbf-xponift:ani")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

    # Remove ONU V-ANI interfaces
    print("\nCleaning up ONU V-ANI interfaces.")
    entries = netconf_get_ietf_interface_names(nc, name_match=onu, if_type="bbf-xponift:v-ani")
    print(f"Removing ietf-interface entries: {entries}")
    for entry in entries:
        netconf_remove_ietf_interface_by_name(nc, entry)

if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--all", action="store_true", dest="all", default=False, required=False, help="Delete all BBF YANG configuration (except SLAs).")
    parser.add_argument(      "--all_onu", action="store_true", dest="all_onu", default=False, required=False, help="Delete all ONU related service configuration.")
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt",  action="store", dest="olt", default=None, required=False, help="Delete all configuration for a specific OLT PON by OLT MAC address (e.g., e8:b4:70:70:0c:9c)")
    parser.add_argument(      "--olt_port", action="store", dest="olt_port", default=None, required=False, help="Delete all configuration for a specific OLT PON by OLT Port number. The OLT Port number could be a logical or physical port number representing the switch port (e.g., LLDP switch port ID)")
    parser.add_argument(      "--onu",  action="store", dest="onu", default=None, required=False, help="Delete all configuration for a specific ONU by ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if nc:
        if args.all:
            netconf_remove_all_config(nc)
        elif args.all_onu:
            netconf_remove_all_onu_service_config(nc)
        elif args.olt:
            olt_port = netconf_get_olt_port_from_mac_addr(nc, args.olt)
            if olt_port:
                netconf_remove_config_for_olt_port(nc, olt_port)
            else:
                print(f"ERROR: OLT port for {args.olt} not found.")
        elif args.olt_port:
            netconf_remove_config_for_olt_port(nc, args.olt_port)
        elif args.onu:
            netconf_remove_config_for_onu(nc, args.onu)
