#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import sys
import os
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt_port", action="store", dest="olt_port", default=None, required=False, help="OLT Port number. This could be a logical port number or a physical port number representing the switch port (e.g., LLDP switch port ID)")
    parser.add_argument(      "--olt_tag", action="store", dest="olt_tag", default=None, required=True, help="Tag to be added by the OLT")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    # The variables {{VAR}} be substituted in the Netconf requests from the
    # .xml files sent below
    options = {
        "{{OLT_PORT}}" : args.olt_port,
        "{{OLT_TAG}}" : args.olt_tag,
        "{{ONU}}" : args.onu,
    }

    # Delete OLT forwarding and VLAN configuration
    nc.edit_config(
        message = "6-olt-1to1-forwarding",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <!-- Delete the NNI Network on the OLT (???) -->
                <bbf-l2-fwd:forwarding xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding">
                    <bbf-l2-fwd:forwarders>
                    <!-- forwarder configuration from User 1 to S-TAG 201 -->
                    <bbf-l2-fwd:forwarder nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-l2-fwd:name>{{ONU}}_s{{OLT_TAG}}.c21.c0</bbf-l2-fwd:name>
                    </bbf-l2-fwd:forwarder>
                    <!-- forwarder configuration from User 1 to S-TAG 201 -->
                    <bbf-l2-fwd:forwarder nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-l2-fwd:name>{{ONU}}_s{{OLT_TAG}}.c22.c0</bbf-l2-fwd:name>
                    </bbf-l2-fwd:forwarder>
                    <!-- forwarder configuration from User 1 to S-TAG 201 -->
                    <bbf-l2-fwd:forwarder nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-l2-fwd:name>{{ONU}}_s{{OLT_TAG}}.c23.c0</bbf-l2-fwd:name>
                    </bbf-l2-fwd:forwarder>
                    <!-- forwarder configuration from User 1 to S-TAG 201 -->
                    <bbf-l2-fwd:forwarder nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-l2-fwd:name>{{ONU}}_s{{OLT_TAG}}.c24.c0</bbf-l2-fwd:name>
                    </bbf-l2-fwd:forwarder>
                    <!-- forwarder configuration from User 1 to S-TAG 201 -->
                    <bbf-l2-fwd:forwarder nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-l2-fwd:name>{{ONU}}_s{{OLT_TAG}}.c25.c0</bbf-l2-fwd:name>
                    </bbf-l2-fwd:forwarder>
                    </bbf-l2-fwd:forwarders>
                </bbf-l2-fwd:forwarding>
                <!-- Delete OLT VLAN sub-interfaces -->
                <if:interfaces xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces">
                    <!-- Delete OLT VLAN NNI sub-interfaces -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>network1.s{{OLT_TAG}}.c21.c0</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>network1.s{{OLT_TAG}}.c22.c0</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>network1.s{{OLT_TAG}}.c23.c0</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>network1.s{{OLT_TAG}}.c24.c0</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>network1.s{{OLT_TAG}}.c25.c0</if:name>
                    </if:interface>
                    <!-- Delete OLT VLAN PON sub-interfaces -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <if:name>olt-{{ONU}}-eth.1.21</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <if:name>olt-{{ONU}}-eth.2.22</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <if:name>olt-{{ONU}}-eth.3.23</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <if:name>olt-{{ONU}}-eth.4.24</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <if:name>olt-{{ONU}}-eth.5.25</if:name>
                    </if:interface>
                </if:interfaces>
                <!-- Delete the Subscriber Profile for this ONU -->
                <subscriber-profiles xmlns="urn:bbf:yang:bbf-subscriber-profiles">
                    <subscriber-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <name>{{ONU}}.21</name>
                    </subscriber-profile>
                    <subscriber-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <name>{{ONU}}.22</name>
                    </subscriber-profile>
                    <subscriber-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <name>{{ONU}}.23</name>
                    </subscriber-profile>
                    <subscriber-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <name>{{ONU}}.24</name>
                    </subscriber-profile>
                    <subscriber-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <name>{{ONU}}.25</name>
                    </subscriber-profile>
                </subscriber-profiles>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete Tconts, Gemports, and Upstream SLAs
    nc.edit_config(
        message = "5-gemports",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running />
            </target>
            <config>
                <bbf-xpongemtcont:xpongemtcont xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont">
                    <!-- Delete tconts -->
                    <bbf-xpongemtcont:tconts>
                    <bbf-xpongemtcont:tcont nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>vani-{{ONU}}-tcont.1</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:tcont>
                    <bbf-xpongemtcont:tcont nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>vani-{{ONU}}-tcont.2</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:tcont>
                    <bbf-xpongemtcont:tcont nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>vani-{{ONU}}-tcont.3</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:tcont>
                    <bbf-xpongemtcont:tcont nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>vani-{{ONU}}-tcont.4</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:tcont>
                    <bbf-xpongemtcont:tcont nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>vani-{{ONU}}-tcont.5</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:tcont>
                    </bbf-xpongemtcont:tconts>
                    <!-- Delete gemports -->
                    <bbf-xpongemtcont:gemports>
                    <bbf-xpongemtcont:gemport nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>gem-{{ONU}}-eth.1-tc0</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:gemport>
                    <bbf-xpongemtcont:gemport nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>gem-{{ONU}}-eth.2-tc0</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:gemport>
                    <bbf-xpongemtcont:gemport nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>gem-{{ONU}}-eth.3-tc0</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:gemport>
                    <bbf-xpongemtcont:gemport nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>gem-{{ONU}}-eth.4-tc0</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:gemport>
                    <bbf-xpongemtcont:gemport nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>gem-{{ONU}}-eth.5-tc0</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:gemport>
                    </bbf-xpongemtcont:gemports>
                    <!-- Delete the Upstream SLA -->
                    <!-- <bbf-xpongemtcont:traffic-descriptor-profiles> -->
                    <!-- <bbf-xpongemtcont:traffic-descriptor-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"> -->
                    <!--     <bbf-xpongemtcont:name>sla-add-ctag</bbf-xpongemtcont:name> -->
                    <!-- </bbf-xpongemtcont:traffic-descriptor-profile> -->
                    <!-- </bbf-xpongemtcont:traffic-descriptor-profiles> -->
                </bbf-xpongemtcont:xpongemtcont>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete ONU sub-interfaces
    nc.edit_config(
        message = "4-onu-vlan-sub-interfaces",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566757">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <interfaces xmlns="urn:ietf:params:xml:ns:yang:ietf-interfaces">
                    <!-- Delete the ONU vlan sub-interface -->
                    <interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <name>onu-{{ONU}}-eth.1.21</name>
                    </interface>
                    <interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <name>onu-{{ONU}}-eth.2.22</name>
                    </interface>
                    <interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <name>onu-{{ONU}}-eth.3.23</name>
                    </interface>
                    <interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <name>onu-{{ONU}}-eth.4.24</name>
                    </interface>
                    <interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <name>onu-{{ONU}}-eth.5.25</name>
                    </interface>
                </interfaces>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete classifiers
    nc.edit_config(
        message = "3-classifiers",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566757">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <!-- Delete Policy Profiles -->
                <bbf-qos-pol:qos-policy-profiles xmlns:bbf-qos-pol="urn:bbf:yang:bbf-qos-policies">
                    <!-- QoS Policy Profile for tcont1 -->
                    <bbf-qos-pol:policy-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex1-qos-policy-profile-tc1</bbf-qos-pol:name>
                    </bbf-qos-pol:policy-profile>
                    <!-- QoS Policy Profile for tcont2 -->
                    <bbf-qos-pol:policy-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex2-qos-policy-profile-tc2</bbf-qos-pol:name>
                    </bbf-qos-pol:policy-profile>
                    <!-- QoS Policy Profile for tcont3 -->
                    <bbf-qos-pol:policy-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex3-qos-policy-profile-tc3</bbf-qos-pol:name>
                    </bbf-qos-pol:policy-profile>
                    <!-- QoS Policy Profile for tcont4 -->
                    <bbf-qos-pol:policy-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex4-qos-policy-profile-tc4</bbf-qos-pol:name>
                    </bbf-qos-pol:policy-profile>
                    <!-- QoS Policy Profile for tcont5 -->
                    <bbf-qos-pol:policy-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex5-qos-policy-profile-tc5</bbf-qos-pol:name>
                    </bbf-qos-pol:policy-profile>
                </bbf-qos-pol:qos-policy-profiles>
                <!-- Delete Policies -->
                <bbf-qos-pol:policies xmlns:bbf-qos-pol="urn:bbf:yang:bbf-qos-policies">
                    <!-- QoS Policy for tcont1 -->
                    <bbf-qos-pol:policy nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex1-policy-tc-1</bbf-qos-pol:name>
                    </bbf-qos-pol:policy>
                    <!-- QoS Policy for tcont2 -->
                    <bbf-qos-pol:policy nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex2-policy-tc-2</bbf-qos-pol:name>
                    </bbf-qos-pol:policy>
                    <!-- QoS Policy for tcont3 -->
                    <bbf-qos-pol:policy nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex3-policy-tc-3</bbf-qos-pol:name>
                    </bbf-qos-pol:policy>
                    <!-- QoS Policy for tcont4 -->
                    <bbf-qos-pol:policy nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex4-policy-tc-4</bbf-qos-pol:name>
                    </bbf-qos-pol:policy>
                    <!-- QoS Policy for tcont5 -->
                    <bbf-qos-pol:policy nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-pol:name>ex5-policy-tc-5</bbf-qos-pol:name>
                    </bbf-qos-pol:policy>
                </bbf-qos-pol:policies>
                <!-- Delete Classifiers -->
                <bbf-qos-cls:classifiers xmlns:bbf-qos-cls="urn:bbf:yang:bbf-qos-classifiers">
                    <!-- QoS Classifier for tcont1 -->
                    <bbf-qos-cls:classifier-entry nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-cls:name>ex1-classifier-tc-1</bbf-qos-cls:name>
                    </bbf-qos-cls:classifier-entry>
                    <!-- QoS Classifier for tcont2 -->
                    <bbf-qos-cls:classifier-entry nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-cls:name>ex2-classifier-tc-2</bbf-qos-cls:name>
                    </bbf-qos-cls:classifier-entry>
                    <!-- QoS Classifier for tcont3 -->
                    <bbf-qos-cls:classifier-entry nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-cls:name>ex3-classifier-tc-3</bbf-qos-cls:name>
                    </bbf-qos-cls:classifier-entry>
                    <!-- QoS Classifier for tcont4 -->
                    <bbf-qos-cls:classifier-entry nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-cls:name>ex4-classifier-tc-4</bbf-qos-cls:name>
                    </bbf-qos-cls:classifier-entry>
                    <!-- QoS Classifier for tcont5 -->
                    <bbf-qos-cls:classifier-entry nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-qos-cls:name>ex5-classifier-tc-5</bbf-qos-cls:name>
                    </bbf-qos-cls:classifier-entry>
                </bbf-qos-cls:classifiers>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete Link Table entries
    nc.edit_config(
        message = "2-link-table",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566754">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <bbf-lt:link-table xmlns:bbf-lt="urn:bbf:yang:bbf-link-table">
                    <!-- Delete v-ani to ani mapping -->
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>ani-{{ONU}}</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                    <!-- Delete v-enet to uni mapping -->
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>onu-{{ONU}}-eth.1</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>onu-{{ONU}}-eth.2</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>onu-{{ONU}}-eth.3</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>onu-{{ONU}}-eth.4</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>onu-{{ONU}}-eth.5</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                </bbf-lt:link-table>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete ONU Interfaces
    nc.edit_config(
        message = "1-onu-interfaces",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566754">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <if:interfaces xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces">
                    <!-- Delete the v-ani interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>vani-{{ONU}}</if:name>
                    </if:interface>
                    <!-- Delete the ani interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>ani-{{ONU}}</if:name>
                    </if:interface>
                    <!-- Delete the v-enet interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>olt-v-enet-{{ONU}}.1</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>olt-v-enet-{{ONU}}.2</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>olt-v-enet-{{ONU}}.3</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>olt-v-enet-{{ONU}}.4</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>olt-v-enet-{{ONU}}.5</if:name>
                    </if:interface>
                    <!-- Delete the uni/veip interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>onu-{{ONU}}-eth.1</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>onu-{{ONU}}-eth.2</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>onu-{{ONU}}-eth.3</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>onu-{{ONU}}-eth.4</if:name>
                    </if:interface>
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>onu-{{ONU}}-eth.5</if:name>
                    </if:interface>
                </if:interfaces>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )
