#!/usr/bin/env bash
# --------------------------------------------------------------------------#
#  Copyright (C) 2022 by Tibit Communications, Inc.                         #
#  All rights reserved.                                                     #
#                                                                           #
#     _______ ____  _ ______                                                #
#    /_  __(_) __ )(_)_  __/                                                #
#     / / / / __  / / / /                                                   #
#    / / / / /_/ / / / /                                                    #
#   /_/ /_/_____/_/ /_/                                                     #
#                                                                           #
#                                                                           #
#  Distributed as Tibit-Customer confidential                               #
#                                                                           #
# --------------------------------------------------------------------------#

TMPFILE=$(mktemp /tmp/tibit_nc_rpc.XXXXXX)

# Usage
usage () {
    cat <<HELP_USAGE

    $0 olt-name <true|false>

    Set the PON port enable status for an OLT device.

    Options:
      -h|--help       Display this help and usage text.
      -p|--port       TCP port number used to connect to the NETCONF Server (default: 830).
      -s|--server     Hostname or IP address used to connect to the NETCONF Server (default: localhost).
      -u|--username   Username with access to the NETCONF Server (default: $USER).

    Example:
      $0 70:b3:d5:52:37:24 true

HELP_USAGE

exit 0
}

# Defaults
HOST=localhost
PORT=830

# Parse command line arguments
PARAMS=""
while (( "$#" )); do
    case "$1" in
        -s|--server)
            HOST=$2
            shift 2
            ;;
        -p|--port)
            PORT=$2
            shift 2
            ;;
        -u|--username)
            USER=$2
            shift 2
            ;;
        -h|--help)
            usage
            ;;
        --) # end argument parsing
            shift
            break
            ;;
        -*|--*=) # unsupported option
            echo "Error: Unsupported option $1" >&2
            usage
            ;;
        *) # preserve positional arguments
            PARAMS="$PARAMS $1"
            shift
            ;;
    esac
done
# set positional arguments in their proper place
eval set -- "$PARAMS"

[ -z "$1" ] && { usage; }
[ -z $2 ] && { usage; }

OLT="${1}"
ENABLE=$2

echo "Modifying PON port enable status to '${ENABLE}' for OLT '${OLT}'."

cat >${TMPFILE}  << EDITCONFIGXML
  <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
    <target>
      <running/>
    </target>
    <config>
      <tibitcntlr:olt xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
        <tibitcntlr:olt>
          <tibitcntlr:name>${OLT}</tibitcntlr:name>
          <tibitcntlr:olt>
            <tibitcntlr:pon-enable>${ENABLE}</tibitcntlr:pon-enable>
          </tibitcntlr:olt>
        </tibitcntlr:olt>
      </tibitcntlr:olt>
    </config>
  </edit-config>
EDITCONFIGXML

PATH=${PATH}:/opt/tibit/netconf/bin netopeer2-cli <<COMMANDINPUT
connect --host ${HOST} --port ${PORT} --login ${USER}
user-rpc --content ${TMPFILE}
exit
COMMANDINPUT

rm -rf ${TMPFILE}
