#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import sys
import json
from lxml import etree
import ncclient
from ncclient import manager
import argparse
import time


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-d", "--delete", action="store_true", dest="delete", default=False, required=False, help="Delete the ONUs.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument("-n", "--num", action="store", dest="num", default=1, required=False, help="Number of ONUs to create or delete.")
    parser.add_argument("-w", "--password", action="store", dest="password", default=None, required=False, help="Password.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=True, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Connect to the Netconf Server
    with manager.connect(host=args.host, port=args.port, username=args.user, password=args.password, hostkey_verify=False) as conn:

        # If the delete option was specified, set the Netconf operation to 'remove'. This will remove
        # the ONU and return success regardless if the ONU exists or not.
        # Otherwise, set the Netconf operation to 'merge', which with create a new record if the the
        # ONU does not exist or update an existing ONU record.
        if args.delete:
            operation = 'remove'
        else:
            operation = 'merge'

        # Limit the number of ONUs to 256 based on the logic below.
        num_onus = int(args.num)
        if num_onus > 256:
            print("WARNING: Limiting create to 256 ONUs.")
            num_onus = 256

        # Create or delete all ONUS
        for cnt in range(0, num_onus):

            # Define the ONU name (key)
            name = "TBIT000000{:02x}".format(cnt)

            # Define the device-id, which is a Serial Number for GPON ONUs.
            # (In this example, ONU name and device-id are set to the same value.)
            device_id = "TBIT000000{:02x}".format(cnt)

            if args.delete:
                print("Deleting ONU with name {}, device-id {}".format(name, device_id))
            else:
                print("Creating ONU with name {}, device-id {}".format(name, device_id))

            # Define the body for the <edit-config> request
            ONU_CONFIG = '''
                <nc:config xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <tibitcntlr:onu xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                        <tibitcntlr:onu nc:operation="{}">
                        <tibitcntlr:name>{}</tibitcntlr:name>
                        <tibitcntlr:device-id>{}</tibitcntlr:device-id>
                        <tibitcntlr:onu>
                            <tibitcntlr:service-config>DISABLED</tibitcntlr:service-config>
                        </tibitcntlr:onu>
                        </tibitcntlr:onu>
                    </tibitcntlr:onu>
                </nc:config>
                '''.format(operation,name,device_id)

            # Send the <edit-config> request
            rpc_rsp = conn.edit_config(target='running', config=ONU_CONFIG).xml

            # Display the response if verbose output is enabled
            if args.verbose:
                print("<rpc-reply>")
                root = etree.fromstring(bytes(rpc_rsp, encoding='utf-8'))
                print(etree.tostring(root, pretty_print=True, encoding='unicode'))