#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# --------------------------------------------------------------------------#

""" Get all ONUs via BBF Yang models.

This BBF YANG Example script retrieves all ONUs, their name, and their current state.

Example:

  ./bbf_list_onus.py


usage: bbf_list_onus.py [--help] [-h HOST] [--olt OLT] [--olt_port OLT_PORT] [-w PASSWD]
                                 [-p PORT] [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --olt OLT             The olt mac address to filter results on
  --olt_port OLT_PORT   The olt port number to filter results on
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys

from lxml import etree

from netconf_driver import NetconfDriver
from netconf_utils import get_dict_from_etree, nat_sort


def get_ietf_hardware_entries(nc):
    """ Get a list of all ietf-hardware entries

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    hardware_entries = []
    data_xml=f'''<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name/>
                            <hw:serial-num/>
                            <hw:firmware-rev/>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                </rpc>'''
    xml = nc.get(data_xml=data_xml, message="/hw:hardware/hw:component")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        if isinstance(data['hardware']['component'], list):
            hardware_entries = data['hardware']['component']
        else:
            hardware_entries = [data['hardware']['component']]
    return hardware_entries


def get_bbf_vani_interface_by_ssn(nc, ssn):
    """ Get the V-ANI interface by ONU Serial Number

    Args:
        nc (obj): Netconf connection to use for this request.
        ssn (str): The expected serial number of the desired vani interface.
        This is located in the interface's v-ani section under the
        expected-serial-number field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """

    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-xponvani="urn:bbf:yang:bbf-xponvani"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces>
                          <if:interface>
                            <bbf-xponvani:v-ani>
                              <expected-serial-number>
                              {ssn}
                              </expected-serial-number>
                            </bbf-xponvani:v-ani>
                          </if:interface>
                        </if:interfaces>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces/if:interface/bbf-xponvani:v-ani[expected-serial-number={ssn}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'interface' in data['interfaces']:
        data = data['interfaces']['interface']
    else:
        data = None
    return data


def get_ietf_interface_state_by_name(nc, name):
    """ Get an interface-state entry by interface name

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name of the desired /ietf:interfaces-state/ietf:interface

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-xponift="urn:bbf:yang:bbf-xpon-if-type"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces-state>
                          <if:interface>
                            <if:name>
                                {name}
                            </if:name>
                          </if:interface>
                        </if:interfaces-state>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces-state/if:interface[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces-state' in data and 'interface' in data['interfaces-state']:
        data = data['interfaces-state']['interface']
    else:
        data = None
    return data


def get_channel_termination_states_by_ssn(nc, ssn):
    """ Get a list of BBF Channel Termination interface-state entries for this ONU.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name of the desired /ietf:interfaces-state/ietf:interface

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    xmlns:bbf-xponift="urn:bbf:yang:bbf-xpon-if-type"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces-state>
                          <if:interface>
                            <bbf-xpon:channel-termination xmlns:bbf-xpon="urn:bbf:yang:bbf-xpon">
                              <bbf-xpon-onu-s:onus-present-on-local-channel-termination xmlns:bbf-xpon-onu-s="urn:bbf:yang:bbf-xpon-onu-state">
                                <bbf-xpon-onu-s:onu>
                                  <detected-serial-number>{ssn}</detected-serial-number>
                                </bbf-xpon-onu-s:onu>
                              </bbf-xpon-onu-s:onus-present-on-local-channel-termination>
                            </bbf-xpon:channel-termination>
                          </if:interface>
                        </if:interfaces-state>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces-state/if:interface/bbf-xpon:channel-termination/bbf-xpon-onu-s:onus-present-on-local-channel-termination/bbf-xpon-onu-s:onu[detected-serial-number={ssn}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces-state' in data and 'interface' in data['interfaces-state']:
        if isinstance(data['interfaces-state']['interface'], list):
            data = data['interfaces-state']['interface']
        else:
            data = [data['interfaces-state']['interface']]
    else:
        data = []
    return data


def get_tibit_bbf_olt_interface_map_by_channel_termination(nc, channel_termination_ref):
    """ Get a Tibit YANG BBF OLT interface mapping entry by PON Channel Termination reference.

    Args:
        nc (obj): Netconf connection to use for this request.
        device_id (str): The device-id used for the interface that appears in the device-id field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                <get>
                  <filter type="subtree">
                    <tibit-bbf-if:interfaces xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces">
                      <tibit-bbf-if:olt-interface-map>
                        <tibit-bbf-if:olt>
                          <tibit-bbf-if:pon>
                            <channel-termination-ref>{channel_termination_ref}</channel-termination-ref>
                          </tibit-bbf-if:pon>
                        </tibit-bbf-if:olt>
                      </tibit-bbf-if:olt-interface-map>
                    </tibit-bbf-if:interfaces>
                  </filter>
                </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map/tibit-bbf-if:pon[channel-termination-ref={channel_termination_ref}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'olt-interface-map' in data['interfaces'] and 'olt' in data['interfaces']['olt-interface-map']:
        data = data['interfaces']['olt-interface-map']['olt']
    else:
        data = None
    return data


def get_ietf_hardware_names(nc):
    """  Get a list of all ietf-hardware entry names

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    hardware_names = []
    data_xml=f'''<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name/>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                </rpc>'''
    xml = nc.get(data_xml=data_xml, message="nc:data/hw:hardware/hw:component/hw:name")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        for name in data['hardware']['component']:
            hardware_names.append(name['name'])
    return hardware_names


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False,
                        help="NETCONF Server IP address or hostname.")
    parser.add_argument("--olt", action="store", dest="olt", default=None, help="olt mac address")
    parser.add_argument("--olt_port", action="store", dest="olt_port", default=None, help="olt device port")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False,
                        help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False,
                        help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False,
                        help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Create a connection to the Netconf server.
    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    #
    # Gather information via BBF YANG for displaying the list of ONUs.
    #
    onus = []

    # Get a list of ONUs from the ietf-hardware entries
    components = get_ietf_hardware_entries(nc)
    for component in components:
        onu = {}
        if 'ONU' in component['name']:
            # Parse the ONU identifier from the component name (GPON Serial Number or EPON MAC address)
            onu['id'] = component['name'].split(" ")[2]
            onu['firmware-rev'] = str(component['firmware-rev'])
            onus.append(onu)

    # Get the V-ANI interface configuration for each ONU
    for onu in onus:
        # Get the v-ANI interface by its serial number
        vani_interface = \
            get_bbf_vani_interface_by_ssn(nc, onu['id'])
        # Set the vani-name and get the state of the channel-termination interface
        if vani_interface:
            onu['vani-name'] = str(vani_interface['name'])
        else:
            onu['vani-name'] = ""

    # Get the V-ANI interface state for each ONU
    for onu in onus:
        vani_interface_st = None
        if onu['vani-name']:
            vani_interface_st = get_ietf_interface_state_by_name(nc, onu['vani-name'])

        # Set the vani-name and get the state of the channel-termination interface
        if vani_interface_st:
            # Strip the name space from the presence state
            onu['status'] = vani_interface_st['v-ani']['onu-presence-state'].split(":")[1]
            onu['channel-termination'] = vani_interface_st['v-ani']['onu-present-on-this-olt']['onu-present-on-this-channel-termination']
        else:
            onu['status'] = ""
            onu['channel-termination'] = ""

    # If the ONU has no V-ANI configuration or state, see if the ONU is present
    # on an OLT without a V-ANI
    for onu in onus:
        channel_termination_states = get_channel_termination_states_by_ssn(nc, onu['id'])
        for channel_termination_st in channel_termination_states:
            # Strip the name space from the presence state
            onu['status'] = channel_termination_st['channel-termination']['onus-present-on-local-channel-termination']['onu']['onu-presence-state'].split(":")[1]
            onu['channel-termination'] = channel_termination_st['name']
            break

    # Get the OLT information for each ONU
    for onu in onus:
        olt_mac = ""
        olt_switch_port = ""

        # If the ONU is configured with a V-ANI on specific OLT, look up the ONU status from the OLT
        if onu['channel-termination']:
            # Look up the OLT MAC for this ONU's channel termination
            olt_interface_map = get_tibit_bbf_olt_interface_map_by_channel_termination(nc, onu['channel-termination'])
            if olt_interface_map:
                channel_termination_ref = olt_interface_map['pon']['channel-termination-ref']
                olt_mac = olt_interface_map['device-id']

        # Get the OLT Switch Port
        if olt_mac:
            hardware_list = get_ietf_hardware_names(nc)
            for name in hardware_list:
                if "OLT" in name and olt_mac in name:
                    # Parse the switch port out of the name
                    olt_switch_port = name.split(" ")[3].strip("[]")

        # Update the ONU record
        onu["olt-mac"] = olt_mac
        onu["switch-port"] = olt_switch_port

    #
    # Display the list of ONUs
    #

    # Print table header row
    print("")
    print(f"{'ONU': <20}"
            f"{'V-ANI name': <24}"
            f"{'OLT': <24}"
            f"{'Switch Port': <18}"
            f"{'Firmware Version': <18}"
            f"{'Status': <30}")

    # Display ONU entries sorted by Serial Number
    # print(onus)
    for onu in sorted(onus, key=nat_sort('id')):
        # Filter by OLT MAC address
        if args.olt and args.olt != onu['olt-mac']:
            continue
        # Filter by OLT Switch Port
        if args.olt_port and args.olt_port != onu['switch-port']:
            continue
        # Display the row
        print(f"{onu['id']: <20}"
                f"{onu['vani-name']: <24}"
                f"{onu['olt-mac']: <24}"
                f"{onu['switch-port']: <18}"
                f"{onu['firmware-rev']: <18}"
                f"{onu['status']: <30}")
