#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# --------------------------------------------------------------------------#

""" Get OLT information via BBF Yang models.

This BBF YANG Example script retrieves all information for an OLT.

Example:

  ./bbf_get_olt.py --olt e0:b0:40:40:c0:06 or ./bbf_get_olt.py --olt_port 1/0/7


usage: bbf_get_olt.py [--help] [-h HOST] --olt OLT --olt_port OLT_PORT [-w PASSWD]
                                 [-p PORT] [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --olt OLT             The olt mac address (This or the olt port are required)
  --olt_port OLT_PORT   The olt port number (This or the olt mac are required)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys
from lxml import etree
from netconf_driver import NetconfDriver
from netconf_utils import get_dict_from_etree, nat_sort, mw_to_dbm


def get_tibit_bbf_olt_interface_map(nc, device_id):
    """ Get the Tibit YANG BBF OLT interface mapping entry by OLT device id (MAC Address)

    Args:
        nc (obj): Netconf connection to use for this request.
        device_id (str): The device-id used for the interface that appears in the device-id field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                <get>
                  <filter type="subtree">
                    <tibit-bbf-if:interfaces xmlns:tibit-bbf-if="urn:com:tibitcom:ns:yang:bbf:interfaces">
                      <tibit-bbf-if:olt-interface-map>
                        <tibit-bbf-if:olt>
                          <tibit-bbf-if:device-id>{device_id}</tibit-bbf-if:device-id>
                        </tibit-bbf-if:olt>
                      </tibit-bbf-if:olt-interface-map>
                    </tibit-bbf-if:interfaces>
                  </filter>
                </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/tibit-bbf-if:interfaces/tibit-bbf-if:olt-interface-map[device_id={device_id}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'olt-interface-map' in data['interfaces'] and 'olt' in data['interfaces']['olt-interface-map']:
        data = data['interfaces']['olt-interface-map']['olt']
    else:
        data = None
    return data


def get_ietf_interface_by_name(nc, name):
    """ Get an ietf-interface entry by interface name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name used for the interface that appears in the name field.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces>
                          <if:interface>
                            <if:name>
                                {name}
                            </if:name>
                          </if:interface>
                        </if:interfaces>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces/if:interface[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces' in data and 'interface' in data['interfaces']:
        data = data['interfaces']['interface']
    else:
        data = None
    return data


def get_ietf_interface_state_by_name(nc, name):
    """ Get an IETF interface-state entry by interface name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The name of the desired /ietf:interfaces-state/ietf:interface

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f"""<rpc
                    xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces"
                    message-id="">
                  <get>
                      <filter type="subtree">
                        <if:interfaces-state>
                          <if:interface>
                            <if:name>
                                {name}
                            </if:name>
                          </if:interface>
                        </if:interfaces-state>
                      </filter>
                    </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message=f"/if:interfaces-state/if:interface[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'interfaces-state' in data and 'interface' in data['interfaces-state']:
        data = data['interfaces-state']['interface']
    else:
        data = None
    return data


def get_ietf_hardware_by_name(nc, name):
    """ Get an ietf-hardware entry by componet name.

    Args:
        nc (obj): Netconf connection to use for this request.
        name (str): The detected name of the desired hardware component

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f''' <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:bbf-hw-xcvr="urn:bbf:yang:bbf-hardware-transceivers"
                                     xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name>{name}</hw:name>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                 </rpc>'''
    xml = nc.get(data_xml=data_xml, message=f"/hw:hardware/hw:component[name={name}]")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        data = data['hardware']['component']
    else:
        data = None
    return data


def get_ietf_hardware_names(nc):
    """ Get a list of all ietf-hardware entry names

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    data_xml=f'''<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="">
                    <get>
                      <filter type="subtree">
                        <hw:hardware xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware">
                          <hw:component>
                            <hw:name/>
                          </hw:component>
                        </hw:hardware>
                      </filter>
                    </get>
                </rpc>'''
    xml = nc.get(data_xml=data_xml, message="nc:data/hw:hardware/hw:component/hw:name")
    hardware_names = []
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'hardware' in data and 'component' in data['hardware']:
        for name in data['hardware']['component']:
            hardware_names.append(name['name'])
    return hardware_names


def get_bbf_forwarding_table(nc):
    """ Used to get all BBF L2 forwarder entries

    Args:
        nc (obj): Netconf connection to use for this request.

    Returns:
        dict: It returns a python dict representation of the xml data.

    """
    forwarding_table = []
    data_xml=f"""<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
                    message-id="">
                  <get>
                    <filter type="subtree">
                      <bbf-l2-fwd:forwarding xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding">
                        <bbf-l2-fwd:forwarders/>
                      </bbf-l2-fwd:forwarding>
                    </filter>
                  </get>
                </rpc>"""
    xml = nc.get(data_xml=data_xml, message="/bbf-l2-fwd:forwarding")
    data = get_dict_from_etree(etree.fromstring(xml))
    if data and 'forwarding' in data and 'forwarders' in data['forwarding'] and 'forwarder' in data['forwarding']['forwarders']:
        if isinstance(data['forwarding']['forwarders']['forwarder'], list):
            forwarding_table = data['forwarding']['forwarders']['forwarder']
        else:
            forwarding_table.append(data['forwarding']['forwarders']['forwarder'])
    else:
        data = None
    return forwarding_table


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False,
                        help="NETCONF Server IP address or hostname.")
    parser.add_argument("--olt", action="store", dest="olt", default=None, help="olt mac address")
    parser.add_argument("--olt_port", action="store", dest="olt_port", default=None, help="olt device port")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False,
                        help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False,
                        help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False,
                        help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Create a connection to the Netconf server.
    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Initialize variables
    olt_name = None
    olt_mac = None
    olt_port = None

    # OLT MAC address or switch port must be specified
    if not args.olt and not args.olt_port:
        print("ERROR: No OLT specified")
        sys.exit(1)

    # Get a list of hardware names and scan for the OLT device
    # This is done by parsing the OLT MAC address or Switch port from the
    # hardware entry name.
    #   OLT name format: XGS-PON OLT e8:b4:70:70:0c:9c [1/0/1]
    hardware_list = get_ietf_hardware_names(nc)
    for i in hardware_list:
        if "OLT" in i and args.olt and args.olt in i:
            olt_name = i
            olt_mac = args.olt
            olt_port = i.split(" ")[3].strip("[]")
        elif "OLT" in i and args.olt_port and args.olt_port in i:
            olt_name = i
            olt_mac = i.split(" ")[2]
            olt_port = args.olt_port

    # Return an error if no OLT was found
    if not olt_mac or not olt_port:
        print("ERROR: OLT does not exist")
        sys.exit(1)

    #
    # Gather OLT MicroPlug device status
    #

    # Get the ietf-hardware entry for this OLT device
    hardware_interface =  get_ietf_hardware_by_name(nc, olt_name)
    # Update the values used in the final output
    if hardware_interface:
        hardware_interface['xcvr-temp'] = int(hardware_interface['transceiver']['diagnostics']['temperature'])/1000
        tx_power_mW = float(hardware_interface['transceiver-link']['diagnostics']['tx-power']) / 10.0
        hardware_interface['tx-power'] = float(mw_to_dbm(tx_power_mW))
        rx_power_mW = float(hardware_interface['transceiver-link']['diagnostics']['rx-power']) / 10.0
        hardware_interface['rx-power'] = float(mw_to_dbm(rx_power_mW))
        hardware_interface['laser-temp'] = int(hardware_interface['transceiver-link']['diagnostics']['laser-temperature'])/1000
    else:
        # Entry is missing for this device, fill in a blank entry for display below
        hardware_interface = {
            'description': "",
            'hardware-rev': "",
            'firmware-rev': "",
            'serial-num': "",
            'mfg-name': "",
            'model-name': "",
            'xcvr-temp': 0.0,
            'tx-power': 0.0,
            'rx-power': 0.0,
            'laser-temp': 0.0
        }

    #
    # Gather Tibit OLT device to BBF YANG interface mapping entry
    #

    # Get the BBF YANG interface names for this OLT
    olt_interface_map = get_tibit_bbf_olt_interface_map(nc, olt_mac)
    if olt_interface_map and 'nni' in olt_interface_map and 'interface' in olt_interface_map['nni']:
        nni_interface_ref = olt_interface_map['nni']['interface']
    else:
        nni_interface_ref = None
    if olt_interface_map and 'pon' in olt_interface_map and 'channel-group-ref' in olt_interface_map['pon']:
        pon_channel_group_ref = olt_interface_map['pon']['channel-group-ref']
    else:
        pon_channel_group_ref = None
    if olt_interface_map and 'pon' in olt_interface_map and 'channel-pair-ref' in olt_interface_map['pon']:
        pon_channel_pair_ref = olt_interface_map['pon']['channel-pair-ref']
    else:
        pon_channel_pair_ref = None
    if olt_interface_map and 'pon' in olt_interface_map and 'channel-partition-ref' in olt_interface_map['pon']:
        pon_channel_partition_ref = olt_interface_map['pon']['channel-partition-ref']
    else:
        pon_channel_partition_ref = None
    if olt_interface_map and 'pon' in olt_interface_map and 'channel-termination-ref' in olt_interface_map['pon']:
        pon_channel_termination_ref = olt_interface_map['pon']['channel-termination-ref']
    else:
        pon_channel_termination_ref = None


    #
    # Gather NNI Port information
    #

    # Get NNI interface information
    nni_interface = None
    if nni_interface_ref:
        nni_interface = get_ietf_interface_by_name(nc, nni_interface_ref)
    if not nni_interface:
        # Entry is missing for this device, fill in a blank entry for display below
        nni_interface = {
            'oper-status': "",
            'last-change': "",
            'phys-address': "",
        }

    #
    # Gather PON Port information
    #

    # PON Channel Group Config
    pon_channel_group = None
    if pon_channel_group_ref:
        pon_channel_group = get_ietf_interface_by_name(nc, pon_channel_group_ref)
        if pon_channel_group and 'channel-group' in pon_channel_group:
            pon_channel_group = pon_channel_group['channel-group']
    if pon_channel_group:
        # Convert polling period to ms
        pon_channel_group['polling-period'] = int(pon_channel_group['polling-period']) * 100
    else:
        # Entry is missing for this device, fill in a blank entry for display below
        pon_channel_group = {
            'polling-period': 0
        }

    # PON Channel Group State
    pon_channel_group_st = None
    if pon_channel_group_ref:
        pon_channel_group_st = get_ietf_interface_state_by_name(nc, pon_channel_group_ref)
        if pon_channel_group_st and 'channel-group' in pon_channel_group_st and 'pon-pools' in pon_channel_group_st['channel-group'] and 'pon-pool' in pon_channel_group_st['channel-group']['pon-pools']:
            pon_channel_group_st = pon_channel_group_st['channel-group']
            # Flatten pon-pool resource info into the pon_channel_group_st for easier access
            if 'pon-pools' in pon_channel_group_st and 'pon-pool' in pon_channel_group_st['pon-pools']:
                pon_channel_group_st['consumed-alloc-id-values'] = pon_channel_group_st['pon-pools']['pon-pool']['consumed-resources']['alloc-id-values']
                pon_channel_group_st['consumed-gemport-values'] = pon_channel_group_st['pon-pools']['pon-pool']['consumed-resources']['gemport-values']
                pon_channel_group_st['consumed-onu-ids'] = pon_channel_group_st['pon-pools']['pon-pool']['consumed-resources']['onu-ids']
                pon_channel_group_st['available-alloc-id-values'] = pon_channel_group_st['pon-pools']['pon-pool']['available-resources']['alloc-id-values']
                pon_channel_group_st['available-gemport-values'] = pon_channel_group_st['pon-pools']['pon-pool']['available-resources']['gemport-values']
                pon_channel_group_st['available-onu-ids'] = pon_channel_group_st['pon-pools']['pon-pool']['available-resources']['onu-ids']
    if not pon_channel_group_st:
        # Entry is missing for this device, fill in a blank entry for display below
        pon_channel_group_st = {
            'consumed-alloc-id-values': "",
            'consumed-gemport-values': "",
            'consumed-onu-ids': "",
            'available-alloc-id-values': "",
            'available-gemport-values': "",
            'available-onu-ids': ""
        }

    # PON Channel Partition Config
    pon_channel_partition = None
    if pon_channel_partition_ref:
        pon_channel_partition = get_ietf_interface_by_name(nc, pon_channel_partition_ref)
        if pon_channel_partition and 'channel-partition' in pon_channel_partition:
            pon_channel_partition = pon_channel_partition['channel-partition']
    if pon_channel_partition:
        # Map BBF YANG fiber distance to 'Standard' or 'Extended'
        if pon_channel_partition['closest-onu-distance'] == "20" and pon_channel_partition['maximum-differential-xpon-distance'] == "20":
            pon_channel_partition['fiber-distance'] = "Extended(20..40 km)"
        else:
            pon_channel_partition['fiber-distance'] = "Standard(0..20 km)"
    else:
        # Entry is missing for this device, fill in a blank entry for display below
        pon_channel_partition = {
            'downstream-fec': "",
            'closest-onu-distance': "",
            'maximum-differential-xpon-distance': "",
            'fiber-distance': "",
        }

    # PON Channel Termination Config
    pon_channel_termination = None
    if pon_channel_termination_ref:
        pon_channel_termination = get_ietf_interface_by_name(nc, pon_channel_termination_ref)
    if not pon_channel_termination:
        # Entry is missing for this device, fill in a blank entry for display below
        pon_channel_termination = {
            'enabled': "",
            'oper-status': "",
            'last-change': "",
            'phys-address': "",
        }

    # PON Channel termination State
    onu_states = []
    pon_channel_termination_st = None
    if pon_channel_termination_ref:
        pon_channel_termination_st = get_ietf_interface_state_by_name(nc, pon_channel_termination_ref)
        if pon_channel_termination_st and 'channel-termination' in pon_channel_termination_st:
            pon_channel_termination_st = pon_channel_termination_st['channel-termination']
            # Parse ONU Registration Status
            if 'onus-present-on-local-channel-termination' in pon_channel_termination_st:
                if isinstance(pon_channel_termination_st['onus-present-on-local-channel-termination']['onu'], list):
                    onus = pon_channel_termination_st['onus-present-on-local-channel-termination']['onu']
                else:
                    onus = [pon_channel_termination_st['onus-present-on-local-channel-termination']['onu']]
                for onu in onus:
                    if 'v-ani-ref' not in onu:
                        onu['v-ani-ref'] = 'Unknown'
                    if 'onu-id' not in onu:
                        onu['onu-id'] = '-'
                    # Strip the namepace off the state
                    onu['onu-presence-state'] = onu['onu-presence-state'].split(':')[1]
                    onu_states.append(onu)
            # If there are no 'active-defects' insert a blank.
            if 'active-defects' not in pon_channel_termination_st:
                pon_channel_termination_st['active-defects'] = ""
    if not pon_channel_termination_st:
        # Entry is missing for this device, fill in a blank entry for display below
        pon_channel_termination_st = {
            'pon-id-display': "",
            'active-defects': "",
        }

    #
    # Gather Forwarding Table information (VLAN Tags)
    #

    nni_networks = []
    nni_network_onus = {}

    # Get the information needed for the forwarding table
    for forwarder in get_bbf_forwarding_table(nc):
        nni_network_onus[forwarder['name']] = []

        # Get the list of forwarder ports
        if 'ports' in forwarder and 'port' in forwarder['ports']:
            if not isinstance(forwarder['ports']['port'], list):
                forwarder_ports = [forwarder['ports']['port']]
            else:
                forwarder_ports = forwarder['ports']['port']
        else:
            forwarder_ports = []

        # Gather VLAN tagging information from NNI-side forwarder ports
        nni_networks_num = 0
        for port in forwarder_ports:
            # Get the VLAN sub-interface for this forwarder port
            vlan_sub_interface = get_ietf_interface_by_name(nc, port['sub-interface'])

            if not vlan_sub_interface:
                continue

            # Determine if this is an NNI or PON-side vlan_sub_interface
            if 'interface-usage' in vlan_sub_interface and 'interface-usage' in vlan_sub_interface['interface-usage']:
                interface_usage = vlan_sub_interface['interface-usage']['interface-usage']
            else:
                # Assume user-port (i.e., PON-side if not specified)
                interface_usage = 'user-port'

            # Only process NNI-side interfaces in this loop
            if interface_usage != 'network-port':
                continue

            # Verify this VLAN sub-interface is on this OLT
            if vlan_sub_interface['subif-lower-layer']['interface'] != nni_interface_ref:
                continue

            # VLAN Tags can be parsed from the NNI vlan_sub_interface
            olt_tag = "untagged"
            onu_tag = "untagged"
            vlan_tags = vlan_sub_interface['inline-frame-processing']['ingress-rule']['rule']['flexible-match']['match-criteria']['tag']
            if isinstance(vlan_tags, dict):
                vlan_tags = [vlan_tags]
            for index, tag in enumerate(vlan_tags):
                # Determine the TPID for this tag
                if tag['dot1q-tag']['tag-type'] == 'bbf-dot1qt:s-vlan':
                    tpid = 's-vlan'
                else:
                    tpid = 'c-vlan'

                # Assume the outer vlan is the OLT tag and the inner vlan is the ONU tag
                if index == 0:
                    olt_tag = tpid + "/" + str(tag['dot1q-tag']['vlan-id'])
                else:
                    onu_tag = tpid + "/" + str(tag['dot1q-tag']['vlan-id'])
            nni_networks.append(
                {
                    'name': forwarder['name'],
                    'olt_tag': olt_tag,
                    'onu_tag': onu_tag
                }
            )
            nni_networks_num += 1

        # Stop here if there are no applicable NNI Networks
        if nni_networks_num == 0:
            continue

        # Gather the list of ONUs from PON-side forwarder ports
        for port in forwarder_ports:
            # Get the VLAN sub-interface for this forwarder port
            vlan_sub_interface = get_ietf_interface_by_name(nc, port['sub-interface'])

            if not vlan_sub_interface:
                continue

            # Determine if this is an NNI or PON-side vlan_sub_interface
            if 'interface-usage' in vlan_sub_interface and 'interface-usage' in vlan_sub_interface['interface-usage']:
                interface_usage = vlan_sub_interface['interface-usage']['interface-usage']
            else:
                # Assume user-port (i.e., PON-side if not specified)
                interface_usage = 'user-port'

            if interface_usage == 'user-port':
                # Attached ONUs can be parsed from PON-size vlan_sub_interfaces
                oltvenet_name = vlan_sub_interface['subif-lower-layer']['interface']
                vani_name = get_ietf_interface_by_name(nc, oltvenet_name)['olt-v-enet']['lower-layer-interface']
                onu = get_ietf_interface_by_name(nc, vani_name)['v-ani']['expected-serial-number']
                nni_network_onus[forwarder['name']].append(onu)


    #
    # Display OLT Information
    #

    print("")
    print("OLT Device Info")
    print(f"  device id:                 {olt_mac}")
    print(f"  switch port:               {olt_port}")
    print(f"  description:               {hardware_interface['description']}")
    print(f"  operational status:        {nni_interface['oper-status']}")
    print(f"  last change:               {nni_interface['last-change']}")
    print(f"  hardware revision:         {hardware_interface['hardware-rev']}")
    print(f"  firmware revision:         {hardware_interface['firmware-rev']}")
    print(f"  serial number:             {hardware_interface['serial-num']}")
    print(f"  manufacturer:              {hardware_interface['mfg-name']}")
    print(f"  model:                     {hardware_interface['model-name']}")
    print(f"  transceiver temperature:   {hardware_interface['xcvr-temp']:4.2f}°C")
    print(f"  laser:")
    print(f"    tx power:                {hardware_interface['tx-power']:7.2f} dBm")
    print(f"    idle rx power:           {hardware_interface['rx-power']:7.2f} dBm")
    print(f"    temperature:             {hardware_interface['laser-temp']:4.2f}°C")
    print("")
    print("NNI Port")
    print(f"  mac address:               {nni_interface['phys-address']}")
    print("")
    print("PON Port")
    print(f"  admin status:              {pon_channel_termination['enabled']}")
    print(f"  operational status:        {pon_channel_termination['oper-status']}")
    print(f"  last change:               {pon_channel_termination['last-change']}")
    print(f"  mac address:               {pon_channel_termination['phys-address']}")
    print(f"  fec:                       {pon_channel_partition['downstream-fec']}")
    print(f"  polling period:            {pon_channel_group['polling-period']} ms")
    print(f"  fiber reach:               {pon_channel_partition['fiber-distance']}")
    print(f"  pon-id:                    {pon_channel_termination_st['pon-id-display']}")
    print(f"  active defects:            {pon_channel_termination_st['active-defects']}")
    print("  pon resources")
    print("    consumed")
    print(f"      alloc-id values:       {pon_channel_group_st['consumed-alloc-id-values']}")
    print(f"      gemport values:        {pon_channel_group_st['consumed-gemport-values']}")
    print(f"      onu-ids:               {pon_channel_group_st['consumed-onu-ids']}")
    print("    available")
    print(f"      alloc-id values:       {pon_channel_group_st['available-alloc-id-values']}")
    print(f"      gemport values:        {pon_channel_group_st['available-gemport-values']}")
    print(f"      onu-ids:               {pon_channel_group_st['available-onu-ids']}")
    print("")
    print("BBF Interfaces")
    print(f"  nni:                       {nni_interface_ref}")
    print(f"  pon channel group:         {pon_channel_group_ref}")
    print(f"  pon channel pair:          {pon_channel_pair_ref}")
    print(f"  pon channel partition:     {pon_channel_partition_ref}")
    print(f"  pon channel termination:   {pon_channel_termination_ref}")
    print("")
    print("ONU States")
    # Print table header row
    print("  "
            f"{'ONU ID': <8}"
            f"{'Serial Number': <16}"
            f"{'V-ANI': <24}"
            f"{'ONU Status '}")
    # Display ONU entries sorted by ONU ID
    for onu in sorted(onu_states, key=nat_sort('onu-id')):
        print("  "
                f"{onu['onu-id']: <8}"
                f"{onu['detected-serial-number']: <16}"
                f"{onu['v-ani-ref']: <24}"
                f"{onu['onu-presence-state']}")
    print("")
    print("NNI Networks")
    # Print table header row
    print("  "
            f"{'OLT Tag': <14}"
            f"{'ONU Tag': <14}"
            f"{'BBF Forwarder': <32}"
            f"{'ONUs'}")
    # Display NNI Network entries sorted by OLT Tag + ONU Tag
    for nni_network in sorted(nni_networks, key=nat_sort(['olt_tag', 'onu_tag'])):
        name = nni_network['name']
        if nni_network_onus[name]:
            onu = nni_network_onus[name][0]
        else:
            onu = ""
        print("  "
                f"{nni_network['olt_tag']: <14}"
                f"{nni_network['onu_tag']: <14}"
                f"{name: <32}"
                f"{onu}")
        # Print remaining ONUs
        if len(nni_network_onus[name]) > 1:
            for onu in nni_network_onus[name][1:]:
                print("  "
                    f"{' ': <60}"
                    f"{onu}")
