#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
#--------------------------------------------------------------------------#

''' Tibit MCMS PON Manager Database Import Utility.

The MCMS PON Manager Database Import Utility is a Python script used to
import device firmware, pictures, and ONU Service Configuration (SRV-CFG)
files into MongoDB and the PON Manager database initialization seed file
datastore (/opt/tibit/ponmgr/api/databaseSeedFiles). This tool is similar
to the JsonToMongo utility provided with MCMS PON Controller.

Note: Root access is requires to run this script.


--------------
Database Files
--------------

The script requires a 'directory' argument, which is a path to a directory
containing the set of database files for import. The file names must comply
with the formats listed below in order for  files to be imported by the
script. Files without the proper name formats are ignored.

ONU Service Configuration Files (SRV-CFG):

ONU Service configuration file names have the following format:
SRV-<Info>-CFG.json, where <Info> typically contains the Vendor name and
type of service provided by the SRV-CFG file. For example, SRV-BFWS-AddCTag-CFG.json;
SRV-DPOE-AddCTag-CFG.json)

Tibit OLT and ONU Firmware:

OLT firmware file names have the following format: <Version>-<device>-FW.bin, where
<Version> is the version of the firmware contained in the file, and <device> is 'OLT'
or 'ONU. For example, R2.2.0-OLT-FW.bin; R2.2.0-ONU-FW.bin.

Third Party ONU Firmware:

ONU firmware file names from third party vendors have the following format:
FW-<PON Type>-<Vendor>-<Model>-<Version>.bin, where
  <PON Type> is the type of PON: 'GPON' or 'EPON',
  <Vendor> is the four character vendor code that represents the manufacturer of the ONU,
  <Model> is the ONU model number for this firmware image,
  <Version> is the version of the firmware contained in this file,
For example: FW-GPON-BFWS-150A-3.2.30.bin

Picture Files:

Device picture file names have the following format:
PIC-<Device Type>-<Vendor>-<Model>.jpg, where
  <Device Type> is the type of device: CNTL, OLT, ONU, or SWI
  <Vendor> is a string that represents the manufacturer of the device,
  <Model> is the device model number for this picture,
For example: PIC-ONU-BFWS-150A.jpg


------------
Script Usage
------------

Example - Import files into both MongoDB and PON Manager database seed file datastore:
  sudo ./db_import.py --db mongodb://127.0.0.1/tibit_pon_controller ./database_files

Example - Import files into MongoDB only:
  sudo ./db_import.py --db mongodb://127.0.0.1/tibit_pon_controller --no-update-seed-files ./database_files

Example - Import files into the PON Manager database seed file datastore only:
  sudo ./db_import.py ./database_files


usage: db_import.py [-h] [-d DB_URI] [--update-seed-files] [--no-update-seed-files] <directory>

positional arguments:
  directory             Directory of files to import.

optional arguments:
  -h, --help            show this help message and exit
  -d DB_URI, --db DB_URI
                        Import files into the specified database (e.g.,
                        mongodb://127.0.0.1/tibit_pon_controller). (default:
                        None)
  --update-seed-files   Update the PON Manager database initialization seed
                        files (default). (default: True)
  --no-update-seed-files
                        Do not update the PON Manager database initialization
                        seed files. (default: True)

'''

import sys
import os
import os.path
import argparse
import json
import shutil
import traceback
import gridfs
import pymongo
from bson.binary import Binary

PM_SEED_FILES_PATH='/opt/tibit/ponmgr/api/databaseSeedFiles'
PM_METADATA_FILE = PM_SEED_FILES_PATH + '/metadata.json'

# This script requires root
if os.geteuid() != 0:
    sys.exit("Root level access is required. Please use 'sudo' to run the db_import tool.")

#######################################
# Get a list of the ONU configuration files
#######################################

def get_onu_service_config_list(database_dir):
    """Get a list of service config files in the database"""
    files = []
    for filename in os.listdir(database_dir):
        #print "Filename is " + filename
        if filename.startswith("SRV-") and filename.endswith("-CFG.json"):
            #print "Qualified Filename is " + filename
            files.append(filename)

    files.sort()
    return files


#######################################
# Get a list of the OLT firmware files
#######################################

def get_olt_firmware_list(database_dir):
    """Get a list of firmware files in the database"""
    files = []
    for filename in os.listdir(database_dir):
        if filename.endswith("-OLT-FW.bin"):
            #print "Qualified Filename is " + filename
            files.append(filename)

    files.sort()
    return files

#######################################
# Get a list of the ONU firmware files
#######################################

def get_onu_firmware_list(database_dir, verbose=False):
    """Get a list of firmware files in the database"""

    # Tibit Files
    files = {}
    for filename in os.listdir(database_dir):
        # Tibit Files
        if filename.endswith("-ONU-FW.bin"):
            files[filename] = {}
            files[filename]["Version"] = filename.rstrip("-ONU-FW.bin")
            files[filename]["Compatible Manufacturer"] = "TIBITCOM"
            files[filename]["Compatible Model"] = ["MicroPlug ONU"]

        # Alpha Files
        if filename.startswith("bcm34000_SFU_nand"):
            if filename == "bcm34000_SFU_nand_fs_image_128_ubi_5.02L.05_007-190828_1408.w":
                files[filename] = {}
                files[filename]["Compatible Manufacturer"] = "ALPH"
                files[filename]["Compatible Model"] = ["34000"]
                files[filename]["Version"] = "5025_007_SFU24"
            elif filename == "bcm34000_SFU_nand_fs_image_128_ubi_5.02L.05_004-181212_1349.w":
                files[filename] = {}
                files[filename]["Compatible Manufacturer"] = "ALPH"
                files[filename]["Compatible Model"] = ["34000"]
                files[filename]["Version"] = "5025_003_SFU"
            elif filename == "bcm34000_SFU_nand_fs_image_128_ubi__5.02L.05_002-181019_1359.w":
                files[filename] = {}
                files[filename]["Compatible Manufacturer"] = "ALPH"
                files[filename]["Compatible Model"] = ["34000"]
                files[filename]["Version"] = "5025_002_SFU29"
            else:
                print("ERROR: Unknown Alpha Firmware {}".format(filename))

        # BFW Files
        if filename.startswith("WAC5-SR"):
            files[filename] = {}
            files[filename]["Compatible Manufacturer"] = "BFWS"
            files[filename]["Compatible Model"] = ["ONT205"]
            pieces = filename.lstrip("WAC5-SR").split("-")
            files[filename]["Version"] = str(pieces[0])

        # ARCN Files
        if filename.startswith("PRV6505B-ZZ_"):
            files[filename] = {}
            files[filename]["Compatible Manufacturer"] = "ARCN"
            files[filename]["Compatible Model"] = ["PRV6505B-ZZ"]
            files[filename]["Version"] = str(filename.lstrip("PRV6505B-ZZ_").rstrip("_build47_128_debug_pureubi.w"))

        # Generic Format
        if filename.startswith("FW-GPON-") or filename.startswith("FW-EPON-"):
            files[filename] = {}
            files[filename]["Compatible Manufacturer"]  = str(filename[8:12])
            middle = str(filename[13:].rstrip(".bin"))
            middle = middle.split("-")
            files[filename]["Compatible Model"]         = [str(middle[0])]
            files[filename]["Version"]                  = str(middle[1])
            if len(middle) > 2:
                files[filename]["Version"]              = str(middle[1]) + "-" + str(middle[2])
            else:
                files[filename]["Version"]              = str(middle[1])

        if verbose and filename in files:
            print(files[filename])

    return files

#######################################
# Get a list of the picture files
#######################################

def get_picture_file_list(database_dir):
    """Get a list of picture files in the database"""
    files = []
    for filename in os.listdir(database_dir):
        if filename.endswith(".png") or filename.endswith(".svg") or filename.endswith(".jpg"):
            files.append(filename)

    files.sort()
    return files

#######################################################
#   Get the ONU Service Configuration File
#######################################################

def onu_get_srv_cfg(database_dir, filename):
    """Gets for the ONU Service Configuration from the file"""
    onu_srv_cfg = None

    file_path = database_dir + "/" + filename

    # Check if the file exists
    exists = os.path.isfile(file_path)
    if exists is True:
        # Open the file to get the information into a dictionary
        with open(file_path, 'rt') as file_object:
            onu_srv_cfg = json.load(file_object)
    else:
        # Fail and Quit
        print("ERROR: ONU Config File Missing: " + filename)
        sys.exit(0)

    return onu_srv_cfg

#############################################
# Replace a document in MongoDB
#############################################

def db_replace(mongo_db, collection_name, doc_id, doc):
    """Replace a document in the database."""
    rc = True

    # Insert the given document into the collection defined by device, collection_type, and/or deviceId
    try:
        collection = mongo_db[collection_name]
        result = collection.replace_one({"_id": doc_id}, doc, upsert=True)

        if result.acknowledged and (result.matched_count > 0 or result.upserted_id != None):
            if result.matched_count:
                print("Replaced {} in MongoDB collection {}.".format(doc_id, collection_name))
            else:
                print("Inserted {} into MongoDB collection {}.".format(doc_id, collection_name))
        else:
            print("ERROR: Collection {} replace document {} in MongoDB failed.".format(collection_name, doc_id))
            rc = False

    except Exception as err:
        print("ERROR: Collection {} replace document {} in MongoDB failed.".format(collection_name, doc_id))
        print(err)
        rc = False

    return rc

#############################################
# Replace a file in MongoDB
#############################################

def db_replace_file(mongo_db, collection, doc_id, database_dir, file_name, metadata):
    """Replace a GridFS file in the database."""
    rc = True

    # Delete previous file version
    try:
        fs = gridfs.GridFSBucket(db=mongo_db, bucket_name=collection)
        exists = fs.find({'_id': doc_id}).count() != 0
        if exists:
            fs.delete(file_id=doc_id)

        file_path = database_dir + "/" + file_name

        # Open a stream to the file on the filesystem
        fileContent = Binary(open(file_path, mode='rb').read())

        # Import the file into the DB
        grid = fs.open_upload_stream_with_id(file_id=doc_id, filename=file_name, metadata=metadata)
        grid.write(fileContent)
        grid.close()
        if exists:
            print("Replaced {} in MongoDB collection {}.".format(file_name, collection))
        else:
            print("Inserted {} into MongoDB collection {}.".format(file_name, collection))

    except Exception:
        print(f"ERROR: Failed to write file '{file_name}' to MongoDB collection '{collection}'.")
        print(traceback.format_exc())
        rc = False

    return rc

##########################################################
# Update PON Manager database metadata for firmware
##########################################################

def pm_update_firmware_metadata(device_type, fw_file_name, metadata):
    """Update PON Manager metadata for a firmware image file."""
    rc = True

    # Construct the key
    if device_type == "OLT":
        key = f"firmwares/olts/{fw_file_name}"
    elif device_type == "ONU":
        key = f"firmwares/onus/{fw_file_name}"

    # Update the PM metadata.json
    try:
        with open(PM_METADATA_FILE, 'r') as json_f:
            json_data = json.load(json_f)
        json_data[key] = metadata
        with open(PM_METADATA_FILE, 'w') as json_f:
            json.dump(json_data, json_f)
        print(f"Updated firmware metadata for '{fw_file_name}'.")
    except Exception:
        print(f"ERROR: Failed to update {PM_METADATA_FILE} for {fw_file_name}.")
        print(traceback.format_exc())
        rc = False

    return rc

##########################################################
# Update PON Manager database metadata for pictures
##########################################################

def pm_update_picture_metadata(device_type, pic_file_name, metadata):
    """Update PON Manager metadata for a picture file."""
    rc = True

    # Construct the key
    if device_type == "CNTL":
        key = f"pictures/controllers/{pic_file_name}"
    elif device_type == "OLT":
        key = f"pictures/olts/{pic_file_name}"
    elif device_type == "ONU":
        key = f"pictures/onus/{pic_file_name}"
    elif device_type == "SWI":
        key = f"pictures/switches/{pic_file_name}"

    # Update the PM metadata.json
    try:
        with open(PM_METADATA_FILE, 'r') as json_f:
            json_data = json.load(json_f)
        json_data[key] = metadata
        with open(PM_METADATA_FILE, 'w') as json_f:
            json.dump(json_data, json_f)
        print(f"Updated picture metadata for '{pic_file_name}'.")
    except Exception:
        print(f"ERROR: Failed to update {PM_METADATA_FILE}for {pic_file_name}.")
        print(traceback.format_exc())
        rc = False

    return rc

##########################################################
# Import the database seed file into PON Manager
##########################################################

def pm_import_database_seed_file(collection, device_type, import_dir, file_name):
    """Copy a file into the PON Manager database initialization file store."""
    rc = True

    src = f"{import_dir}/{file_name}"
    if collection == "SRV-CFG":
        dst = f"{PM_SEED_FILES_PATH}/documents/srvConfigs/{file_name}"
    elif collection == "OLT-FIRMWARE":
        dst = f"{PM_SEED_FILES_PATH}/firmwares/olts/{file_name}"
    elif collection == "ONU-FIRMWARE":
        dst = f"{PM_SEED_FILES_PATH}/firmwares/onus/{file_name}"
    elif collection == "PICTURES":
        if device_type == "CNTL":
            dst = f"{PM_SEED_FILES_PATH}/pictures/controllers/{file_name}"
        elif device_type == "OLT":
            dst = f"{PM_SEED_FILES_PATH}/pictures/olts/{file_name}"
        elif device_type == "ONU":
            dst = f"{PM_SEED_FILES_PATH}/pictures/onus/{file_name}"
        elif device_type == "SWI":
            dst = f"{PM_SEED_FILES_PATH}/pictures/switches/{file_name}"

    # Copy the file into the PON Manager database seed files.
    try:
        shutil.copy(src, dst)
        print(f"Imported seed file '{file_name}'.")
    except Exception:
        print(f"ERROR: Failed to copy seed file '{file_name}' to '{dst}'.")
        print(traceback.format_exc())
        rc = False

    return rc

#############################################
# Main
#############################################

def main():
    # Example command line invocation:
    #
    # sudo ./db_import.py --db mongodb://127.0.0.1/tibit_pon_controller ./database_files
    #
    parser = argparse.ArgumentParser(add_help=True,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="db_uri", default=None, required=False, help="Import files into the specified database (e.g., mongodb://127.0.0.1/tibit_pon_controller).")
    parser.add_argument(      "--update-seed-files", action="store_true", dest="update_seed_files", required=False, help="Update the PON Manager database initialization seed files (default).")
    parser.add_argument(      "--no-update-seed-files", action="store_false", dest="update_seed_files", required=False, help="Do not update the PON Manager database initialization seed files.")
    parser.add_argument('directory', nargs=argparse.REMAINDER, type=str, help="Directory of files to import.")
    parser.set_defaults(update_seed_files=True)
    parser.parse_args()
    args = parser.parse_args()

    # Verify the import directory is specified
    if len(args.directory) < 1:
        print("ERROR: Missing the 'directory' to import.")
        print()
        parser.print_help()
        sys.exit(1)

    # JSON database is passed in on the command line
    database_dir = os.path.realpath(args.directory[0])

    # Verify the specified path exists and is a directory.
    if not os.path.isdir(database_dir):
        print(f"ERROR: Specified path '{args.directory[0]}'' does not exist or is not a directory.")
        print()
        parser.print_help()
        sys.exit(1)

    # Setup the MongoDB client if --db was specified on the command line
    mongo_db = None
    if args.db_uri:
        try:
            # Parse and Validate the URI
            db_uri_parsed = pymongo.uri_parser.parse_uri(args.db_uri)
            mongo_host = db_uri_parsed['nodelist'][0][0]
            mongo_port = db_uri_parsed['nodelist'][0][1]
            mongo_db_name = db_uri_parsed['database']
        except ValueError as error:
            print(f"ERROR: Invalid Mongo Database Connection URI '{args.db_uri}', error {error}.")
            print()
            sys.exit(1)

        # Connect to the MongoDB Server as specified in the mongodb:// connection URI
        mongo_client = pymongo.MongoClient(args.db_uri)
        # print(f"{db_client}")

        # Get a handle to the database reference in the mongodb:// connection URI
        mongo_db = mongo_client.get_default_database()
        # print(f"{db}")

        print(f"Importing files from {args.directory[0]} into Mongo Database '{mongo_db_name}' on host {mongo_host}, port {mongo_port}.")
    else:
        print("No database files updated.")

    if args.update_seed_files:
        print(f"Importing files from {args.directory[0]} into PON Manager database initialization seed file store.")
    else:
        print("No PON Manager database initialization seed files updated.")

    if not args.db_uri and not args.update_seed_files:
        # Nothing to do.
        print()
        print("No files imported.")
        sys.exit(1)


    #########################################
    # Import ONU Service Configuration Files
    #########################################

    # Get the list of service configuration files
    onu_srv_cfg_files = get_onu_service_config_list(database_dir)
    print("Found " + str(len(onu_srv_cfg_files)) + " ONU Configuration Files")
    # print(onu_srv_cfg_files)

    # Loop over the config files
    for onu_srv_cfg_file in onu_srv_cfg_files:
        # Get the config from the file
        onu_srv_cfg = onu_get_srv_cfg(database_dir, onu_srv_cfg_file)

        # Rewrite the config file
        if mongo_db:
            doc_id = onu_srv_cfg_file.replace("SRV-", "").replace("-CFG.json", "")
            db_replace(mongo_db, "SRV-CFG", doc_id, onu_srv_cfg)

        # Import the file into the PON Manager database seed files directory
        if args.update_seed_files:
            pm_import_database_seed_file("SRV-CFG", "ONU", database_dir, onu_srv_cfg_file)


    #########################################
    # Import OLT firmware images
    #########################################
    firmware_files = get_olt_firmware_list(database_dir)

    for firmware_file in firmware_files:
        doc_id = firmware_file.rstrip(".bin")

        # Build meta data
        if firmware_file[:5] == "A1.0.":
            models = ["180713", "171205"]
        else:
            models = ["180713"]
        metadata = {"Version": firmware_file.rstrip("-OLT-FW.bin"), "Compatible Manufacturer": "TIBITCOM", "Compatible Model": models}

        # (Re)add the file to the DB
        if mongo_db:
            db_replace_file(
                mongo_db=mongo_db,
                collection="OLT-FIRMWARE",
                doc_id=firmware_file.rstrip(".bin"),
                database_dir=database_dir,
                file_name=firmware_file,
                metadata=metadata)

        # Update the PON Manager database initialization file store.
        if args.update_seed_files:
            # Import the file into the PON Manager database seed files directory
            pm_import_database_seed_file("OLT-FIRMWARE", "OLT", database_dir, firmware_file)

            # Update the PON Manager seed file metadata.json file
            pm_update_firmware_metadata("OLT", firmware_file, metadata)

    #########################################
    # Import ONU firmware images
    #########################################
    firmware_files = get_onu_firmware_list(database_dir)

    for firmware_file, metadata in firmware_files.items():

        # (Re)add the file to the DB
        if mongo_db:
            db_replace_file(
                mongo_db=mongo_db,
                collection="ONU-FIRMWARE",
                doc_id=firmware_file,
                database_dir=database_dir,
                file_name=firmware_file,
                metadata=metadata)

        # Update the PON Manager database initialization file store.
        if args.update_seed_files:
            # Import the file into the PON Manager database seed files directory
            pm_import_database_seed_file("ONU-FIRMWARE", "ONU", database_dir, firmware_file)

            # Update the PON Manager seed file metadata.json file
            pm_update_firmware_metadata("ONU", firmware_file, metadata)

    #########################################
    # Import device pictures
    #########################################
    image_files = get_picture_file_list(database_dir)

    for image_file in image_files:

        if image_file.endswith(".png"):
            doc_id = image_file[:-4]
        elif image_file.endswith(".jpg"):
            doc_id = image_file[:-4]
        elif image_file.endswith(".svg"):
            doc_id = image_file[:-4]
        else:
            doc_id = image_file

        #Determining metadata
        metadataArray = doc_id.split("-")
        metadata = {"Device Type": metadataArray[1].upper()}

        # (Re)add the file to the DB
        if mongo_db:
            db_replace_file(
                mongo_db=mongo_db,
                collection="PICTURES",
                doc_id=doc_id,
                database_dir=database_dir,
                file_name=image_file,
                metadata=metadata)

        # Update the PON Manager database initialization file store.
        if args.update_seed_files:
            # Import the file into the PON Manager database seed files directory
            pm_import_database_seed_file("PICTURES", metadataArray[1].upper(), database_dir, image_file)

            # Update the PON Manager seed file metadata.json file
            pm_update_picture_metadata(metadataArray[1].upper(), image_file, metadata)

if __name__ == '__main__':
    main()
