#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
# --------------------------------------------------------------------------#

''' Tibit Bulk Configuration Utility.

The Tibit Bulk Configuration Utility is a Python script that parses a CSV file
which contains provisioning data for one or more OLT or ONU devices. The script
creates or updates entries in MongoDB for each device row in the CSV file.


===============
CSV File Format
===============

The CSV file contains a header row followed by one or more rows of device
configuration data, where each row of the file represents configuration for a
single device. Each column in the header row represents a database atttribute
and is represented by the format: "[collection][field1][field2]".

For example, the following header column identifies the Address field in the OLT
Configuration collection in the database (OLT-CFG).

Header Column: [OLT-CFG][OLT][Address]

MongoDB Document:

    OLT-CFG = {
        "OLT" : {
            "Address" : "Petaluma, CA",
        }
    }


------------------
Device Identifiers
------------------

The first column must contain the Device ID. The Device ID is used as the key to
identify a specific document in MongoDB, and maps to the _id field in a MongoDB
document.

First Header Column: [OLT-CFG][_id]


------
Arrays
------

Arrays (or lists) are represented in a header column by an integer enclosed in
brackets, where the integer value reprents the index in to the array (e.g.,
array[0]).

For example, the following header columns reference firmware bank '0' and
firmware bank '1' in an OLT Configuration document in the database (OLT-CFG).

Header Column #1: [OLT-CFG][OLT][FW Bank Files][0]

Header Column #2: [OLT-CFG][OLT][FW Bank Files][1]

MongoDB Document:

    OLT-CFG = {
        "OLT" : {
            "FW Bank Files" : [
                "R2.0.0-OLT-FW.bin",
                "R1.3.1-OLT-FW.bin"
            ],
        }
    }


----------------
Attribute Values
----------------

All attribute values are represented as strings in the CSV file. However, values
must be written into the database with the proper JSON datatype. Since the CSV
file does not explicitly define a datatype for attributes, the script follows
specific set of conversion rules when writing values to the database.

The script processes CSV attribute values according to the following ordered
rules:

1) Values enclosed by triple quotes """<value>""" are always considered a string
   regardless of the format of <value>. Triple quoting is the method to add
   quotes around a value in CSV.

Values with no quotes (or enclosed by a single set of quotes) are processed
according to the following order.

2) If the value contains a single decimal point and conversion to a Python
   float() succeeds, the value is inserted into the document as a float.

3) If the value conversion to a Python int() succeeds, the value is inserted
   into the document as a integer. Note if a number must be represented in the
   DB as a string, enclose it in triple quotes. For example, """9600""" will
   insert the value into the database as a string instead of an integer.

4) If the value converted to lower case is equal to the string "true" or
   "false", the value is inserted into the document as a Boolean true or false
   value.

5) If all of the above conversions 2..4 fail, the default operation inserts the
   value into the database as a string.

In addition to the conversion rules listed above, the script applies the
following special cases for blank values:
* Blank cells (i.e., ",,") are skipped and the document is not updated.
* Blank strings can be inserted using triple quotes using the following
format: """""". This is the same as (1) except the <value> is blank and has
length 0.


------------
Script Usage
------------

* NOTE: the user used in the script must have the appropriate permissions to read/create/update/delete the intended resource *

Example - bulk create devices from a CSV file:
  ./bulk_configure.py -u user.email@domain.com -p myPassword1! -f ./olt_bulk_config.csv

Example - bulk create devices from a CSV file, overwriting existing entries if they are already in the database:
  ./bulk_configure.py -u user.email@domain.com -p myPassword1! -f ./olt_bulk_config.csv --overwrite

Example - bulk delete devices listed in a CSV file:
  ./bulk_configure.py -u user.email@domain.com -p myPassword1! -f ./olt_bulk_config.csv --remove


usage: bulk_configure.py [-h] [-d DATABASE_ID] [-l URL] -u USER_EMAIL -p USER_PASSWORD -f FILE
                         [--insert-missing-fields] [--overwrite] [-r] [-v] [-t]

optional arguments:
  -h, --help            show this help message and exit
  -d DATABASE_ID, --db DATABASE_ID
                        ID of the database in the PON Manager REST API server to use
  -l URL, --url URL     URL of the PON Manager REST API server (e.g., https://10.2.10.222/api).
                        (default: https://127.0.0.1/api)
  -u EMAIL, --user EMAIL
                        User email address to authenticate with. (default: None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: None)
  -f FILE, --file FILE  A CSV formatted file containing the device
                        provisioning. (default: None)
  --insert-missing      Force insert fields from the CSV which are are
                        missing from the default document. (default: False)
  --overwrite           Force overwrite the entry if it already exists in the
                        database. (default: False)
  -r, --remove          Remove the device entries from the database. (default:
                        False)
  -t, --test            Test the script without making any changes to the database.
                        (default: False)
  -v, --verbose         Display verbose output. (default: False)

'''

import requests
import re
import urllib.parse
import argparse
import csv
import urllib
import urllib3
import sys
import bson.json_util as json
import json as pyjson

from io import StringIO
from typing import Any, Optional

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)


#
# Logging utility functions
#

# ANSI escape sequences for terminal coloring
class term_colors:
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKCYAN = '\033[96m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'


def ERROR(*args, **kwargs):
    print(f"{term_colors.FAIL}ERROR: ", end='')
    print(*args, **kwargs)
    print(f"{term_colors.ENDC}", end='')


def INFO(*args, **kwargs):
    print(*args, **kwargs)


def WARNING(*args, **kwargs):
    print(f"{term_colors.WARNING}WARNING: ", end='')
    print(*args, **kwargs)
    print(f"{term_colors.ENDC}", end='')


def HIGHLIGHT_ER(condition, *args, **kwargs):
    sio = StringIO()
    if condition:
        print(f"{term_colors.FAIL}{term_colors.BOLD}", end='', file=sio)
    print(*args, **kwargs, end='', file=sio)
    if condition:
        print(f"{term_colors.ENDC}", end='', file=sio)
    return sio.getvalue()


def HIGHLIGHT_OK(condition, *args, **kwargs):
    sio = StringIO()
    if condition:
        print(f"{term_colors.OKGREEN}", end='', file=sio)
    print(*args, **kwargs, end='', file=sio)
    if condition:
        print(f"{term_colors.ENDC}", end='', file=sio)
    return sio.getvalue()


def HIGHLIGHT_WA(condition, *args, **kwargs):
    sio = StringIO()
    if condition:
        print(f"{term_colors.WARNING}", end='', file=sio)
    print(*args, **kwargs, end='', file=sio)
    if condition:
        print(f"{term_colors.ENDC}", end='', file=sio)
    return sio.getvalue()


#
# Utility functions that utilize regular expressions for parsing.
#

RE_MATCH_LIST_INDEX = re.compile(r"\[(\d+)\]$")


def re_parse_elem_list(elem):
    """ Parse the list key and list index from a path element.

    Path elements that represent arrays have the following
    format: "<list_key>[<list_index>]". This function parses the key and
    index from a path element.

    Args:
        elem: An element from a path string.

    Returns:
        list_key: A string identifying the list (or array) attribute name.
        list_index: An integer representing the index into the list.
    """
    list_key = None
    list_index = None
    matches = re.search(RE_MATCH_LIST_INDEX, elem)
    if matches:
        list_index = int(matches.group(1))
        list_key = elem.replace(f"[{list_index}]", "")
    return list_key, list_index


RE_MATCH_BRACKETED_PATH = re.compile(r"\[(.*?)\]")


def re_parse_bracketed_path(path):
    """ Parse a string representing the path to an attributed in a MongoDB document.

    MongoDB documents are a nested python dictionary. The path identifying an attribute
    in the document uses a braketed format: "[collection][node][array][0]"

    Args:
        path: The path identifying an attribute using the bracketed format.

    Returns:
        parsed_path: A list of string represting the elements or individual nodes in the path.
    """
    parsed_path = []
    matches = re.findall(RE_MATCH_BRACKETED_PATH, path)
    for match_value in matches:
        if match_value.isdigit():
            if len(parsed_path) > 0:
                parsed_path[-1] += f"[{match_value}]"
        else:
            parsed_path.append(match_value)

    return parsed_path


RE_MATCH_ISO8601_TIMESTAMP = re.compile(r'^(-?(?:[1-9][0-9]*)?[0-9]{4})-(1[0-2]|0[1-9])-(3[01]|0[1-9]|[12][0-9])T(2[0-3]|[01][0-9]):([0-5][0-9]):([0-5][0-9])(\.[0-9]+)?(Z|[+-](?:2[0-3]|[01][0-9]):[0-5][0-9])?$').match


def re_db_timestamp_is_valid(str_val):
    """ Parse and validate a string representing an PON Controller Timestamp.

    This function converts the PON Controller Timestamp to ISO8601 format before validating it.

    Args:
        str_val: A PON Controller timestamp value.

    Returns:
        is_valid: Boolean value of 'True' if the timestamp is valid; 'False' otherwise.
    """
    is_valid = False
    # Convert to ISO8601 for validation
    str_val = str(str_val) + "Z"
    str_val = str_val.replace(' ', 'T')
    try:
        if RE_MATCH_ISO8601_TIMESTAMP(str_val) is not None:
            is_valid = True
    except:
        pass
    return is_valid


#
# MongoDB Document Class
#

class MongoDbDoc(dict):
    """ MongoDB Document

    The MongoDB Document class is a wrapper around a dictionary representing
    a MongoDB json document. This wrapper class provides "deep" read and write
    methods for getting and setting values in a nested dictionary. It also
    provides utility methods for automatically converting string values into
    the appropriate json datatypes (i.e., int, float, boolean, etc.).

    """

    def __init__(self, dict_):
        """ MongoDb Document constructor.

        Args:
            dict_: A dictionary containing the MongoDB document.
        """
        if dict_ is None:
            dict_ = {}
        super(MongoDbDoc, self).__init__(dict_)

    @staticmethod
    def _try_int(val_str):
        """ Utility to attempt conversion from a string to an integer. """
        try:
            val = int(val_str)
        except:
            val = None
        return val

    @staticmethod
    def _try_float(val_str):
        """ Utility to attempt conversion from a string to an float. """
        try:
            val = float(val_str)
        except:
            val = None
        return val

    @staticmethod
    def _try_boolean(val_str):
        """ Utility to attempt conversion from a string to an boolean. """
        if val_str.lower() == 'true':
            val = True
        elif val_str.lower() == 'false':
            val = False
        else:
            val = None
        return val

    @staticmethod
    def _try_object(val_str):
        """ Utility to attempt conversion from a string to a Python list or dictionary. """
        try:
            val = json.loads(val_str)
        except:
            val = None
        return val

    @staticmethod
    def _try_quoted_string(val_str):
        """ Utility to attempt conversion from a quoted "string" to a string. """
        # Parse double quoted string
        if val_str.startswith('"') and val_str.endswith('"'):
            val = val_str[1:-1]
        # Parse single quoted string
        elif val_str.startswith("'") and val_str.endswith("'"):
            val = val_str[1:-1]
        else:
            val = None
        return val

    def _cast_value(self, val_str):
        """
        Attemp to convert a string value from the CSV file to a MongoDB/json
        data type, such as integer, float, boolean, list/array, etc.
        """
        val = None
        # Strip whitespace from the string
        val_str = val_str.strip()

        # If the value begins with a quote or double quote, assume this is a quoted string.
        if val_str.startswith("'") or val_str.startswith('"'):
            val = self._try_quoted_string(val_str)

        # Not currently used
        # # If the value is a list (begins with a '[') or an object (begins with a '{'), parse
        # # the value as a json object.
        # if val is None:
        #     if val_str.startswith("[") or val_str.startswith("{"):
        #         val = self._try_object(val_str)

        # If the value contains a decimal point ('.'), attemp to convert to a float.
        if val is None:
            if '.' in val_str:
                val = self._try_float(val_str)

        # Attempt to convert to an integer
        if val is None:
            val = self._try_int(val_str)

        # Attempt to convert to a boolean
        if val is None:
            val = self._try_boolean(val_str)

        # Finally, use the originalstring if all of the convertions failed above.
        if val is None:
            val = val_str

        return val

    def dumps(self, sort_keys=True, indent=4, separators=(',', ': ')) -> str:
        """ Pretty print the document to a string. """
        return json.dumps(self, sort_keys=sort_keys, indent=indent, separators=separators)

    def read(self, path: str, default: Optional[Any] = None) -> Any:
        """
        Read the value of an attribute in a Mongo document. This method performs
        a "deep get" from a nested dictionary. If the attribute does not exist,
        the value of 'default' is returned.

        Args:
            path: The path to the nested attribute in the document.
            default: Default value to return if the attribute does not exist.

        Returns:
            dict_: The attribute value.
        """
        dict_ = self
        parsed_path = re_parse_bracketed_path(path)
        for elem in parsed_path:
            try:
                dict_ = dict_[elem]
            except:
                dict_ = default
                break

        return dict_

    def write(self, path: str, val: Any, force: bool) -> bool:
        """
        Write the value of an attribute in a Mongo document. This method performs
        a "deep set" in a nested dictionary. If the attribute exists in the document,
        the existing value is overwritten by 'val'. If the attribute does not exist,
        the write operation depends on the value of 'force'. If the attribute does not
        exist and 'force' is true, the new attribute identified by 'path' and
        associated value are created in the document. Otherwise, if 'force' is false,
        the write operation fails and the document remains unchanged.

        Args:
            path: The path to the nested attribute in the document.
            val: Value to write in the document for the attribute.
            force: Add the field to the document if it does not exist.

        Returns:
            result: True if the write operation succeeds, False otherwise.
        """
        result = False
        dict_ = self
        parsed_path = re_parse_bracketed_path(path)
        last = len(parsed_path) - 1
        for index, elem in enumerate(parsed_path):
            # If the element contains a list index, process node as a list
            list_key, list_index = re_parse_elem_list(elem)
            if list_key and list_index is not None:
                # Process the element as a list
                if list_key in dict_ or force:
                    if list_key not in dict_ or not isinstance(dict_[list_key], list):
                        dict_[list_key] = []
                    if index == last:
                        # Process the last element in the path
                        if list_index >= len(dict_[list_key]):
                            # Insert a new value
                            dict_[list_key].insert(list_index, self._cast_value(val))
                        else:
                            # Update an existing value
                            dict_[list_key][list_index] = self._cast_value(val)
                        result = True
                    else:
                        # Process an intermediate element in the path
                        if list_key not in dict_ or list_index >= len(dict_[list_key]):
                            dict_[list_key].insert(list_index, {})
                        dict_ = dict_[list_key][list_index]
                else:
                    # Error - key is missing
                    break
            else:
                # Otherwise process the node as a dictionary
                if elem in dict_ or force:
                    if index == last:
                        dict_[elem] = self._cast_value(val)
                        result = True
                    else:
                        if elem not in dict_:
                            dict_[elem] = {}
                        dict_ = dict_[elem]
                else:
                    # Error - key is missing
                    break

        return result


class SimulatedResponse:
    """
    Used to replicate a requests.Response object for mimicking a response from the API
    """

    def __init__(self):
        self.status_code = None
        self.text = None


class PonManagerAPIClient:
    """
    PON Manager REST API Driver class
    """

    def __init__(self, base_url, verbose):
        self.base_url = base_url
        self.host = urllib.parse.urlparse(self.base_url).netloc
        self.session_id = None
        self.csrf_token = None
        self.user = None
        self.verbose = verbose
        self.headers = {
            'Content-Type': 'application/json',
            'Referer': self.base_url
        }
        self.created_count = 0
        self.updated_count = 0
        self.deleted_count = 0
        self.error_count = 0

    def login(self, user, password):
        # Authenticate the user
        method = "POST"
        endpoint = "v1/users/authenticate/"
        url = self.base_url + "/" + endpoint
        headers = {
            'Content-Type': 'application/json'
        }
        # Build request body data for authentication
        body = json.dumps({
            "email": user,
            "password": password
        })
        response = requests.request(method=method, url=url, headers=headers, data=body, verify=False)

        if response.status_code == 200:
            if self.verbose:
                INFO(f"Logged in user {user}")
        else:
            ERROR(f"Failed to login user {user}. Exiting...")
            INFO(response.text)
            sys.exit(1)

        # Parse the Session ID and CSRF token cookies from the auth response
        cookies = response.headers['Set-Cookie']

        # CSRF Token Cookie - check for https cookies first, http second
        cookie = ""
        if "__Host-csrftoken" in cookies:
            csrf_token = re.search('__Host-csrftoken=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "__Host-csrftoken=" + csrf_token
        else:
            csrf_token = re.search('csrftoken=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "csrftoken=" + csrf_token
        self.csrf_token = csrf_token

        # Session ID Cookie - check for https cookies first, http second
        if "__Host-sessionid" in cookies:
            session_id = re.search('__Host-sessionid=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "; __Host-sessionid=" + session_id
        else:
            session_id = re.search('sessionid=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "; sessionid=" + session_id
        self.session_id = session_id

        # Add authentication cookie to the headers
        self.headers['Cookie'] = cookie

        # Add CSRF Token to the headers
        if self.csrf_token:
            self.headers['X-CSRFToken'] = self.csrf_token

        # Save the user email
        self.user = user

    def select_database(self, database_id):
        # Select the database
        return self.request(
            method="PUT",
            url="v1/databases/session/",
            data=json.dumps({"data": database_id}),
            expected_statuses=(200,))

    def logout(self):
        # Log out the user
        return self.request(
            method="GET",
            url="v1/users/logout/",
            data=None,
            expected_statuses=(204,))

    def request(self, method, url, data=None, expected_statuses=(200,), is_test=False):
        # Update the URL with the base_url
        if url[0] == '/':
            url = self.base_url + url
        else:
            url = self.base_url + '/' + url

        if isinstance(data, dict):
            data = json.dumps(data)

        # Send the request if not in test mode. Else simulate a response
        if not is_test:
            response = requests.request(
                method=method,
                url=url,
                headers=self.headers,
                data=data,
                verify=False)
        else:
            response = SimulatedResponse()
            response.text = json.dumps({"data": {"_id": "Default"}})

            if method.upper() == "GET":
                response.status_code = 200
            elif method.upper() == "DELETE":
                response.status_code = 204
            else:
                # Check if the document already exists
                existence_check_response = requests.request(
                    method="GET",
                    url=url,
                    headers=self.headers,
                    verify=False)
                existence_data = json.loads(existence_check_response.text)
                resource_exists = False
                if "data" in existence_data and existence_data["data"] is not None:
                    if isinstance(existence_data["data"], list):
                        resource_exists = len(existence_data["data"]) > 0
                    else:
                        resource_exists = True

                # Simulate POST method not allowing writing an existing resource
                if method.upper() == "PUT":
                    if "database" in url:
                        sys.exit(0)
                    if resource_exists:
                        response.status_code = 200
                    else:
                        response.status_code = 201
                elif method.upper() == "POST":
                    if resource_exists:
                        response.status_code = 409
                        response.text = json.dumps({"status": "fail", "details": "<Placeholder error message>"})
                    else:
                        response.status_code = 201

        # Print the response
        self.print_response_content(
            method=method,
            url=url,
            expected_statuses=expected_statuses,
            status=response.status_code,
            data=response.text)

        return response

    def print_response_content(self, url, method, expected_statuses, status, data=None):
        """ Prints the response result in formatted output """
        passed = status in expected_statuses

        if data is not None and status != 204:
            if isinstance(data, str):
                data = pyjson.loads(data)

        response_dict = {
            "User": self.user,
            "Method": method,
            "URL": url,
            "Status": status,
            "Detail": "PASSED" if passed else "FAILED"
        }

        # Increment counts for summary message
        if passed:
            # Ignore database selection response
            if "database" not in url:
                if status == 201:
                    self.created_count += 1
                    INFO(f"Successfully created document for {url}")
                elif status == 204 and method.upper() == "DELETE":
                    self.deleted_count += 1
                    INFO(f"Successfully deleted document for {url}")
                elif status == 200 and method.upper() == "PUT":
                    self.updated_count += 1
                    INFO(f"Successfully updated document for {url}")
            elif self.verbose:
                INFO(f"Successfully selected database")
        else:
            self.error_count += 1
            # Determine error message contents based on status code and response data contents
            if status == 400:
                if data and "details" in data:
                    ERROR(f"{url} JSON Validation Failed: {data['details']['message']}\n{data['details']}")
                else:
                    ERROR(f"{url} JSON Validation Failed:\n{data['details']}")
            elif status == 403:
                ERROR(f"User {self.user} is forbidden from accessing this resource")
            elif status == 409:
                if data and "details" in data:
                    if isinstance(data["details"], str) and "{" in data["details"]:
                        data["details"] = json.loads(data["details"])
                    if type(data["details"]) != str and "message" in data["details"]:
                        ERROR(f"Could not create resource at {url}\n{data['details']['message']}")
                    else:
                        ERROR(f"Could not create resource at {url}\n{data['details']}")
                else:
                    ERROR(f"Could not create resource at {url}")
            elif status == 500:
                ERROR(f"Server Error:\n{data}")

        # Print raw response if in verbose mode
        if self.verbose:
            INFO(f"{url} raw response data:\n{json.dumps(response_dict)}")


#
# Script's main() function.
#

def main(args):
    """ Main function for BulkConfigure script.

    Args:
        args: Command line arguments.
    """
    INFO(f"Processing bulk configuration from '{args.file}'")

    # Create REST API client
    web_client = PonManagerAPIClient(args.url, args.verbose)

    # Login the user and select the database
    web_client.login(args.user, args.password)
    web_client.select_database(args.db)

    column_collections = []
    column_paths = []
    stats = {
        'field warnings': 0,
        'field errors': 0,
        'collection warnings': 0,
        'collection errors': 0
    }

    csv_data = None
    try:
        csv_file = open(args.file, encoding='utf-8-sig')
        csv_data = csv.reader(csv_file, delimiter=',', dialect='excel')
    except OSError as err:
        ERROR(f"Failed processing csv file, {err}")
        sys.exit(1)

    if csv_data:
        #
        # Parse header row
        #

        # First line contains the header row, which contains the keys.
        row = next(csv_data)
        for column_value in row:
            column_value = column_value.strip()
            parsed_path = re_parse_bracketed_path(column_value)
            column_collections.append(parsed_path[0])
            column_paths.append(column_value[len(f"[{parsed_path[0]}]"):])

        # Debug output
        if args.verbose:
            print(f"column collections  = {column_collections}")
            print(f"column paths = {column_paths}")

        #
        # Parse device configuration data rows
        #

        default_document = None

        # Parse the remaining rows as data
        for row in csv_data:
            # Overlay the csv data on top of the default
            db_documents = {}
            device_id = None

            for column_index, column_value in enumerate(row):
                # The first column must contain the _id for all related collections
                if column_index == 0:
                    device_id = column_value
                    # If the device ID is a .json quoted string, remove the outer quotes
                    if device_id.startswith('"') and device_id.endswith('"'):
                        device_id = device_id[1:-1]

                # If the document for the specified collection has not been initialized yet, do that now.
                db_collection_name = column_collections[column_index]
                if db_collection_name not in db_documents:
                    # Start with the default record.
                    if default_document is None:
                        # Determine url from collection name
                        url = None
                        if db_collection_name == "CNTL-CFG":
                            url = "v1/controllers/configs/Default/"
                        elif db_collection_name == "SWI-CFG":
                            url = "v1/switches/configs/Default/"
                        elif db_collection_name == "OLT-CFG":
                            url = "v1/olts/configs/Default/"
                        elif db_collection_name == "ONU-CFG":
                            url = "v1/onus/configs/Default/"
                        default_doc_response = web_client.request(method="GET", url=url, is_test=args.test)
                        default_doc_response_data = json.loads(default_doc_response.text)
                        if "data" in default_doc_response_data:
                            default_document = default_doc_response_data["data"]
                        else:
                            ERROR(f"Could not get Default document for {db_collection_name}. Please create the Default configuration. Exiting...")
                            sys.exit(1)
                    db_documents[db_collection_name] = MongoDbDoc(default_document)

                    # Update the Device ID in the record.
                    db_documents[db_collection_name]['_id'] = device_id

                # Remove only needs the _id - skip further processing
                if args.remove:
                    continue

                # If the cell is blank, skip over updating this field
                if len(column_value) <= 0:
                    continue

                # Get the current document base on the collection name
                db_doc = db_documents[db_collection_name]

                # Debug output
                # print(f"{column_paths[column_index]} = '{column_value}'")

                # Validate "Create Date" timestamp fields from the CSV file
                if column_paths[column_index].endswith("[Create Date]"):
                    if column_value:
                        if not re_db_timestamp_is_valid(column_value):
                            full_path = f"[{db_collection_name}]{column_paths[column_index]}"
                            ERROR(f"Failed to set {full_path} in document for {db_doc['_id']}, invalid timestamp format.")
                            stats['field errors'] += 1
                            continue

                # Write the column value to the document
                if not db_doc.write(column_paths[column_index], column_value, force=False):
                    full_path = f"[{db_collection_name}]{column_paths[column_index]}"
                    if args.insert_missing:
                        WARNING(f"Inserting missing field {full_path} in document for {db_doc['_id']}, allowed by the --insert-missing option.")
                        db_doc.write(column_paths[column_index], column_value, force=True)
                        stats['field warnings'] += 1
                    else:
                        ERROR(f"Failed to set {full_path} in document for {db_doc['_id']}, field does not exist. Use --insert-missing to add missing fields.")
                        stats['field errors'] += 1

            # Update the database for each document parsed from the CSV file
            for db_collection_name, db_doc in db_documents.items():
                # Determine url from collection name
                url = None
                if db_collection_name == "CNTL-CFG":
                    url = "v1/controllers/configs/"
                elif db_collection_name == "SWI-CFG":
                    url = "v1/switches/configs/"
                elif db_collection_name == "OLT-CFG":
                    url = "v1/olts/configs/"
                elif db_collection_name == "ONU-CFG":
                    url = "v1/onus/configs/"

                # Update the entry in the database
                if args.remove:
                    # If the '--remove' option was specified, delete the device from the database
                    if args.verbose:
                        print(f"{db_collection_name} = ", end='')
                        print({'_id': db_doc['_id']})
                    url += db_doc['_id'] + "/"
                    web_client.request(method="DELETE", url=url, expected_statuses=(204,), is_test=args.test)
                else:
                    # Otherwise, add (or update) the entry in the database
                    # Overwrite the NETCONF Name not explicitly set in the CSV file.
                    netconf_name = db_doc.read("[NETCONF][Name]")
                    if netconf_name == 'Default':
                        db_doc.write("[NETCONF][Name]", db_doc['_id'], force=False)
                    # Debug output
                    if args.verbose:
                        print(f"{db_collection_name} = ", end='')
                        print(db_doc.dumps())
                    # If the '--overwrite' option was specified, replace existing documents in the database
                    if args.overwrite:
                        # Update (or create) the document in MongoDB
                        url += db_doc['_id'] + "/"
                        web_client.request(method="PUT", url=url, data=json.dumps({"data": db_doc}), expected_statuses=(200, 201), is_test=args.test)
                    else:
                        # Create the document to MongoDB
                        web_client.request(method="POST", url=url, data=json.dumps({"data": db_doc}), expected_statuses=(201,), is_test=args.test)

        # Print summary
        print()
        print("---")
        print(f"Summary:")
        print("  " + HIGHLIGHT_OK(bool(web_client.created_count), f"created = {web_client.created_count}"), end='')
        print(", " + HIGHLIGHT_WA(bool(web_client.updated_count), f"updated = {web_client.updated_count}"), end='')
        print(", " + HIGHLIGHT_OK(bool(web_client.deleted_count), f"deleted = {web_client.deleted_count}"), end='')
        print(", " + HIGHLIGHT_ER(bool(web_client.error_count), f"errors = {web_client.error_count}"), end='')
        print()
        print("  " + HIGHLIGHT_WA(bool(stats['field warnings']), f"field warnings = {stats['field warnings']}"), end='')
        print(", " + HIGHLIGHT_ER(bool(stats['field errors']), f"field errors = {stats['field errors']}"), end='')
        print()
        print("  " + HIGHLIGHT_WA(bool(stats['collection warnings']), f"collection warnings = {stats['collection warnings']}"), end='')
        print(", " + HIGHLIGHT_ER(bool(stats['collection errors']), f"collection errors = {stats['collection errors']}"), end='')
        print()
        if args.test:
            print()
            WARNING(f"No changes were applied to the database due to the --test option.")
            WARNING(f"Run the command without hte --test option to commit changes to the database.")
            print()


# Example command line invocation to configure devices in bulk.
#
# $ ./BulkConfigure.py -f olt_bulk_config.csv
if __name__ == '__main__':
    parser = argparse.ArgumentParser(add_help=True, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="db", default="Default", required=False, help="ID of the database on the PON Manager REST API server to use.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://127.0.0.1/api", required=False, help="URL of the PON Manager REST API server (e.g., https://10.2.10.222/api).")
    parser.add_argument("-p", "--password", action="store", dest="password", required=True, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", required=True, help="User email address to authenticate with.")
    parser.add_argument("-f", "--file", action="store", dest="file", default=None, required=True, help="A CSV formatted file containing the device provisioning.")
    parser.add_argument("--insert-missing", action="store_true", dest="insert_missing", default=False, required=False, help="Force insert collections and fields from the CSV which are are missing from the default document.")
    parser.add_argument("--overwrite", action="store_true", dest="overwrite", default=False, required=False, help="Force overwrite the entry if it already exists in the database.")
    parser.add_argument("-r", "--remove", action="store_true", dest="remove", default=False, required=False, help="Remove the device entries from the database.")
    parser.add_argument("-t", "--test", action="store_true", dest="test", default=False, required=False, help="Perform a dry-run and report results without applying any modifications to the database.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Display verbose output.")
    main(parser.parse_args())
