"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import json

from django.conf import settings

from log.PonManagerLogger import pon_manager_logger
from database_manager import database_manager


def update_session_database_id(get_response):
    def middleware(request):
        """ Updates the request.session database field for every request if it is not the correct selection """
        _url = str(request.path)

        # All request except for users/authenticate/ and user/exist/
        if "user/logout/" in _url or "users/logout/" in _url:
            try:
                database_manager.remove_session_database_id(request.session.session_key)
            except KeyError:
                pass  # Do not care if the session key was not found if only trying to remove it
        elif request.user.is_authenticated and "users/authenticate/" not in _url and "user/exist/" not in _url:
            try:
                database_id = database_manager.get_session_database_id(request.session.session_key)
            except (KeyError, TypeError):
                try:
                    database_id = database_manager.get_users_selected_database(request.user.email)
                    database_manager.set_session_database_id(request.session.session_key, database_id)
                except KeyError:
                    pon_manager_logger.warning("User's selected database was not found. Switching user to 'Default'")
                    database_id = "Default"
                    database_manager.set_users_selected_database(request.user.email, database_id)
                    database_manager.set_session_database_id(request.session.session_key, database_id)

            request.session['database'] = database_id

        # Execute view for request
        response = get_response(request)

        # Logging in request
        # Must be done after the view or else request.user will be Anonymous and request.session.session_key will be None
        if "users/authenticate/" in _url and response.status_code == 200:
            database_id = database_manager.get_users_selected_database(request.user.email)
            database_manager.set_session_database_id(request.session.session_key, database_id)
            request.session['database'] = database_id

        return response

    return middleware


def set_headers(get_response):
    def middleware(request):
        """ Sets the response headers for every response before sending it to the client """
        cookie_id = "__Host-sessionexpire="
        secure_flag = "Secure"

        if "__Host-" not in settings.CSRF_COOKIE_NAME:
            cookie_id = "sessionexpire="
            secure_flag = ""
        headers = {
            'set-cookie': cookie_id + request.session.get_expiry_date().strftime('%Y-%m-%dT%H:%M:%SZ') +
                          "; expires=" + request.session.get_expiry_date().strftime('%a, %d %b %Y %H:%M:%S GMT') +
                          "; Max-Age=31449600; Path=/; SameSite=Strict; " + secure_flag
        }

        response = get_response(request)
        response.headers = headers

        return response

    return middleware


def handle_server_errors(get_response):
    def middleware(request):
        """ Overrides the response body in the case of certain errors """
        response = get_response(request)

        if response.status_code >= 500:
            # Only overwrite default message. Allows for custom 500 catching and message if desired
            if "<h1>" in str(response.content):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": "Internal server error. See server logs for details."
                    }
                })
            # Catch APIException for database being offline/empty
            elif "detail" in json.loads(response.content):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": json.loads(response.content)["detail"]
                    }
                })
        # Alert client if they made a request to a URL without a trailing slash
        elif response.status_code == 404:
            # Only overwrite default message
            if "<h1>" in str(response.content) and not str(request.path).endswith("/"):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": "URL not found. All URLs must end with a slash ('/')"
                    }
                })
        elif response.status_code == 403:
            # Only overwrite default message
            if "<h1>" in str(response.content):
                response.content = json.dumps({
                    "status": "fail",
                    "details": {
                        "message": "You are not authorized to access this resource"
                    }
                })

        return response

    return middleware


def log_request(get_response):
    def middleware(request):
        """ Logs the request to the PON Controller database before returning response """
        response = None

        try:
            full_uri = request.build_absolute_uri()
            # Splitting full_uri on a path with spaces causes error
            formatted_path = request.path.replace(" ", "%20")
            params = full_uri.split(formatted_path)[1]
            url = request.path + params
            method = request.method
            email = "unauthenticated"
            data_change = ""

            # Get client's IP address
            forwarded = request.META.get('HTTP_X_FORWARDED_FOR')
            if forwarded:
                client_addr = forwarded.split(',')[0]
            else:
                client_addr = request.META.get('REMOTE_ADDR')


            if request.user.is_authenticated:
                email = request.user.email.lower()

            # Complete request
            response = get_response(request)

            status = response.status_code
            if hasattr(response, "data_change"):
                data_change = response.data_change
            if hasattr(response, "data"):
                if isinstance(response.data, dict) and "details" in response.data:
                    data_change = str(response.data['details'])
                    if isinstance(response.data['details'], dict) and "message" in response.data['details']:
                        data_change = str(response.data['details']['message'])
            # Check for email again (won't have email previously if request is to login)
            if request.user.is_authenticated:
                email = request.user.email.lower()

            try:
                # All per request logging is handled here
                # If the request is a PUT/POST/PATCH then the old and new data must be passed to the PonManagerApiResponse object
                log_message = "{} {} {} {} status: {} {}".format(client_addr, email, method.upper(), url, status, data_change)
                if int(status) < 400:
                    pon_manager_logger.info(log_message)
                elif int(status) == 400:
                    pon_manager_logger.warning(log_message)
                elif int(status) > 400:
                    pon_manager_logger.error(log_message)
            except Exception as e:
                try:
                    pon_manager_logger.error(f"Failed to log request: {e}")
                except Exception:
                    print(f"Failed to log request: {e}")
        except Exception as e:
            if response is None:
                response = get_response(request)
            pon_manager_logger.error(f"Failed to log request for: {request.build_absolute_uri()} - {e}")

        return response

    return middleware
