"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, inline_serializer
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import PonManagerApiResponse, validate_query_params, validate_data, get_nested_value, permission_required_any_of


# ==================================================
# ========== All SLA Configurations View ===========
# ==================================================
class Slas(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_sla_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['sla', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_slas', raise_exception=True))
    @validate_query_params(collection="SLA-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """ Get all SLA configurations """
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="SLA-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="SLA-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_sla_config",
        request=inline_serializer(name="SLA-CFG", fields={"data": JSONField(help_text="SLA-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['sla', 'config', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_global_config_slas', raise_exception=True))
    @validate_data(collection="SLA-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided SLA config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="SLA-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            sla_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"SLA configuration with id {sla_id} already exists")

        return response


# ==================================================
# =========== One SLA Configuration View ===========
# ==================================================
class OneSla(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_sla_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['sla', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_slas', raise_exception=True))
    def get(self, request, sla_id, version):
        """ Get the specified SLA configuration """
        res_data = database_manager.find_one(database_id=request.session.get("database"), collection="SLA-CFG", query={"_id": sla_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_sla_config",
        request=inline_serializer(name="SLA-CFG", fields={"data": JSONField(help_text="SLA-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['sla', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_global_config_slas', 'global_config.can_create_global_config_slas'], raise_exception=True))
    @validate_data(collection="SLA-CFG", resource_id_param="sla_id")
    def put(self, request, data, sla_id, version):
        """ Update the provided SLA configuration """
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SLA-CFG", query={"_id": sla_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_sla_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['sla', 'config', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_global_config_slas', raise_exception=True))
    def delete(self, request, sla_id, version):
        """ Delete the specified SLA configuration """
        database_manager.delete_one(database_id=request.session.get("database"), collection="SLA-CFG", query={"_id": sla_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
