"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, inline_serializer
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import PonManagerApiResponse, get_nested_value, validate_data, validate_query_params, permission_required_any_of


# ==================================================
# ======== All Service Configurations View =========
# ==================================================
class ServiceConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_service_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['service', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_services', raise_exception=True))
    @validate_query_params(collection="SRV-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """ Get all Service configurations """
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="SRV-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="SRV-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_service_config",
        request=inline_serializer(name="SRV-CFG", fields={"data": JSONField(help_text="SRV-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['service', 'config', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_global_config_services', raise_exception=True))
    @validate_data(collection="SRV-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided Service config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="SRV-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            srv_cfg_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"Service configuration with id {srv_cfg_id} already exists")

        return response


# ==================================================
# ========= One Service Configuration View =========
# ==================================================
class OneServiceConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_service_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['service', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_services', raise_exception=True))
    def get(self, request, srv_cfg_id, version):
        """ Get the specified Service configuration """
        res_data = database_manager.find_one(database_id=request.session.get("database"), collection="SRV-CFG", query={"_id": srv_cfg_id})
        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_service_config",
        request=inline_serializer(name="SRV-CFG", fields={"data": JSONField(help_text="SRV-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['service', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_global_config_services', 'global_config.can_create_global_config_services'], raise_exception=True))
    @validate_data(collection="SRV-CFG", resource_id_param="srv_cfg_id")
    def put(self, request, data, srv_cfg_id, version):
        """ Update the provided Service configuration """
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SRV-CFG",
                                                             query={"_id": srv_cfg_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_service_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['service', 'config', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_global_config_services', raise_exception=True))
    def delete(self, request, srv_cfg_id, version):
        """ Delete the specified SERVICE configuration """
        database_manager.delete_one(database_id=request.session.get("database"), collection="SRV-CFG", query={"_id": srv_cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
