"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import base64
import io
import json
import os
import gridfs
from PIL import Image

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from resizeimage import resizeimage
from rest_framework.fields import CharField, ListField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, inline_serializer
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import RequestSerializer
from utils.tools import PonManagerApiResponse, get_nested_value, validate_data, validate_query_params, permission_required_any_of

# ==================================================
# === Firmware Files Metadata Schema Definition ====
# ==================================================

FIRMWARE_FILES_SCHEMA = {
    "properties": {
        "_id": {
            "type": "string"
        },
        # Base64 Encoded string
        "file": {
            "type": "string"
        },
        "filename": {
            "type": "string"
        },
        "metadata": {
            "type": "object",
            "properties": {
                "Compatible Manufacturer": {
                    "type": "string"
                },
                "Compatible Model": {
                    "type": "array",
                    "items": {
                        "type": "string"
                    }
                },
                "Version": {
                    "type": "string"
                }
            },
            "required": [
                "Compatible Manufacturer",
                "Compatible Model",
                "Version"
            ]
        },
        "required": [
            "_id",
            "filename",
            "metadata"
        ]
    }
}

_MAX_PICTURE_HEIGHT = 400


# ==================================================
# =========== OLT Firmware Metadata View ===========
# ==================================================
class OltMetadata(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_firmware_metadata",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['firmware', 'olt', 'metadata', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_files', raise_exception=True))
    @validate_query_params(collection="OLT-FIRMWARE", schema=FIRMWARE_FILES_SCHEMA)
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """ Get the metadata for all OLT Firmware files """
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="OLT-FIRMWARE.files",
                                             query=query, distinct = distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="OLT-FIRMWARE.files",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# =========== ONU Firmware Metadata View ===========
# ==================================================
class OnuMetadata(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_onu_firmware_metadata",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['firmware', 'onu', 'metadata', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_files', raise_exception=True))
    @validate_query_params(collection="ONU-FIRMWARE", schema=FIRMWARE_FILES_SCHEMA)
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """ Get the metadata for all ONU Firmware files """
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="ONU-FIRMWARE.files",
                                             query=query, distinct = distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="ONU-FIRMWARE.files",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# =========== One OLT Firmware File View ===========
# ==================================================
class OltFiles(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_firmware_file",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['firmware', 'olt', 'file', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_files', raise_exception=True))
    def get(self, request, filename, version):
        """ Get the metadata for the specified OLT Firmware file """
        res_data = database_manager.find_one(database_id=request.session.get("database"), collection="OLT-FIRMWARE.files",
                                             query={"filename": filename})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_olt_firmware_file",
        request=RequestSerializer(name="OltFirmwareUpload", data_fields={
            "file": CharField(help_text="base64 encoded file content"),
            "metadata": inline_serializer(name="OltFirmwareUploadMetadata", fields={
                "Compatible Manufacturer": CharField(),
                "Compatible Model": ListField(),
                "Version": CharField()
            })
        }),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500),
        },
        tags=['firmware', 'olt', 'file', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_global_config_files', raise_exception=True))
    @validate_data(collection="OLT-FIRMWARE", resource_id_param=None, schema=FIRMWARE_FILES_SCHEMA)
    def post(self, request, data, filename, version):
        """ Upload the provided OLT firmware file """
        file_contents = get_nested_value(request.data, ['data', 'file'])
        metadata = get_nested_value(request.data, ['data', 'metadata'])
        if file_contents is None or metadata is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: { file: <base64 encoded file data>, metadata: <OLT Firmware metadata JSON> } }'")
        else:
            fs = gridfs.GridFSBucket(db=database_manager.get_database(request.session.get("database")), bucket_name="OLT-FIRMWARE")
            file_id = os.path.splitext(filename)[0]

            if fs.find({"_id": file_id}).count() == 0:
                if str(file_contents).find(",") >= 0:
                    file_data = base64.b64decode(json.dumps(file_contents).split(",")[1])
                else:
                    file_data = base64.b64decode(json.dumps(file_contents))

                try:
                    up = fs.open_upload_stream_with_id(file_id=file_id, filename=filename, metadata=metadata)
                    up.write(file_data)
                    up.close()
                    response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=filename)
                except Exception as e:
                    raise APIException(e)
            else:
                response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"OLT Firmware file with id {file_id} already exists")

        return response

    @extend_schema(
        operation_id="put_olt_firmware_file",
        request=RequestSerializer(name="OltFirmwareMetadata", data_fields={
            "metadata": inline_serializer(name="OltFirmwareDataMetadata", fields={
                "Compatible Manufacturer": CharField(),
                "Compatible Model": ListField(),
                "Version": CharField()
            })
        }),
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['firmware', 'olt', 'file', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_global_config_files', 'global_config.can_create_global_config_files'], raise_exception=True))
    @validate_data(collection="OLT-FIRMWARE", resource_id_param=None, schema=FIRMWARE_FILES_SCHEMA)
    def put(self, request, data, filename, version):
        """ Update the metadata for the specified OLT firmware file """
        metadata = get_nested_value(request.data, ['data', 'metadata'])
        if metadata is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: { metadata: <OLT Firmware metadata JSON> } }'")
        else:
            update_document = {"$set": {"metadata": metadata}}
            update_result = database_manager.update_one(database_id=request.session.get("database"), collection="OLT-FIRMWARE.files",
                                                        query={"filename": filename}, update_document=update_document)
            if update_result.matched_count > 0:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details="OLT Firmware file not found")

        return response

    @extend_schema(
        operation_id="delete_olt_firmware_file",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['firmware', 'olt', 'file', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_global_config_files', raise_exception=True))
    def delete(self, request, filename, version):
        """ Delete the specified OLT firmware file """
        fs = gridfs.GridFSBucket(db=database_manager.get_database(request.session.get("database")), bucket_name="OLT-FIRMWARE")
        file_id = os.path.splitext(filename)[0]
        fs.delete(file_id=file_id)

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== One ONU Firmware File View ===========
# ==================================================
class OnuFiles(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_onu_firmware_file",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['firmware', 'onu', 'file', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_files', raise_exception=True))
    def get(self, request, filename, version):
        """ Get the metadata for the specified ONU Firmware file """
        res_data = database_manager.find_one(database_id=request.session.get("database"), collection="ONU-FIRMWARE.files",
                                             query={"filename": filename})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_onu_firmware_file",
        request=RequestSerializer(name="OnuFirmwareUpload", data_fields={
            "file": CharField(help_text="base64 encoded file content"),
            "metadata": inline_serializer(name="OnuFirmwareUploadMetadata", fields={
                "Compatible Manufacturer": CharField(),
                "Compatible Model": ListField(),
                "Version": CharField()
            })
        }),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500),
        },
        tags=['firmware', 'onu', 'file', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_global_config_files', raise_exception=True))
    @validate_data(collection="ONU-FIRMWARE", resource_id_param=None, schema=FIRMWARE_FILES_SCHEMA)
    def post(self, request, data, filename, version):
        """ Upload the provided ONU firmware file """
        file_contents = get_nested_value(request.data, ['data', 'file'])
        metadata = get_nested_value(request.data, ['data', 'metadata'])
        if file_contents is None or metadata is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: { file: <base64 encoded file data>, metadata: <ONU Firmware metadata JSON> } }'")
        else:
            fs = gridfs.GridFSBucket(db=database_manager.get_database(request.session.get("database")), bucket_name="ONU-FIRMWARE")

            if fs.find({"_id": filename}).count() == 0:
                if str(file_contents).find(",") >= 0:
                    file_data = base64.b64decode(json.dumps(file_contents).split(",")[1])
                else:
                    file_data = base64.b64decode(json.dumps(file_contents))
                try:
                    up = fs.open_upload_stream_with_id(file_id=filename, filename=filename, metadata=metadata)
                    up.write(file_data)
                    up.close()
                    response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=filename)
                except Exception as e:
                    raise APIException(e)
            else:
                response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"ONU Firmware file with id {filename} already exists")

        return response

    @extend_schema(
        operation_id="put_onu_firmware_file",
        request=RequestSerializer(name="OnuFirmwareMetadata", data_fields={
            "metadata": inline_serializer(name="OnuFirmwareDataMetadata", fields={
                "Compatible Manufacturer": CharField(),
                "Compatible Model": ListField(),
                "Version": CharField()
            })
        }),
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['firmware', 'onu', 'file', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_global_config_files', 'global_config.can_create_global_config_files'], raise_exception=True))
    @validate_data(collection="ONU-FIRMWARE", resource_id_param=None, schema=FIRMWARE_FILES_SCHEMA)
    def put(self, request, data, filename, version):
        """ Update the metadata for the specified ONU firmware file """
        metadata = get_nested_value(data, ['metadata'])
        if metadata is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: { metadata: <ONU Firmware metadata JSON> } }'")
        else:
            update_document = {"$set": {"metadata": metadata}}
            update_result = database_manager.update_one(database_id=request.session.get("database"), collection="ONU-FIRMWARE.files",
                                                        query={"filename": filename}, update_document=update_document)
            if update_result.matched_count > 0:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details="ONU Firmware file not found")

        return response

    @extend_schema(
        operation_id="delete_onu_firmware_file",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['firmware', 'onu', 'file', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_global_config_files', raise_exception=True))
    def delete(self, request, filename, version):
        """ Delete the specified ONU firmware file """
        fs = gridfs.GridFSBucket(db=database_manager.get_database(request.session.get("database")), bucket_name="ONU-FIRMWARE")
        fs.delete(file_id=filename)

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ===== Picture File Metadata Schema Definition ====
# ==================================================
PICTURE_FILES_SCHEMA = {
    "properties": {
        "_id": {
            "type": "string"
        },
        # Base64 Encoded string
        "file": {
            "type": "string"
        },
        "filename": {
            "type": "string"
        },
        "metadata": {
            "type": "object",
            "properties": {
                "Device Type": {
                    "type": "string",
                    "enum": ["CNTL", "OLT", "ONU", "SWI"]
                }
            },
            "required": [
                "Device Type"
            ]
        },
        "required": [
            "_id",
            "filename",
            "metadata"
        ]
    }
}


# ==================================================
# =========== Picture File Metadata View ===========
# ==================================================
class PictureMetadata(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_picture_metadata",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['picture', 'metadata', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_files', raise_exception=True))
    @validate_query_params(collection="PICTURES.files", schema=PICTURE_FILES_SCHEMA)
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """ Get the metadata for all Picture files """
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="PICTURES.files",
                                             query=query, distinct = distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="PICTURES.files",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============ One Picture File View ===============
# ==================================================
class PictureFiles(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_picture_file",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['picture', 'file', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_files', raise_exception=True))
    def get(self, request, filename, version):
        """ Get the metadata for the specified Picture file """
        # Device configs use picture ID, not filename. So we have to reference the picture by ID when getting it in network tabs
        file_id = filename
        fs = gridfs.GridFSBucket(db=database_manager.get_database(request.session.get("database")), bucket_name="PICTURES")
        file_iter = fs.find({"_id": file_id})
        if file_iter.count() > 0:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=base64.b64encode(file_iter.next().read()))
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details=f"Picture with ID {file_id} was not found")
        file_iter.close()

        return response

    @extend_schema(
        operation_id="post_picture_file",
        request=RequestSerializer(name="PictureUpload", data_fields={
            "file": CharField(help_text="base64 encoded file content"),
            "metadata": inline_serializer(name="PictureUploadMetadata", fields={
                "Device Type": CharField()
            })
        }),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500),
        },
        tags=['picture', 'file', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_global_config_files', raise_exception=True))
    @validate_data(collection="PICTURES.files", resource_id_param=None, schema=PICTURE_FILES_SCHEMA)
    def post(self, request, data, filename, version):
        """ Upload the provided Picture file """
        file_contents = get_nested_value(request.data, ['data', 'file'])
        metadata = get_nested_value(request.data, ['data', 'metadata'])
        if file_contents is None or metadata is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: { file: <base64 encoded file data>, metadata: <Picture metadata JSON> } }'")
        else:
            fs = gridfs.GridFSBucket(db=database_manager.get_database(request.session.get("database")), bucket_name="PICTURES")
            file_id = os.path.splitext(filename)[0]

            if fs.find({"_id": file_id}).count() == 0:
                response = None
                if str(file_contents).find(",") >= 0:
                    file_data = base64.b64decode(json.dumps(file_contents).split(",")[1])
                else:
                    file_data = base64.b64decode(json.dumps(file_contents))

                try:
                    # Resize image height if needed
                    with Image.open(io.BytesIO(file_data)) as image:
                        if image.size[1] > _MAX_PICTURE_HEIGHT:
                            # Keep existing file format if in filename, else use '.jpg'
                            file_format = "JPEG"
                            path = os.path.splitext(filename)
                            if len(path) > 1:
                                if str(path[1])[1:].upper() in ["JPG", "JPEG"]:
                                    file_format = "JPEG"
                                elif str(path[1])[1:].upper() == ["PNG"]:
                                    file_format = str(path[1])[1:].upper()

                            image_bytes = io.BytesIO()
                            resized_image = resizeimage.resize_height(image, _MAX_PICTURE_HEIGHT)
                            # save doesn't support RGBA mode, change to RGB
                            if resized_image.mode == "RGBA":
                                resized_image.mode = "RGB"
                            resized_image.save(image_bytes, format=file_format)
                            file_data = image_bytes.getvalue()
                except IOError:
                    response = PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR, details="Could not read image file data")

                if response is None:
                    try:
                        up = fs.open_upload_stream_with_id(file_id=file_id, filename=filename, metadata=metadata)
                        up.write(file_data)
                        up.close()
                        response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=filename)
                    except Exception as e:
                        raise APIException(e)
            else:
                response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"Picture file with id {file_id} already exists")

        return response

    @extend_schema(
        operation_id="put_picture_file",
        request=RequestSerializer(name="PictureUpdate", data_fields={
            "metadata": inline_serializer(name="PictureUpdateMetadata", fields={
                "Device Type": CharField()
            })
        }),
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['picture', 'file', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_global_config_files', 'global_config.can_create_global_config_files'], raise_exception=True))
    @validate_data(collection="PICTURES.files", resource_id_param=None, schema=PICTURE_FILES_SCHEMA)
    def put(self, request, data, filename, version):
        """ Update the metadata for the specified Picture file """
        metadata = get_nested_value(request.data, ['data', 'metadata'])
        if metadata is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: { metadata: <Picture metadata JSON> } }'")
        else:
            update_document = {"$set": {"metadata": metadata}}
            update_result = database_manager.update_one(database_id=request.session.get("database"), collection="PICTURES.files",
                                                        query={"filename": filename}, update_document=update_document)
            if update_result.matched_count > 0:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details="Picture file not found")

        return response

    @extend_schema(
        operation_id="delete_picture_file",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['picture', 'file', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_global_config_files', raise_exception=True))
    def delete(self, request, filename, version):
        """ Delete the specified Picture file """
        fs = gridfs.GridFSBucket(db=database_manager.get_database(request.session.get("database")), bucket_name="PICTURES")
        file_id = os.path.splitext(filename)[0]
        fs.delete(file_id=file_id)

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
